/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration that defines how the capacity provider scales compute instances based on demand and policies.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityProviderScalingConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityProviderScalingConfig.Builder, CapacityProviderScalingConfig> {
    private static final SdkField<Integer> MAX_V_CPU_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxVCpuCount").getter(getter(CapacityProviderScalingConfig::maxVCpuCount))
            .setter(setter(Builder::maxVCpuCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxVCpuCount").build()).build();

    private static final SdkField<String> SCALING_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingMode").getter(getter(CapacityProviderScalingConfig::scalingModeAsString))
            .setter(setter(Builder::scalingMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingMode").build()).build();

    private static final SdkField<List<TargetTrackingScalingPolicy>> SCALING_POLICIES_FIELD = SdkField
            .<List<TargetTrackingScalingPolicy>> builder(MarshallingType.LIST)
            .memberName("ScalingPolicies")
            .getter(getter(CapacityProviderScalingConfig::scalingPolicies))
            .setter(setter(Builder::scalingPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingPolicies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetTrackingScalingPolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetTrackingScalingPolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MAX_V_CPU_COUNT_FIELD,
            SCALING_MODE_FIELD, SCALING_POLICIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer maxVCpuCount;

    private final String scalingMode;

    private final List<TargetTrackingScalingPolicy> scalingPolicies;

    private CapacityProviderScalingConfig(BuilderImpl builder) {
        this.maxVCpuCount = builder.maxVCpuCount;
        this.scalingMode = builder.scalingMode;
        this.scalingPolicies = builder.scalingPolicies;
    }

    /**
     * <p>
     * The maximum number of vCPUs that the capacity provider can provision across all compute instances.
     * </p>
     * 
     * @return The maximum number of vCPUs that the capacity provider can provision across all compute instances.
     */
    public final Integer maxVCpuCount() {
        return maxVCpuCount;
    }

    /**
     * <p>
     * The scaling mode that determines how the capacity provider responds to changes in demand.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalingMode} will
     * return {@link CapacityProviderScalingMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #scalingModeAsString}.
     * </p>
     * 
     * @return The scaling mode that determines how the capacity provider responds to changes in demand.
     * @see CapacityProviderScalingMode
     */
    public final CapacityProviderScalingMode scalingMode() {
        return CapacityProviderScalingMode.fromValue(scalingMode);
    }

    /**
     * <p>
     * The scaling mode that determines how the capacity provider responds to changes in demand.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalingMode} will
     * return {@link CapacityProviderScalingMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #scalingModeAsString}.
     * </p>
     * 
     * @return The scaling mode that determines how the capacity provider responds to changes in demand.
     * @see CapacityProviderScalingMode
     */
    public final String scalingModeAsString() {
        return scalingMode;
    }

    /**
     * For responses, this returns true if the service returned a value for the ScalingPolicies property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasScalingPolicies() {
        return scalingPolicies != null && !(scalingPolicies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of scaling policies that define how the capacity provider scales compute instances based on metrics and
     * thresholds.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScalingPolicies} method.
     * </p>
     * 
     * @return A list of scaling policies that define how the capacity provider scales compute instances based on
     *         metrics and thresholds.
     */
    public final List<TargetTrackingScalingPolicy> scalingPolicies() {
        return scalingPolicies;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxVCpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(scalingModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasScalingPolicies() ? scalingPolicies() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityProviderScalingConfig)) {
            return false;
        }
        CapacityProviderScalingConfig other = (CapacityProviderScalingConfig) obj;
        return Objects.equals(maxVCpuCount(), other.maxVCpuCount())
                && Objects.equals(scalingModeAsString(), other.scalingModeAsString())
                && hasScalingPolicies() == other.hasScalingPolicies()
                && Objects.equals(scalingPolicies(), other.scalingPolicies());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityProviderScalingConfig").add("MaxVCpuCount", maxVCpuCount())
                .add("ScalingMode", scalingModeAsString())
                .add("ScalingPolicies", hasScalingPolicies() ? scalingPolicies() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MaxVCpuCount":
            return Optional.ofNullable(clazz.cast(maxVCpuCount()));
        case "ScalingMode":
            return Optional.ofNullable(clazz.cast(scalingModeAsString()));
        case "ScalingPolicies":
            return Optional.ofNullable(clazz.cast(scalingPolicies()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MaxVCpuCount", MAX_V_CPU_COUNT_FIELD);
        map.put("ScalingMode", SCALING_MODE_FIELD);
        map.put("ScalingPolicies", SCALING_POLICIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CapacityProviderScalingConfig, T> g) {
        return obj -> g.apply((CapacityProviderScalingConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityProviderScalingConfig> {
        /**
         * <p>
         * The maximum number of vCPUs that the capacity provider can provision across all compute instances.
         * </p>
         * 
         * @param maxVCpuCount
         *        The maximum number of vCPUs that the capacity provider can provision across all compute instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxVCpuCount(Integer maxVCpuCount);

        /**
         * <p>
         * The scaling mode that determines how the capacity provider responds to changes in demand.
         * </p>
         * 
         * @param scalingMode
         *        The scaling mode that determines how the capacity provider responds to changes in demand.
         * @see CapacityProviderScalingMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityProviderScalingMode
         */
        Builder scalingMode(String scalingMode);

        /**
         * <p>
         * The scaling mode that determines how the capacity provider responds to changes in demand.
         * </p>
         * 
         * @param scalingMode
         *        The scaling mode that determines how the capacity provider responds to changes in demand.
         * @see CapacityProviderScalingMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityProviderScalingMode
         */
        Builder scalingMode(CapacityProviderScalingMode scalingMode);

        /**
         * <p>
         * A list of scaling policies that define how the capacity provider scales compute instances based on metrics
         * and thresholds.
         * </p>
         * 
         * @param scalingPolicies
         *        A list of scaling policies that define how the capacity provider scales compute instances based on
         *        metrics and thresholds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicies(Collection<TargetTrackingScalingPolicy> scalingPolicies);

        /**
         * <p>
         * A list of scaling policies that define how the capacity provider scales compute instances based on metrics
         * and thresholds.
         * </p>
         * 
         * @param scalingPolicies
         *        A list of scaling policies that define how the capacity provider scales compute instances based on
         *        metrics and thresholds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicies(TargetTrackingScalingPolicy... scalingPolicies);

        /**
         * <p>
         * A list of scaling policies that define how the capacity provider scales compute instances based on metrics
         * and thresholds.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lambda.model.TargetTrackingScalingPolicy.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.lambda.model.TargetTrackingScalingPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lambda.model.TargetTrackingScalingPolicy.Builder#build()} is called
         * immediately and its result is passed to {@link #scalingPolicies(List<TargetTrackingScalingPolicy>)}.
         * 
         * @param scalingPolicies
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lambda.model.TargetTrackingScalingPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scalingPolicies(java.util.Collection<TargetTrackingScalingPolicy>)
         */
        Builder scalingPolicies(Consumer<TargetTrackingScalingPolicy.Builder>... scalingPolicies);
    }

    static final class BuilderImpl implements Builder {
        private Integer maxVCpuCount;

        private String scalingMode;

        private List<TargetTrackingScalingPolicy> scalingPolicies = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityProviderScalingConfig model) {
            maxVCpuCount(model.maxVCpuCount);
            scalingMode(model.scalingMode);
            scalingPolicies(model.scalingPolicies);
        }

        public final Integer getMaxVCpuCount() {
            return maxVCpuCount;
        }

        public final void setMaxVCpuCount(Integer maxVCpuCount) {
            this.maxVCpuCount = maxVCpuCount;
        }

        @Override
        public final Builder maxVCpuCount(Integer maxVCpuCount) {
            this.maxVCpuCount = maxVCpuCount;
            return this;
        }

        public final String getScalingMode() {
            return scalingMode;
        }

        public final void setScalingMode(String scalingMode) {
            this.scalingMode = scalingMode;
        }

        @Override
        public final Builder scalingMode(String scalingMode) {
            this.scalingMode = scalingMode;
            return this;
        }

        @Override
        public final Builder scalingMode(CapacityProviderScalingMode scalingMode) {
            this.scalingMode(scalingMode == null ? null : scalingMode.toString());
            return this;
        }

        public final List<TargetTrackingScalingPolicy.Builder> getScalingPolicies() {
            List<TargetTrackingScalingPolicy.Builder> result = CapacityProviderScalingPoliciesListCopier
                    .copyToBuilder(this.scalingPolicies);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setScalingPolicies(Collection<TargetTrackingScalingPolicy.BuilderImpl> scalingPolicies) {
            this.scalingPolicies = CapacityProviderScalingPoliciesListCopier.copyFromBuilder(scalingPolicies);
        }

        @Override
        public final Builder scalingPolicies(Collection<TargetTrackingScalingPolicy> scalingPolicies) {
            this.scalingPolicies = CapacityProviderScalingPoliciesListCopier.copy(scalingPolicies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scalingPolicies(TargetTrackingScalingPolicy... scalingPolicies) {
            scalingPolicies(Arrays.asList(scalingPolicies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scalingPolicies(Consumer<TargetTrackingScalingPolicy.Builder>... scalingPolicies) {
            scalingPolicies(Stream.of(scalingPolicies).map(c -> TargetTrackingScalingPolicy.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CapacityProviderScalingConfig build() {
            return new CapacityProviderScalingConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
