/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.lambda.LambdaAsyncClient;
import software.amazon.awssdk.services.lambda.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationResponse;
import software.amazon.awssdk.services.lambda.model.GetFunctionRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionResponse;
import software.amazon.awssdk.services.lambda.model.LambdaRequest;
import software.amazon.awssdk.services.lambda.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultLambdaAsyncWaiter implements LambdaAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final LambdaAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetFunctionResponse> functionExistsWaiter;

    private final AsyncWaiter<GetFunctionConfigurationResponse> functionActiveWaiter;

    private final AsyncWaiter<GetFunctionConfigurationResponse> functionUpdatedWaiter;

    private final AsyncWaiter<GetFunctionResponse> functionActiveV2Waiter;

    private final AsyncWaiter<GetFunctionResponse> functionUpdatedV2Waiter;

    private final AsyncWaiter<GetFunctionConfigurationResponse> publishedVersionActiveWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultLambdaAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = LambdaAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.functionExistsWaiter = AsyncWaiter.builder(GetFunctionResponse.class).acceptors(functionExistsWaiterAcceptors())
                .overrideConfiguration(functionExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.functionActiveWaiter = AsyncWaiter.builder(GetFunctionConfigurationResponse.class)
                .acceptors(functionActiveWaiterAcceptors())
                .overrideConfiguration(functionActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.functionUpdatedWaiter = AsyncWaiter.builder(GetFunctionConfigurationResponse.class)
                .acceptors(functionUpdatedWaiterAcceptors())
                .overrideConfiguration(functionUpdatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.functionActiveV2Waiter = AsyncWaiter.builder(GetFunctionResponse.class).acceptors(functionActiveV2WaiterAcceptors())
                .overrideConfiguration(functionActiveV2WaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.functionUpdatedV2Waiter = AsyncWaiter.builder(GetFunctionResponse.class)
                .acceptors(functionUpdatedV2WaiterAcceptors())
                .overrideConfiguration(functionUpdatedV2WaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.publishedVersionActiveWaiter = AsyncWaiter.builder(GetFunctionConfigurationResponse.class)
                .acceptors(publishedVersionActiveWaiterAcceptors())
                .overrideConfiguration(publishedVersionActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        return functionActiveWaiter.runAsync(() -> client
                .getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        return functionActiveWaiter.runAsync(
                () -> client.getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)),
                functionActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionActiveV2(GetFunctionRequest getFunctionRequest) {
        return functionActiveV2Waiter.runAsync(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionActiveV2(
            GetFunctionRequest getFunctionRequest, WaiterOverrideConfiguration overrideConfig) {
        return functionActiveV2Waiter.runAsync(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)),
                functionActiveV2WaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionExists(GetFunctionRequest getFunctionRequest) {
        return functionExistsWaiter.runAsync(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionExists(GetFunctionRequest getFunctionRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return functionExistsWaiter.runAsync(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)),
                functionExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionUpdated(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        return functionUpdatedWaiter.runAsync(() -> client
                .getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilFunctionUpdated(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        return functionUpdatedWaiter.runAsync(
                () -> client.getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)),
                functionUpdatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionUpdatedV2(GetFunctionRequest getFunctionRequest) {
        return functionUpdatedV2Waiter.runAsync(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionResponse>> waitUntilFunctionUpdatedV2(
            GetFunctionRequest getFunctionRequest, WaiterOverrideConfiguration overrideConfig) {
        return functionUpdatedV2Waiter.runAsync(() -> client.getFunction(applyWaitersUserAgent(getFunctionRequest)),
                functionUpdatedV2WaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilPublishedVersionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) {
        return publishedVersionActiveWaiter.runAsync(() -> client
                .getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFunctionConfigurationResponse>> waitUntilPublishedVersionActive(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest, WaiterOverrideConfiguration overrideConfig) {
        return publishedVersionActiveWaiter.runAsync(
                () -> client.getFunctionConfiguration(applyWaitersUserAgent(getFunctionConfigurationRequest)),
                publishedVersionActiveWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetFunctionResponse>> functionExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> functionActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Active");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Failed");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> functionUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("LastUpdateStatus").value(), "Successful");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("LastUpdateStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (LastUpdateStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("LastUpdateStatus").value(), "InProgress");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionResponse>> functionActiveV2WaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("State").value(), "Active");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Configuration").field("State").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Configuration.State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionResponse>> functionUpdatedV2WaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("LastUpdateStatus").value(), "Successful");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Configuration").field("LastUpdateStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Configuration.LastUpdateStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Configuration").field("LastUpdateStatus").value(), "InProgress");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> publishedVersionActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFunctionConfigurationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Active");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Failed");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration functionExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionActiveV2WaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(300);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration functionUpdatedV2WaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(300);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration publishedVersionActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(312);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static LambdaAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends LambdaRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements LambdaAsyncWaiter.Builder {
        private LambdaAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public LambdaAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public LambdaAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public LambdaAsyncWaiter.Builder client(LambdaAsyncClient client) {
            this.client = client;
            return this;
        }

        public LambdaAsyncWaiter build() {
            return new DefaultLambdaAsyncWaiter(this);
        }
    }
}
