/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class SignRequest extends KmsRequest implements ToCopyableBuilder<SignRequest.Builder, SignRequest> {
    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignRequest::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<SdkBytes> MESSAGE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(SignRequest::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<String> MESSAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignRequest::messageTypeAsString)).setter(setter(Builder::messageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageType").build()).build();

    private static final SdkField<List<String>> GRANT_TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SignRequest::grantTokens))
            .setter(setter(Builder::grantTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GrantTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SIGNING_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignRequest::signingAlgorithmAsString)).setter(setter(Builder::signingAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningAlgorithm").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_ID_FIELD, MESSAGE_FIELD,
            MESSAGE_TYPE_FIELD, GRANT_TOKENS_FIELD, SIGNING_ALGORITHM_FIELD));

    private final String keyId;

    private final SdkBytes message;

    private final String messageType;

    private final List<String> grantTokens;

    private final String signingAlgorithm;

    private SignRequest(BuilderImpl builder) {
        super(builder);
        this.keyId = builder.keyId;
        this.message = builder.message;
        this.messageType = builder.messageType;
        this.grantTokens = builder.grantTokens;
        this.signingAlgorithm = builder.signingAlgorithm;
    }

    /**
     * <p>
     * Identifies an asymmetric CMK. AWS KMS uses the private key in the asymmetric CMK to sign the message. The
     * <code>KeyUsage</code> type of the CMK must be <code>SIGN_VERIFY</code>. To find the <code>KeyUsage</code> of a
     * CMK, use the <a>DescribeKey</a> operation.
     * </p>
     * <p>
     * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias name,
     * prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must use the key ARN or
     * alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
     * alias ARN, use <a>ListAliases</a>.
     * </p>
     * 
     * @return Identifies an asymmetric CMK. AWS KMS uses the private key in the asymmetric CMK to sign the message. The
     *         <code>KeyUsage</code> type of the CMK must be <code>SIGN_VERIFY</code>. To find the <code>KeyUsage</code>
     *         of a CMK, use the <a>DescribeKey</a> operation.</p>
     *         <p>
     *         To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
     *         alias name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must
     *         use the key ARN or alias ARN.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name
     *         and alias ARN, use <a>ListAliases</a>.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * Specifies the message or message digest to sign. Messages can be 0-4096 bytes. To sign a larger message, provide
     * the message digest.
     * </p>
     * <p>
     * If you provide a message, AWS KMS generates a hash digest of the message and then signs it.
     * </p>
     * 
     * @return Specifies the message or message digest to sign. Messages can be 0-4096 bytes. To sign a larger message,
     *         provide the message digest.</p>
     *         <p>
     *         If you provide a message, AWS KMS generates a hash digest of the message and then signs it.
     */
    public SdkBytes message() {
        return message;
    }

    /**
     * <p>
     * Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest. To indicate
     * a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageType} will
     * return {@link MessageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #messageTypeAsString}.
     * </p>
     * 
     * @return Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest. To
     *         indicate a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>.
     * @see MessageType
     */
    public MessageType messageType() {
        return MessageType.fromValue(messageType);
    }

    /**
     * <p>
     * Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest. To indicate
     * a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageType} will
     * return {@link MessageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #messageTypeAsString}.
     * </p>
     * 
     * @return Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest. To
     *         indicate a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>.
     * @see MessageType
     */
    public String messageTypeAsString() {
        return messageType;
    }

    /**
     * Returns true if the GrantTokens property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGrantTokens() {
        return grantTokens != null && !(grantTokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in the
     * <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGrantTokens()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of grant tokens.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a>
     *         in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public List<String> grantTokens() {
        return grantTokens;
    }

    /**
     * <p>
     * Specifies the signing algorithm to use when signing the message.
     * </p>
     * <p>
     * Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlgorithmSpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return Specifies the signing algorithm to use when signing the message. </p>
     *         <p>
     *         Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
     * @see SigningAlgorithmSpec
     */
    public SigningAlgorithmSpec signingAlgorithm() {
        return SigningAlgorithmSpec.fromValue(signingAlgorithm);
    }

    /**
     * <p>
     * Specifies the signing algorithm to use when signing the message.
     * </p>
     * <p>
     * Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlgorithmSpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return Specifies the signing algorithm to use when signing the message. </p>
     *         <p>
     *         Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
     * @see SigningAlgorithmSpec
     */
    public String signingAlgorithmAsString() {
        return signingAlgorithm;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(messageTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(grantTokens());
        hashCode = 31 * hashCode + Objects.hashCode(signingAlgorithmAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SignRequest)) {
            return false;
        }
        SignRequest other = (SignRequest) obj;
        return Objects.equals(keyId(), other.keyId()) && Objects.equals(message(), other.message())
                && Objects.equals(messageTypeAsString(), other.messageTypeAsString())
                && Objects.equals(grantTokens(), other.grantTokens())
                && Objects.equals(signingAlgorithmAsString(), other.signingAlgorithmAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SignRequest").add("KeyId", keyId())
                .add("Message", message() == null ? null : "*** Sensitive Data Redacted ***")
                .add("MessageType", messageTypeAsString()).add("GrantTokens", grantTokens())
                .add("SigningAlgorithm", signingAlgorithmAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "MessageType":
            return Optional.ofNullable(clazz.cast(messageTypeAsString()));
        case "GrantTokens":
            return Optional.ofNullable(clazz.cast(grantTokens()));
        case "SigningAlgorithm":
            return Optional.ofNullable(clazz.cast(signingAlgorithmAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SignRequest, T> g) {
        return obj -> g.apply((SignRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, SignRequest> {
        /**
         * <p>
         * Identifies an asymmetric CMK. AWS KMS uses the private key in the asymmetric CMK to sign the message. The
         * <code>KeyUsage</code> type of the CMK must be <code>SIGN_VERIFY</code>. To find the <code>KeyUsage</code> of
         * a CMK, use the <a>DescribeKey</a> operation.
         * </p>
         * <p>
         * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias
         * name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must use the key
         * ARN or alias ARN.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
         * alias ARN, use <a>ListAliases</a>.
         * </p>
         * 
         * @param keyId
         *        Identifies an asymmetric CMK. AWS KMS uses the private key in the asymmetric CMK to sign the message.
         *        The <code>KeyUsage</code> type of the CMK must be <code>SIGN_VERIFY</code>. To find the
         *        <code>KeyUsage</code> of a CMK, use the <a>DescribeKey</a> operation.</p>
         *        <p>
         *        To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
         *        alias name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you
         *        must use the key ARN or alias ARN.
         *        </p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias
         *        name and alias ARN, use <a>ListAliases</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Specifies the message or message digest to sign. Messages can be 0-4096 bytes. To sign a larger message,
         * provide the message digest.
         * </p>
         * <p>
         * If you provide a message, AWS KMS generates a hash digest of the message and then signs it.
         * </p>
         * 
         * @param message
         *        Specifies the message or message digest to sign. Messages can be 0-4096 bytes. To sign a larger
         *        message, provide the message digest.</p>
         *        <p>
         *        If you provide a message, AWS KMS generates a hash digest of the message and then signs it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(SdkBytes message);

        /**
         * <p>
         * Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest. To
         * indicate a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>.
         * </p>
         * 
         * @param messageType
         *        Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest.
         *        To indicate a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>
         *        .
         * @see MessageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageType
         */
        Builder messageType(String messageType);

        /**
         * <p>
         * Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest. To
         * indicate a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>.
         * </p>
         * 
         * @param messageType
         *        Tells AWS KMS whether the value of the <code>Message</code> parameter is a message or message digest.
         *        To indicate a message, enter <code>RAW</code>. To indicate a message digest, enter <code>DIGEST</code>
         *        .
         * @see MessageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageType
         */
        Builder messageType(MessageType messageType);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(Collection<String> grantTokens);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(String... grantTokens);

        /**
         * <p>
         * Specifies the signing algorithm to use when signing the message.
         * </p>
         * <p>
         * Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
         * </p>
         * 
         * @param signingAlgorithm
         *        Specifies the signing algorithm to use when signing the message. </p>
         *        <p>
         *        Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
         * @see SigningAlgorithmSpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlgorithmSpec
         */
        Builder signingAlgorithm(String signingAlgorithm);

        /**
         * <p>
         * Specifies the signing algorithm to use when signing the message.
         * </p>
         * <p>
         * Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
         * </p>
         * 
         * @param signingAlgorithm
         *        Specifies the signing algorithm to use when signing the message. </p>
         *        <p>
         *        Choose an algorithm that is compatible with the type and size of the specified asymmetric CMK.
         * @see SigningAlgorithmSpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlgorithmSpec
         */
        Builder signingAlgorithm(SigningAlgorithmSpec signingAlgorithm);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private String keyId;

        private SdkBytes message;

        private String messageType;

        private List<String> grantTokens = DefaultSdkAutoConstructList.getInstance();

        private String signingAlgorithm;

        private BuilderImpl() {
        }

        private BuilderImpl(SignRequest model) {
            super(model);
            keyId(model.keyId);
            message(model.message);
            messageType(model.messageType);
            grantTokens(model.grantTokens);
            signingAlgorithm(model.signingAlgorithm);
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final ByteBuffer getMessage() {
            return message == null ? null : message.asByteBuffer();
        }

        @Override
        public final Builder message(SdkBytes message) {
            this.message = StandardMemberCopier.copy(message);
            return this;
        }

        public final void setMessage(ByteBuffer message) {
            message(message == null ? null : SdkBytes.fromByteBuffer(message));
        }

        public final String getMessageTypeAsString() {
            return messageType;
        }

        @Override
        public final Builder messageType(String messageType) {
            this.messageType = messageType;
            return this;
        }

        @Override
        public final Builder messageType(MessageType messageType) {
            this.messageType(messageType == null ? null : messageType.toString());
            return this;
        }

        public final void setMessageType(String messageType) {
            this.messageType = messageType;
        }

        public final Collection<String> getGrantTokens() {
            return grantTokens;
        }

        @Override
        public final Builder grantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder grantTokens(String... grantTokens) {
            grantTokens(Arrays.asList(grantTokens));
            return this;
        }

        public final void setGrantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
        }

        public final String getSigningAlgorithmAsString() {
            return signingAlgorithm;
        }

        @Override
        public final Builder signingAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
            return this;
        }

        @Override
        public final Builder signingAlgorithm(SigningAlgorithmSpec signingAlgorithm) {
            this.signingAlgorithm(signingAlgorithm == null ? null : signingAlgorithm.toString());
            return this;
        }

        public final void setSigningAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SignRequest build() {
            return new SignRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
