/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.nio.ByteBuffer;
import java.time.Instant;
import javax.annotation.Generated;
import software.amazon.awssdk.AmazonWebServiceRequest;
import software.amazon.awssdk.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class ImportKeyMaterialRequest extends AmazonWebServiceRequest implements
        ToCopyableBuilder<ImportKeyMaterialRequest.Builder, ImportKeyMaterialRequest> {
    private final String keyId;

    private final ByteBuffer importToken;

    private final ByteBuffer encryptedKeyMaterial;

    private final Instant validTo;

    private final String expirationModel;

    private ImportKeyMaterialRequest(BuilderImpl builder) {
        this.keyId = builder.keyId;
        this.importToken = builder.importToken;
        this.encryptedKeyMaterial = builder.encryptedKeyMaterial;
        this.validTo = builder.validTo;
        this.expirationModel = builder.expirationModel;
    }

    /**
     * <p>
     * The identifier of the CMK to import the key material into. The CMK's <code>Origin</code> must be
     * <code>EXTERNAL</code>.
     * </p>
     * <p>
     * A valid identifier is the unique key ID or the Amazon Resource Name (ARN) of the CMK. Examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Unique key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier of the CMK to import the key material into. The CMK's <code>Origin</code> must be
     *         <code>EXTERNAL</code>.</p>
     *         <p>
     *         A valid identifier is the unique key ID or the Amazon Resource Name (ARN) of the CMK. Examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Unique key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * The import token that you received in the response to a previous <a>GetParametersForImport</a> request. It must
     * be from the same response that contained the public key that you used to encrypt the key material.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return The import token that you received in the response to a previous <a>GetParametersForImport</a> request.
     *         It must be from the same response that contained the public key that you used to encrypt the key
     *         material.
     */
    public ByteBuffer importToken() {
        return importToken == null ? null : importToken.asReadOnlyBuffer();
    }

    /**
     * <p>
     * The encrypted key material to import. It must be encrypted with the public key that you received in the response
     * to a previous <a>GetParametersForImport</a> request, using the wrapping algorithm that you specified in that
     * request.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return The encrypted key material to import. It must be encrypted with the public key that you received in the
     *         response to a previous <a>GetParametersForImport</a> request, using the wrapping algorithm that you
     *         specified in that request.
     */
    public ByteBuffer encryptedKeyMaterial() {
        return encryptedKeyMaterial == null ? null : encryptedKeyMaterial.asReadOnlyBuffer();
    }

    /**
     * <p>
     * The time at which the imported key material expires. When the key material expires, AWS KMS deletes the key
     * material and the CMK becomes unusable. You must omit this parameter when the <code>ExpirationModel</code>
     * parameter is set to <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>. Otherwise it is required.
     * </p>
     * 
     * @return The time at which the imported key material expires. When the key material expires, AWS KMS deletes the
     *         key material and the CMK becomes unusable. You must omit this parameter when the
     *         <code>ExpirationModel</code> parameter is set to <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>. Otherwise it
     *         is required.
     */
    public Instant validTo() {
        return validTo;
    }

    /**
     * <p>
     * Specifies whether the key material expires. The default is <code>KEY_MATERIAL_EXPIRES</code>, in which case you
     * must include the <code>ValidTo</code> parameter. When this parameter is set to
     * <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>, you must omit the <code>ValidTo</code> parameter.
     * </p>
     * 
     * @return Specifies whether the key material expires. The default is <code>KEY_MATERIAL_EXPIRES</code>, in which
     *         case you must include the <code>ValidTo</code> parameter. When this parameter is set to
     *         <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>, you must omit the <code>ValidTo</code> parameter.
     * @see ExpirationModelType
     */
    public String expirationModel() {
        return expirationModel;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((keyId() == null) ? 0 : keyId().hashCode());
        hashCode = 31 * hashCode + ((importToken() == null) ? 0 : importToken().hashCode());
        hashCode = 31 * hashCode + ((encryptedKeyMaterial() == null) ? 0 : encryptedKeyMaterial().hashCode());
        hashCode = 31 * hashCode + ((validTo() == null) ? 0 : validTo().hashCode());
        hashCode = 31 * hashCode + ((expirationModel() == null) ? 0 : expirationModel().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportKeyMaterialRequest)) {
            return false;
        }
        ImportKeyMaterialRequest other = (ImportKeyMaterialRequest) obj;
        if (other.keyId() == null ^ this.keyId() == null) {
            return false;
        }
        if (other.keyId() != null && !other.keyId().equals(this.keyId())) {
            return false;
        }
        if (other.importToken() == null ^ this.importToken() == null) {
            return false;
        }
        if (other.importToken() != null && !other.importToken().equals(this.importToken())) {
            return false;
        }
        if (other.encryptedKeyMaterial() == null ^ this.encryptedKeyMaterial() == null) {
            return false;
        }
        if (other.encryptedKeyMaterial() != null && !other.encryptedKeyMaterial().equals(this.encryptedKeyMaterial())) {
            return false;
        }
        if (other.validTo() == null ^ this.validTo() == null) {
            return false;
        }
        if (other.validTo() != null && !other.validTo().equals(this.validTo())) {
            return false;
        }
        if (other.expirationModel() == null ^ this.expirationModel() == null) {
            return false;
        }
        if (other.expirationModel() != null && !other.expirationModel().equals(this.expirationModel())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (keyId() != null) {
            sb.append("KeyId: ").append(keyId()).append(",");
        }
        if (importToken() != null) {
            sb.append("ImportToken: ").append(importToken()).append(",");
        }
        if (encryptedKeyMaterial() != null) {
            sb.append("EncryptedKeyMaterial: ").append(encryptedKeyMaterial()).append(",");
        }
        if (validTo() != null) {
            sb.append("ValidTo: ").append(validTo()).append(",");
        }
        if (expirationModel() != null) {
            sb.append("ExpirationModel: ").append(expirationModel()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, ImportKeyMaterialRequest> {
        /**
         * <p>
         * The identifier of the CMK to import the key material into. The CMK's <code>Origin</code> must be
         * <code>EXTERNAL</code>.
         * </p>
         * <p>
         * A valid identifier is the unique key ID or the Amazon Resource Name (ARN) of the CMK. Examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Unique key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyId
         *        The identifier of the CMK to import the key material into. The CMK's <code>Origin</code> must be
         *        <code>EXTERNAL</code>.</p>
         *        <p>
         *        A valid identifier is the unique key ID or the Amazon Resource Name (ARN) of the CMK. Examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Unique key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * The import token that you received in the response to a previous <a>GetParametersForImport</a> request. It
         * must be from the same response that contained the public key that you used to encrypt the key material.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param importToken
         *        The import token that you received in the response to a previous <a>GetParametersForImport</a>
         *        request. It must be from the same response that contained the public key that you used to encrypt the
         *        key material.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importToken(ByteBuffer importToken);

        /**
         * <p>
         * The encrypted key material to import. It must be encrypted with the public key that you received in the
         * response to a previous <a>GetParametersForImport</a> request, using the wrapping algorithm that you specified
         * in that request.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param encryptedKeyMaterial
         *        The encrypted key material to import. It must be encrypted with the public key that you received in
         *        the response to a previous <a>GetParametersForImport</a> request, using the wrapping algorithm that
         *        you specified in that request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptedKeyMaterial(ByteBuffer encryptedKeyMaterial);

        /**
         * <p>
         * The time at which the imported key material expires. When the key material expires, AWS KMS deletes the key
         * material and the CMK becomes unusable. You must omit this parameter when the <code>ExpirationModel</code>
         * parameter is set to <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>. Otherwise it is required.
         * </p>
         * 
         * @param validTo
         *        The time at which the imported key material expires. When the key material expires, AWS KMS deletes
         *        the key material and the CMK becomes unusable. You must omit this parameter when the
         *        <code>ExpirationModel</code> parameter is set to <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>. Otherwise
         *        it is required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validTo(Instant validTo);

        /**
         * <p>
         * Specifies whether the key material expires. The default is <code>KEY_MATERIAL_EXPIRES</code>, in which case
         * you must include the <code>ValidTo</code> parameter. When this parameter is set to
         * <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>, you must omit the <code>ValidTo</code> parameter.
         * </p>
         * 
         * @param expirationModel
         *        Specifies whether the key material expires. The default is <code>KEY_MATERIAL_EXPIRES</code>, in which
         *        case you must include the <code>ValidTo</code> parameter. When this parameter is set to
         *        <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>, you must omit the <code>ValidTo</code> parameter.
         * @see ExpirationModelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpirationModelType
         */
        Builder expirationModel(String expirationModel);

        /**
         * <p>
         * Specifies whether the key material expires. The default is <code>KEY_MATERIAL_EXPIRES</code>, in which case
         * you must include the <code>ValidTo</code> parameter. When this parameter is set to
         * <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>, you must omit the <code>ValidTo</code> parameter.
         * </p>
         * 
         * @param expirationModel
         *        Specifies whether the key material expires. The default is <code>KEY_MATERIAL_EXPIRES</code>, in which
         *        case you must include the <code>ValidTo</code> parameter. When this parameter is set to
         *        <code>KEY_MATERIAL_DOES_NOT_EXPIRE</code>, you must omit the <code>ValidTo</code> parameter.
         * @see ExpirationModelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpirationModelType
         */
        Builder expirationModel(ExpirationModelType expirationModel);
    }

    private static final class BuilderImpl implements Builder {
        private String keyId;

        private ByteBuffer importToken;

        private ByteBuffer encryptedKeyMaterial;

        private Instant validTo;

        private String expirationModel;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportKeyMaterialRequest model) {
            setKeyId(model.keyId);
            setImportToken(model.importToken);
            setEncryptedKeyMaterial(model.encryptedKeyMaterial);
            setValidTo(model.validTo);
            setExpirationModel(model.expirationModel);
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final ByteBuffer getImportToken() {
            return importToken;
        }

        @Override
        public final Builder importToken(ByteBuffer importToken) {
            this.importToken = StandardMemberCopier.copy(importToken);
            return this;
        }

        public final void setImportToken(ByteBuffer importToken) {
            this.importToken = StandardMemberCopier.copy(importToken);
        }

        public final ByteBuffer getEncryptedKeyMaterial() {
            return encryptedKeyMaterial;
        }

        @Override
        public final Builder encryptedKeyMaterial(ByteBuffer encryptedKeyMaterial) {
            this.encryptedKeyMaterial = StandardMemberCopier.copy(encryptedKeyMaterial);
            return this;
        }

        public final void setEncryptedKeyMaterial(ByteBuffer encryptedKeyMaterial) {
            this.encryptedKeyMaterial = StandardMemberCopier.copy(encryptedKeyMaterial);
        }

        public final Instant getValidTo() {
            return validTo;
        }

        @Override
        public final Builder validTo(Instant validTo) {
            this.validTo = validTo;
            return this;
        }

        public final void setValidTo(Instant validTo) {
            this.validTo = validTo;
        }

        public final String getExpirationModel() {
            return expirationModel;
        }

        @Override
        public final Builder expirationModel(String expirationModel) {
            this.expirationModel = expirationModel;
            return this;
        }

        @Override
        public final Builder expirationModel(ExpirationModelType expirationModel) {
            this.expirationModel(expirationModel.toString());
            return this;
        }

        public final void setExpirationModel(String expirationModel) {
            this.expirationModel = expirationModel;
        }

        public final void setExpirationModel(ExpirationModelType expirationModel) {
            this.expirationModel(expirationModel.toString());
        }

        @Override
        public ImportKeyMaterialRequest build() {
            return new ImportKeyMaterialRequest(this);
        }
    }
}
