/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.kms.transform.GrantListEntryMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an entry in a list of grants.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GrantListEntry implements StructuredPojo, ToCopyableBuilder<GrantListEntry.Builder, GrantListEntry> {
    private final String keyId;

    private final String grantId;

    private final String name;

    private final Instant creationDate;

    private final String granteePrincipal;

    private final String retiringPrincipal;

    private final String issuingAccount;

    private final List<String> operations;

    private final GrantConstraints constraints;

    private GrantListEntry(BuilderImpl builder) {
        this.keyId = builder.keyId;
        this.grantId = builder.grantId;
        this.name = builder.name;
        this.creationDate = builder.creationDate;
        this.granteePrincipal = builder.granteePrincipal;
        this.retiringPrincipal = builder.retiringPrincipal;
        this.issuingAccount = builder.issuingAccount;
        this.operations = builder.operations;
        this.constraints = builder.constraints;
    }

    /**
     * <p>
     * The unique identifier for the customer master key (CMK) to which the grant applies.
     * </p>
     * 
     * @return The unique identifier for the customer master key (CMK) to which the grant applies.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * The unique identifier for the grant.
     * </p>
     * 
     * @return The unique identifier for the grant.
     */
    public String grantId() {
        return grantId;
    }

    /**
     * <p>
     * The friendly name that identifies the grant. If a name was provided in the <a>CreateGrant</a> request, that name
     * is returned. Otherwise this value is null.
     * </p>
     * 
     * @return The friendly name that identifies the grant. If a name was provided in the <a>CreateGrant</a> request,
     *         that name is returned. Otherwise this value is null.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The date and time when the grant was created.
     * </p>
     * 
     * @return The date and time when the grant was created.
     */
    public Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The principal that receives the grant's permissions.
     * </p>
     * 
     * @return The principal that receives the grant's permissions.
     */
    public String granteePrincipal() {
        return granteePrincipal;
    }

    /**
     * <p>
     * The principal that can retire the grant.
     * </p>
     * 
     * @return The principal that can retire the grant.
     */
    public String retiringPrincipal() {
        return retiringPrincipal;
    }

    /**
     * <p>
     * The AWS account under which the grant was issued.
     * </p>
     * 
     * @return The AWS account under which the grant was issued.
     */
    public String issuingAccount() {
        return issuingAccount;
    }

    /**
     * <p>
     * The list of operations permitted by the grant.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of operations permitted by the grant.
     * @see GrantOperation
     */
    public List<String> operations() {
        return operations;
    }

    /**
     * <p>
     * A list of key-value pairs that must be present in the encryption context of certain subsequent operations that
     * the grant allows.
     * </p>
     * 
     * @return A list of key-value pairs that must be present in the encryption context of certain subsequent operations
     *         that the grant allows.
     */
    public GrantConstraints constraints() {
        return constraints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((keyId() == null) ? 0 : keyId().hashCode());
        hashCode = 31 * hashCode + ((grantId() == null) ? 0 : grantId().hashCode());
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((creationDate() == null) ? 0 : creationDate().hashCode());
        hashCode = 31 * hashCode + ((granteePrincipal() == null) ? 0 : granteePrincipal().hashCode());
        hashCode = 31 * hashCode + ((retiringPrincipal() == null) ? 0 : retiringPrincipal().hashCode());
        hashCode = 31 * hashCode + ((issuingAccount() == null) ? 0 : issuingAccount().hashCode());
        hashCode = 31 * hashCode + ((operations() == null) ? 0 : operations().hashCode());
        hashCode = 31 * hashCode + ((constraints() == null) ? 0 : constraints().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GrantListEntry)) {
            return false;
        }
        GrantListEntry other = (GrantListEntry) obj;
        if (other.keyId() == null ^ this.keyId() == null) {
            return false;
        }
        if (other.keyId() != null && !other.keyId().equals(this.keyId())) {
            return false;
        }
        if (other.grantId() == null ^ this.grantId() == null) {
            return false;
        }
        if (other.grantId() != null && !other.grantId().equals(this.grantId())) {
            return false;
        }
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.creationDate() == null ^ this.creationDate() == null) {
            return false;
        }
        if (other.creationDate() != null && !other.creationDate().equals(this.creationDate())) {
            return false;
        }
        if (other.granteePrincipal() == null ^ this.granteePrincipal() == null) {
            return false;
        }
        if (other.granteePrincipal() != null && !other.granteePrincipal().equals(this.granteePrincipal())) {
            return false;
        }
        if (other.retiringPrincipal() == null ^ this.retiringPrincipal() == null) {
            return false;
        }
        if (other.retiringPrincipal() != null && !other.retiringPrincipal().equals(this.retiringPrincipal())) {
            return false;
        }
        if (other.issuingAccount() == null ^ this.issuingAccount() == null) {
            return false;
        }
        if (other.issuingAccount() != null && !other.issuingAccount().equals(this.issuingAccount())) {
            return false;
        }
        if (other.operations() == null ^ this.operations() == null) {
            return false;
        }
        if (other.operations() != null && !other.operations().equals(this.operations())) {
            return false;
        }
        if (other.constraints() == null ^ this.constraints() == null) {
            return false;
        }
        if (other.constraints() != null && !other.constraints().equals(this.constraints())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (keyId() != null) {
            sb.append("KeyId: ").append(keyId()).append(",");
        }
        if (grantId() != null) {
            sb.append("GrantId: ").append(grantId()).append(",");
        }
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (creationDate() != null) {
            sb.append("CreationDate: ").append(creationDate()).append(",");
        }
        if (granteePrincipal() != null) {
            sb.append("GranteePrincipal: ").append(granteePrincipal()).append(",");
        }
        if (retiringPrincipal() != null) {
            sb.append("RetiringPrincipal: ").append(retiringPrincipal()).append(",");
        }
        if (issuingAccount() != null) {
            sb.append("IssuingAccount: ").append(issuingAccount()).append(",");
        }
        if (operations() != null) {
            sb.append("Operations: ").append(operations()).append(",");
        }
        if (constraints() != null) {
            sb.append("Constraints: ").append(constraints()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        GrantListEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, GrantListEntry> {
        /**
         * <p>
         * The unique identifier for the customer master key (CMK) to which the grant applies.
         * </p>
         * 
         * @param keyId
         *        The unique identifier for the customer master key (CMK) to which the grant applies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * The unique identifier for the grant.
         * </p>
         * 
         * @param grantId
         *        The unique identifier for the grant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantId(String grantId);

        /**
         * <p>
         * The friendly name that identifies the grant. If a name was provided in the <a>CreateGrant</a> request, that
         * name is returned. Otherwise this value is null.
         * </p>
         * 
         * @param name
         *        The friendly name that identifies the grant. If a name was provided in the <a>CreateGrant</a> request,
         *        that name is returned. Otherwise this value is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The date and time when the grant was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the grant was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The principal that receives the grant's permissions.
         * </p>
         * 
         * @param granteePrincipal
         *        The principal that receives the grant's permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder granteePrincipal(String granteePrincipal);

        /**
         * <p>
         * The principal that can retire the grant.
         * </p>
         * 
         * @param retiringPrincipal
         *        The principal that can retire the grant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retiringPrincipal(String retiringPrincipal);

        /**
         * <p>
         * The AWS account under which the grant was issued.
         * </p>
         * 
         * @param issuingAccount
         *        The AWS account under which the grant was issued.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuingAccount(String issuingAccount);

        /**
         * <p>
         * The list of operations permitted by the grant.
         * </p>
         * 
         * @param operations
         *        The list of operations permitted by the grant.
         * @see GrantOperation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GrantOperation
         */
        Builder operations(Collection<String> operations);

        /**
         * <p>
         * The list of operations permitted by the grant.
         * </p>
         * 
         * @param operations
         *        The list of operations permitted by the grant.
         * @see GrantOperation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GrantOperation
         */
        Builder operations(String... operations);

        /**
         * <p>
         * The list of operations permitted by the grant.
         * </p>
         * 
         * @param operations
         *        The list of operations permitted by the grant.
         * @see GrantOperation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GrantOperation
         */
        Builder operations(GrantOperation... operations);

        /**
         * <p>
         * A list of key-value pairs that must be present in the encryption context of certain subsequent operations
         * that the grant allows.
         * </p>
         * 
         * @param constraints
         *        A list of key-value pairs that must be present in the encryption context of certain subsequent
         *        operations that the grant allows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constraints(GrantConstraints constraints);
    }

    private static final class BuilderImpl implements Builder {
        private String keyId;

        private String grantId;

        private String name;

        private Instant creationDate;

        private String granteePrincipal;

        private String retiringPrincipal;

        private String issuingAccount;

        private List<String> operations;

        private GrantConstraints constraints;

        private BuilderImpl() {
        }

        private BuilderImpl(GrantListEntry model) {
            setKeyId(model.keyId);
            setGrantId(model.grantId);
            setName(model.name);
            setCreationDate(model.creationDate);
            setGranteePrincipal(model.granteePrincipal);
            setRetiringPrincipal(model.retiringPrincipal);
            setIssuingAccount(model.issuingAccount);
            setOperations(model.operations);
            setConstraints(model.constraints);
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final String getGrantId() {
            return grantId;
        }

        @Override
        public final Builder grantId(String grantId) {
            this.grantId = grantId;
            return this;
        }

        public final void setGrantId(String grantId) {
            this.grantId = grantId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final String getGranteePrincipal() {
            return granteePrincipal;
        }

        @Override
        public final Builder granteePrincipal(String granteePrincipal) {
            this.granteePrincipal = granteePrincipal;
            return this;
        }

        public final void setGranteePrincipal(String granteePrincipal) {
            this.granteePrincipal = granteePrincipal;
        }

        public final String getRetiringPrincipal() {
            return retiringPrincipal;
        }

        @Override
        public final Builder retiringPrincipal(String retiringPrincipal) {
            this.retiringPrincipal = retiringPrincipal;
            return this;
        }

        public final void setRetiringPrincipal(String retiringPrincipal) {
            this.retiringPrincipal = retiringPrincipal;
        }

        public final String getIssuingAccount() {
            return issuingAccount;
        }

        @Override
        public final Builder issuingAccount(String issuingAccount) {
            this.issuingAccount = issuingAccount;
            return this;
        }

        public final void setIssuingAccount(String issuingAccount) {
            this.issuingAccount = issuingAccount;
        }

        public final Collection<String> getOperations() {
            return operations;
        }

        @Override
        public final Builder operations(Collection<String> operations) {
            this.operations = GrantOperationListCopier.copy(operations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operations(String... operations) {
            operations(Arrays.asList(operations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operations(GrantOperation... operations) {
            operations(Arrays.asList(operations).stream().map(Object::toString).collect(Collectors.toList()));
            return this;
        }

        public final void setOperations(Collection<String> operations) {
            this.operations = GrantOperationListCopier.copy(operations);
        }

        @SafeVarargs
        public final void setOperations(String... operations) {
            operations(Arrays.asList(operations));
        }

        @SafeVarargs
        public final void setOperations(GrantOperation... operations) {
            operations(Arrays.asList(operations).stream().map(Object::toString).collect(Collectors.toList()));
        }

        public final GrantConstraints getConstraints() {
            return constraints;
        }

        @Override
        public final Builder constraints(GrantConstraints constraints) {
            this.constraints = constraints;
            return this;
        }

        public final void setConstraints(GrantConstraints constraints) {
            this.constraints = constraints;
        }

        @Override
        public GrantListEntry build() {
            return new GrantListEntry(this);
        }
    }
}
