/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalyticsv2.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies either the application code, or the location of the application code, for a Flink-based Kinesis Data
 * Analytics application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CodeContent implements SdkPojo, Serializable, ToCopyableBuilder<CodeContent.Builder, CodeContent> {
    private static final SdkField<String> TEXT_CONTENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TextContent").getter(getter(CodeContent::textContent)).setter(setter(Builder::textContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextContent").build()).build();

    private static final SdkField<SdkBytes> ZIP_FILE_CONTENT_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("ZipFileContent").getter(getter(CodeContent::zipFileContent)).setter(setter(Builder::zipFileContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZipFileContent").build()).build();

    private static final SdkField<S3ContentLocation> S3_CONTENT_LOCATION_FIELD = SdkField
            .<S3ContentLocation> builder(MarshallingType.SDK_POJO).memberName("S3ContentLocation")
            .getter(getter(CodeContent::s3ContentLocation)).setter(setter(Builder::s3ContentLocation))
            .constructor(S3ContentLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3ContentLocation").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TEXT_CONTENT_FIELD,
            ZIP_FILE_CONTENT_FIELD, S3_CONTENT_LOCATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String textContent;

    private final SdkBytes zipFileContent;

    private final S3ContentLocation s3ContentLocation;

    private CodeContent(BuilderImpl builder) {
        this.textContent = builder.textContent;
        this.zipFileContent = builder.zipFileContent;
        this.s3ContentLocation = builder.s3ContentLocation;
    }

    /**
     * <p>
     * The text-format code for a Flink-based Kinesis Data Analytics application.
     * </p>
     * 
     * @return The text-format code for a Flink-based Kinesis Data Analytics application.
     */
    public final String textContent() {
        return textContent;
    }

    /**
     * <p>
     * The zip-format code for a Flink-based Kinesis Data Analytics application.
     * </p>
     * 
     * @return The zip-format code for a Flink-based Kinesis Data Analytics application.
     */
    public final SdkBytes zipFileContent() {
        return zipFileContent;
    }

    /**
     * <p>
     * Information about the Amazon S3 bucket containing the application code.
     * </p>
     * 
     * @return Information about the Amazon S3 bucket containing the application code.
     */
    public final S3ContentLocation s3ContentLocation() {
        return s3ContentLocation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(textContent());
        hashCode = 31 * hashCode + Objects.hashCode(zipFileContent());
        hashCode = 31 * hashCode + Objects.hashCode(s3ContentLocation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CodeContent)) {
            return false;
        }
        CodeContent other = (CodeContent) obj;
        return Objects.equals(textContent(), other.textContent()) && Objects.equals(zipFileContent(), other.zipFileContent())
                && Objects.equals(s3ContentLocation(), other.s3ContentLocation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CodeContent").add("TextContent", textContent()).add("ZipFileContent", zipFileContent())
                .add("S3ContentLocation", s3ContentLocation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TextContent":
            return Optional.ofNullable(clazz.cast(textContent()));
        case "ZipFileContent":
            return Optional.ofNullable(clazz.cast(zipFileContent()));
        case "S3ContentLocation":
            return Optional.ofNullable(clazz.cast(s3ContentLocation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CodeContent, T> g) {
        return obj -> g.apply((CodeContent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CodeContent> {
        /**
         * <p>
         * The text-format code for a Flink-based Kinesis Data Analytics application.
         * </p>
         * 
         * @param textContent
         *        The text-format code for a Flink-based Kinesis Data Analytics application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textContent(String textContent);

        /**
         * <p>
         * The zip-format code for a Flink-based Kinesis Data Analytics application.
         * </p>
         * 
         * @param zipFileContent
         *        The zip-format code for a Flink-based Kinesis Data Analytics application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zipFileContent(SdkBytes zipFileContent);

        /**
         * <p>
         * Information about the Amazon S3 bucket containing the application code.
         * </p>
         * 
         * @param s3ContentLocation
         *        Information about the Amazon S3 bucket containing the application code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ContentLocation(S3ContentLocation s3ContentLocation);

        /**
         * <p>
         * Information about the Amazon S3 bucket containing the application code.
         * </p>
         * This is a convenience that creates an instance of the {@link S3ContentLocation.Builder} avoiding the need to
         * create one manually via {@link S3ContentLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3ContentLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #s3ContentLocation(S3ContentLocation)}.
         * 
         * @param s3ContentLocation
         *        a consumer that will call methods on {@link S3ContentLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3ContentLocation(S3ContentLocation)
         */
        default Builder s3ContentLocation(Consumer<S3ContentLocation.Builder> s3ContentLocation) {
            return s3ContentLocation(S3ContentLocation.builder().applyMutation(s3ContentLocation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String textContent;

        private SdkBytes zipFileContent;

        private S3ContentLocation s3ContentLocation;

        private BuilderImpl() {
        }

        private BuilderImpl(CodeContent model) {
            textContent(model.textContent);
            zipFileContent(model.zipFileContent);
            s3ContentLocation(model.s3ContentLocation);
        }

        public final String getTextContent() {
            return textContent;
        }

        @Override
        public final Builder textContent(String textContent) {
            this.textContent = textContent;
            return this;
        }

        public final void setTextContent(String textContent) {
            this.textContent = textContent;
        }

        public final ByteBuffer getZipFileContent() {
            return zipFileContent == null ? null : zipFileContent.asByteBuffer();
        }

        @Override
        public final Builder zipFileContent(SdkBytes zipFileContent) {
            this.zipFileContent = StandardMemberCopier.copy(zipFileContent);
            return this;
        }

        public final void setZipFileContent(ByteBuffer zipFileContent) {
            zipFileContent(zipFileContent == null ? null : SdkBytes.fromByteBuffer(zipFileContent));
        }

        public final S3ContentLocation.Builder getS3ContentLocation() {
            return s3ContentLocation != null ? s3ContentLocation.toBuilder() : null;
        }

        @Override
        public final Builder s3ContentLocation(S3ContentLocation s3ContentLocation) {
            this.s3ContentLocation = s3ContentLocation;
            return this;
        }

        public final void setS3ContentLocation(S3ContentLocation.BuilderImpl s3ContentLocation) {
            this.s3ContentLocation = s3ContentLocation != null ? s3ContentLocation.build() : null;
        }

        @Override
        public CodeContent build() {
            return new CodeContent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
