/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspacesstreams;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.keyspacesstreams.model.AccessDeniedException;
import software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.InternalServerException;
import software.amazon.awssdk.services.keyspacesstreams.model.KeyspacesStreamsException;
import software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.ResourceNotFoundException;
import software.amazon.awssdk.services.keyspacesstreams.model.ThrottlingException;
import software.amazon.awssdk.services.keyspacesstreams.model.ValidationException;
import software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable;
import software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable;

/**
 * Service client for accessing Amazon Keyspaces Streams. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Amazon Keyspaces (for Apache Cassandra) change data capture (CDC) records change events for Amazon Keyspaces tables.
 * The change events captured in a stream are time-ordered and de-duplicated write operations. Using stream data you can
 * build event driven applications that incorporate near-real time change events from Amazon Keyspaces tables.
 * </p>
 * <p>
 * Amazon Keyspaces CDC is serverless and scales the infrastructure for change events automatically based on the volume
 * of changes on your table.
 * </p>
 * <p>
 * This API reference describes the Amazon Keyspaces CDC stream API in detail.
 * </p>
 * <p>
 * For more information about Amazon Keyspaces CDC, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/cdc.html">Working with change data capture (CDC) streams
 * in Amazon Keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * To learn how Amazon Keyspaces CDC API actions are recorded with CloudTrail, see <a href=
 * "https://docs.aws.amazon.com/keyspaces/latest/devguide/logging-using-cloudtrail.html#service-name-info-in-cloudtrail"
 * >Amazon Keyspaces information in CloudTrail</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * To see the metrics Amazon Keyspaces CDC sends to Amazon CloudWatch, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/metrics-dimensions.html#keyspaces-cdc-metrics">Amazon
 * Keyspaces change data capture (CDC) CloudWatch metrics</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface KeyspacesStreamsClient extends AwsClient {
    String SERVICE_NAME = "cassandra";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "cassandra-streams";

    /**
     * <p>
     * Retrieves data records from a specified shard in an Amazon Keyspaces data stream. This operation returns a
     * collection of data records from the shard, including the primary key columns and information about modifications
     * made to the captured table data. Each record represents a single data modification in the Amazon Keyspaces table
     * and includes metadata about when the change occurred.
     * </p>
     *
     * @param getRecordsRequest
     * @return Result of the GetRecords operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    default GetRecordsResponse getRecords(GetRecordsRequest getRecordsRequest) throws ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            KeyspacesStreamsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves data records from a specified shard in an Amazon Keyspaces data stream. This operation returns a
     * collection of data records from the shard, including the primary key columns and information about modifications
     * made to the captured table data. Each record represents a single data modification in the Amazon Keyspaces table
     * and includes metadata about when the change occurred.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRecordsRequest.Builder} avoiding the need to
     * create one manually via {@link GetRecordsRequest#builder()}
     * </p>
     *
     * @param getRecordsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsRequest.Builder} to create a
     *        request.
     * @return Result of the GetRecords operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    default GetRecordsResponse getRecords(Consumer<GetRecordsRequest.Builder> getRecordsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        return getRecords(GetRecordsRequest.builder().applyMutation(getRecordsRequest).build());
    }

    /**
     * <p>
     * Returns a shard iterator that serves as a bookmark for reading data from a specific position in an Amazon
     * Keyspaces data stream's shard. The shard iterator specifies the shard position from which to start reading data
     * records sequentially. You can specify whether to begin reading at the latest record, the oldest record, or at a
     * particular sequence number within the shard.
     * </p>
     *
     * @param getShardIteratorRequest
     * @return Result of the GetShardIterator operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    default GetShardIteratorResponse getShardIterator(GetShardIteratorRequest getShardIteratorRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesStreamsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a shard iterator that serves as a bookmark for reading data from a specific position in an Amazon
     * Keyspaces data stream's shard. The shard iterator specifies the shard position from which to start reading data
     * records sequentially. You can specify whether to begin reading at the latest record, the oldest record, or at a
     * particular sequence number within the shard.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetShardIteratorRequest.Builder} avoiding the need
     * to create one manually via {@link GetShardIteratorRequest#builder()}
     * </p>
     *
     * @param getShardIteratorRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorRequest.Builder} to create a
     *        request.
     * @return Result of the GetShardIterator operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    default GetShardIteratorResponse getShardIterator(Consumer<GetShardIteratorRequest.Builder> getShardIteratorRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesStreamsException {
        return getShardIterator(GetShardIteratorRequest.builder().applyMutation(getShardIteratorRequest).build());
    }

    /**
     * <p>
     * Returns detailed information about a specific data capture stream for an Amazon Keyspaces table. The information
     * includes the stream's Amazon Resource Name (ARN), creation time, current status, retention period, shard
     * composition, and associated table details. This operation helps you monitor and manage the configuration of your
     * Amazon Keyspaces data streams.
     * </p>
     *
     * @param getStreamRequest
     * @return Result of the GetStream operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default GetStreamResponse getStream(GetStreamRequest getStreamRequest) throws ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            KeyspacesStreamsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns detailed information about a specific data capture stream for an Amazon Keyspaces table. The information
     * includes the stream's Amazon Resource Name (ARN), creation time, current status, retention period, shard
     * composition, and associated table details. This operation helps you monitor and manage the configuration of your
     * Amazon Keyspaces data streams.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetStreamRequest.Builder} avoiding the need to
     * create one manually via {@link GetStreamRequest#builder()}
     * </p>
     *
     * @param getStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest.Builder} to create a
     *        request.
     * @return Result of the GetStream operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default GetStreamResponse getStream(Consumer<GetStreamRequest.Builder> getStreamRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        return getStream(GetStreamRequest.builder().applyMutation(getStreamRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable responses = client.getStreamPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable responses = client.getStreamPaginator(request);
     *     for (software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable responses = client.getStreamPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)} operation.</b>
     * </p>
     *
     * @param getStreamRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default GetStreamIterable getStreamPaginator(GetStreamRequest getStreamRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        return new GetStreamIterable(this, getStreamRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable responses = client.getStreamPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable responses = client.getStreamPaginator(request);
     *     for (software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamIterable responses = client.getStreamPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetStreamRequest.Builder} avoiding the need to
     * create one manually via {@link GetStreamRequest#builder()}
     * </p>
     *
     * @param getStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default GetStreamIterable getStreamPaginator(Consumer<GetStreamRequest.Builder> getStreamRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        return getStreamPaginator(GetStreamRequest.builder().applyMutation(getStreamRequest).build());
    }

    /**
     * <p>
     * Returns a list of all data capture streams associated with your Amazon Keyspaces account or for a specific
     * keyspace or table. The response includes information such as stream ARNs, table associations, creation
     * timestamps, and current status. This operation helps you discover and manage all active data streams in your
     * Amazon Keyspaces environment.
     * </p>
     *
     * @param listStreamsRequest
     * @return Result of the ListStreams operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all data capture streams associated with your Amazon Keyspaces account or for a specific
     * keyspace or table. The response includes information such as stream ARNs, table associations, creation
     * timestamps, and current status. This operation helps you discover and manage all active data streams in your
     * Amazon Keyspaces environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamsRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamsRequest#builder()}
     * </p>
     *
     * @param listStreamsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest.Builder} to create a
     *        request.
     * @return Result of the ListStreams operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default ListStreamsResponse listStreams(Consumer<ListStreamsRequest.Builder> listStreamsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        return listStreams(ListStreamsRequest.builder().applyMutation(listStreamsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable responses = client
     *             .listStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default ListStreamsIterable listStreamsPaginator(ListStreamsRequest listStreamsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        return new ListStreamsIterable(this, listStreamsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable responses = client
     *             .listStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamsRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamsRequest#builder()}
     * </p>
     *
     * @param listStreamsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default ListStreamsIterable listStreamsPaginator(Consumer<ListStreamsRequest.Builder> listStreamsRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesStreamsException {
        return listStreamsPaginator(ListStreamsRequest.builder().applyMutation(listStreamsRequest).build());
    }

    /**
     * Create a {@link KeyspacesStreamsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static KeyspacesStreamsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link KeyspacesStreamsClient}.
     */
    static KeyspacesStreamsClientBuilder builder() {
        return new DefaultKeyspacesStreamsClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default KeyspacesStreamsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
