/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspacesstreams;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse;
import software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamPublisher;
import software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsPublisher;

/**
 * Service client for accessing Amazon Keyspaces Streams asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Amazon Keyspaces (for Apache Cassandra) change data capture (CDC) records change events for Amazon Keyspaces tables.
 * The change events captured in a stream are time-ordered and de-duplicated write operations. Using stream data you can
 * build event driven applications that incorporate near-real time change events from Amazon Keyspaces tables.
 * </p>
 * <p>
 * Amazon Keyspaces CDC is serverless and scales the infrastructure for change events automatically based on the volume
 * of changes on your table.
 * </p>
 * <p>
 * This API reference describes the Amazon Keyspaces CDC stream API in detail.
 * </p>
 * <p>
 * For more information about Amazon Keyspaces CDC, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/cdc.html">Working with change data capture (CDC) streams
 * in Amazon Keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * To learn how Amazon Keyspaces CDC API actions are recorded with CloudTrail, see <a href=
 * "https://docs.aws.amazon.com/keyspaces/latest/devguide/logging-using-cloudtrail.html#service-name-info-in-cloudtrail"
 * >Amazon Keyspaces information in CloudTrail</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * To see the metrics Amazon Keyspaces CDC sends to Amazon CloudWatch, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/metrics-dimensions.html#keyspaces-cdc-metrics">Amazon
 * Keyspaces change data capture (CDC) CloudWatch metrics</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface KeyspacesStreamsAsyncClient extends AwsClient {
    String SERVICE_NAME = "cassandra";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "cassandra-streams";

    /**
     * <p>
     * Retrieves data records from a specified shard in an Amazon Keyspaces data stream. This operation returns a
     * collection of data records from the shard, including the primary key columns and information about modifications
     * made to the captured table data. Each record represents a single data modification in the Amazon Keyspaces table
     * and includes metadata about when the change occurred.
     * </p>
     *
     * @param getRecordsRequest
     * @return A Java Future containing the result of the GetRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetRecordsResponse> getRecords(GetRecordsRequest getRecordsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves data records from a specified shard in an Amazon Keyspaces data stream. This operation returns a
     * collection of data records from the shard, including the primary key columns and information about modifications
     * made to the captured table data. Each record represents a single data modification in the Amazon Keyspaces table
     * and includes metadata about when the change occurred.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRecordsRequest.Builder} avoiding the need to
     * create one manually via {@link GetRecordsRequest#builder()}
     * </p>
     *
     * @param getRecordsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetRecordsResponse> getRecords(Consumer<GetRecordsRequest.Builder> getRecordsRequest) {
        return getRecords(GetRecordsRequest.builder().applyMutation(getRecordsRequest).build());
    }

    /**
     * <p>
     * Returns a shard iterator that serves as a bookmark for reading data from a specific position in an Amazon
     * Keyspaces data stream's shard. The shard iterator specifies the shard position from which to start reading data
     * records sequentially. You can specify whether to begin reading at the latest record, the oldest record, or at a
     * particular sequence number within the shard.
     * </p>
     *
     * @param getShardIteratorRequest
     * @return A Java Future containing the result of the GetShardIterator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetShardIteratorResponse> getShardIterator(GetShardIteratorRequest getShardIteratorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a shard iterator that serves as a bookmark for reading data from a specific position in an Amazon
     * Keyspaces data stream's shard. The shard iterator specifies the shard position from which to start reading data
     * records sequentially. You can specify whether to begin reading at the latest record, the oldest record, or at a
     * particular sequence number within the shard.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetShardIteratorRequest.Builder} avoiding the need
     * to create one manually via {@link GetShardIteratorRequest#builder()}
     * </p>
     *
     * @param getShardIteratorRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetShardIterator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetShardIteratorResponse> getShardIterator(
            Consumer<GetShardIteratorRequest.Builder> getShardIteratorRequest) {
        return getShardIterator(GetShardIteratorRequest.builder().applyMutation(getShardIteratorRequest).build());
    }

    /**
     * <p>
     * Returns detailed information about a specific data capture stream for an Amazon Keyspaces table. The information
     * includes the stream's Amazon Resource Name (ARN), creation time, current status, retention period, shard
     * composition, and associated table details. This operation helps you monitor and manage the configuration of your
     * Amazon Keyspaces data streams.
     * </p>
     *
     * @param getStreamRequest
     * @return A Java Future containing the result of the GetStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetStreamResponse> getStream(GetStreamRequest getStreamRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns detailed information about a specific data capture stream for an Amazon Keyspaces table. The information
     * includes the stream's Amazon Resource Name (ARN), creation time, current status, retention period, shard
     * composition, and associated table details. This operation helps you monitor and manage the configuration of your
     * Amazon Keyspaces data streams.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetStreamRequest.Builder} avoiding the need to
     * create one manually via {@link GetStreamRequest#builder()}
     * </p>
     *
     * @param getStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetStreamResponse> getStream(Consumer<GetStreamRequest.Builder> getStreamRequest) {
        return getStream(GetStreamRequest.builder().applyMutation(getStreamRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamPublisher publisher = client.getStreamPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamPublisher publisher = client.getStreamPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)} operation.</b>
     * </p>
     *
     * @param getStreamRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default GetStreamPublisher getStreamPaginator(GetStreamRequest getStreamRequest) {
        return new GetStreamPublisher(this, getStreamRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamPublisher publisher = client.getStreamPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.GetStreamPublisher publisher = client.getStreamPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getStream(software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetStreamRequest.Builder} avoiding the need to
     * create one manually via {@link GetStreamRequest#builder()}
     * </p>
     *
     * @param getStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    default GetStreamPublisher getStreamPaginator(Consumer<GetStreamRequest.Builder> getStreamRequest) {
        return getStreamPaginator(GetStreamRequest.builder().applyMutation(getStreamRequest).build());
    }

    /**
     * <p>
     * Returns a list of all data capture streams associated with your Amazon Keyspaces account or for a specific
     * keyspace or table. The response includes information such as stream ARNs, table associations, creation
     * timestamps, and current status. This operation helps you discover and manage all active data streams in your
     * Amazon Keyspaces environment.
     * </p>
     *
     * @param listStreamsRequest
     * @return A Java Future containing the result of the ListStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListStreamsResponse> listStreams(ListStreamsRequest listStreamsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all data capture streams associated with your Amazon Keyspaces account or for a specific
     * keyspace or table. The response includes information such as stream ARNs, table associations, creation
     * timestamps, and current status. This operation helps you discover and manage all active data streams in your
     * Amazon Keyspaces environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamsRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamsRequest#builder()}
     * </p>
     *
     * @param listStreamsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListStreamsResponse> listStreams(Consumer<ListStreamsRequest.Builder> listStreamsRequest) {
        return listStreams(ListStreamsRequest.builder().applyMutation(listStreamsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default ListStreamsPublisher listStreamsPaginator(ListStreamsRequest listStreamsRequest) {
        return new ListStreamsPublisher(this, listStreamsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspacesstreams.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamsRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamsRequest#builder()}
     * </p>
     *
     * @param listStreamsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this operation. </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.</li>
     *         <li>InternalServerException The Amazon Keyspaces service encountered an unexpected error while processing
     *         the request. </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.</li>
     *         <li>ValidationException The request validation failed because one or more input parameters failed
     *         validation. </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.</li>
     *         <li>ResourceNotFoundException The requested resource doesn't exist or could not be found. </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KeyspacesStreamsAsyncClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    default ListStreamsPublisher listStreamsPaginator(Consumer<ListStreamsRequest.Builder> listStreamsRequest) {
        return listStreamsPaginator(ListStreamsRequest.builder().applyMutation(listStreamsRequest).build());
    }

    @Override
    default KeyspacesStreamsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link KeyspacesStreamsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static KeyspacesStreamsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link KeyspacesStreamsAsyncClient}.
     */
    static KeyspacesStreamsAsyncClientBuilder builder() {
        return new DefaultKeyspacesStreamsAsyncClientBuilder();
    }
}
