/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The condition used for the target document attribute or metadata field when ingesting documents into Amazon Kendra.
 * You use this with <a
 * href="https://docs.aws.amazon.com/kendra/latest/dg/API_DocumentAttributeTarget.html">DocumentAttributeTarget to apply
 * the condition</a>.
 * </p>
 * <p>
 * For example, you can create the 'Department' target field and have it prefill department names associated with the
 * documents based on information in the 'Source_URI' field. Set the condition that if the 'Source_URI' field contains
 * 'financial' in its URI value, then prefill the target field 'Department' with the target value 'Finance' for the
 * document.
 * </p>
 * <p>
 * Amazon Kendra cannot create a target field if it has not already been created as an index field. After you create
 * your index field, you can create a document metadata field using <code>DocumentAttributeTarget</code>. Amazon Kendra
 * then will map your newly created metadata field to your index field.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentAttributeCondition implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentAttributeCondition.Builder, DocumentAttributeCondition> {
    private static final SdkField<String> CONDITION_DOCUMENT_ATTRIBUTE_KEY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ConditionDocumentAttributeKey")
            .getter(getter(DocumentAttributeCondition::conditionDocumentAttributeKey))
            .setter(setter(Builder::conditionDocumentAttributeKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConditionDocumentAttributeKey")
                    .build()).build();

    private static final SdkField<String> OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Operator").getter(getter(DocumentAttributeCondition::operatorAsString))
            .setter(setter(Builder::operator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operator").build()).build();

    private static final SdkField<DocumentAttributeValue> CONDITION_ON_VALUE_FIELD = SdkField
            .<DocumentAttributeValue> builder(MarshallingType.SDK_POJO).memberName("ConditionOnValue")
            .getter(getter(DocumentAttributeCondition::conditionOnValue)).setter(setter(Builder::conditionOnValue))
            .constructor(DocumentAttributeValue::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConditionOnValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CONDITION_DOCUMENT_ATTRIBUTE_KEY_FIELD, OPERATOR_FIELD, CONDITION_ON_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String conditionDocumentAttributeKey;

    private final String operator;

    private final DocumentAttributeValue conditionOnValue;

    private DocumentAttributeCondition(BuilderImpl builder) {
        this.conditionDocumentAttributeKey = builder.conditionDocumentAttributeKey;
        this.operator = builder.operator;
        this.conditionOnValue = builder.conditionOnValue;
    }

    /**
     * <p>
     * The identifier of the document attribute used for the condition.
     * </p>
     * <p>
     * For example, 'Source_URI' could be an identifier for the attribute or metadata field that contains source URIs
     * associated with the documents.
     * </p>
     * <p>
     * Amazon Kendra currently does not support <code>_document_body</code> as an attribute key used for the condition.
     * </p>
     * 
     * @return The identifier of the document attribute used for the condition.</p>
     *         <p>
     *         For example, 'Source_URI' could be an identifier for the attribute or metadata field that contains source
     *         URIs associated with the documents.
     *         </p>
     *         <p>
     *         Amazon Kendra currently does not support <code>_document_body</code> as an attribute key used for the
     *         condition.
     */
    public final String conditionDocumentAttributeKey() {
        return conditionDocumentAttributeKey;
    }

    /**
     * <p>
     * The condition operator.
     * </p>
     * <p>
     * For example, you can use 'Contains' to partially match a string.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link ConditionOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return The condition operator.</p>
     *         <p>
     *         For example, you can use 'Contains' to partially match a string.
     * @see ConditionOperator
     */
    public final ConditionOperator operator() {
        return ConditionOperator.fromValue(operator);
    }

    /**
     * <p>
     * The condition operator.
     * </p>
     * <p>
     * For example, you can use 'Contains' to partially match a string.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link ConditionOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return The condition operator.</p>
     *         <p>
     *         For example, you can use 'Contains' to partially match a string.
     * @see ConditionOperator
     */
    public final String operatorAsString() {
        return operator;
    }

    /**
     * <p>
     * The value used by the operator.
     * </p>
     * <p>
     * For example, you can specify the value 'financial' for strings in the 'Source_URI' field that partially match or
     * contain this value.
     * </p>
     * 
     * @return The value used by the operator.</p>
     *         <p>
     *         For example, you can specify the value 'financial' for strings in the 'Source_URI' field that partially
     *         match or contain this value.
     */
    public final DocumentAttributeValue conditionOnValue() {
        return conditionOnValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(conditionDocumentAttributeKey());
        hashCode = 31 * hashCode + Objects.hashCode(operatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(conditionOnValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentAttributeCondition)) {
            return false;
        }
        DocumentAttributeCondition other = (DocumentAttributeCondition) obj;
        return Objects.equals(conditionDocumentAttributeKey(), other.conditionDocumentAttributeKey())
                && Objects.equals(operatorAsString(), other.operatorAsString())
                && Objects.equals(conditionOnValue(), other.conditionOnValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentAttributeCondition")
                .add("ConditionDocumentAttributeKey", conditionDocumentAttributeKey()).add("Operator", operatorAsString())
                .add("ConditionOnValue", conditionOnValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConditionDocumentAttributeKey":
            return Optional.ofNullable(clazz.cast(conditionDocumentAttributeKey()));
        case "Operator":
            return Optional.ofNullable(clazz.cast(operatorAsString()));
        case "ConditionOnValue":
            return Optional.ofNullable(clazz.cast(conditionOnValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentAttributeCondition, T> g) {
        return obj -> g.apply((DocumentAttributeCondition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentAttributeCondition> {
        /**
         * <p>
         * The identifier of the document attribute used for the condition.
         * </p>
         * <p>
         * For example, 'Source_URI' could be an identifier for the attribute or metadata field that contains source
         * URIs associated with the documents.
         * </p>
         * <p>
         * Amazon Kendra currently does not support <code>_document_body</code> as an attribute key used for the
         * condition.
         * </p>
         * 
         * @param conditionDocumentAttributeKey
         *        The identifier of the document attribute used for the condition.</p>
         *        <p>
         *        For example, 'Source_URI' could be an identifier for the attribute or metadata field that contains
         *        source URIs associated with the documents.
         *        </p>
         *        <p>
         *        Amazon Kendra currently does not support <code>_document_body</code> as an attribute key used for the
         *        condition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditionDocumentAttributeKey(String conditionDocumentAttributeKey);

        /**
         * <p>
         * The condition operator.
         * </p>
         * <p>
         * For example, you can use 'Contains' to partially match a string.
         * </p>
         * 
         * @param operator
         *        The condition operator.</p>
         *        <p>
         *        For example, you can use 'Contains' to partially match a string.
         * @see ConditionOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConditionOperator
         */
        Builder operator(String operator);

        /**
         * <p>
         * The condition operator.
         * </p>
         * <p>
         * For example, you can use 'Contains' to partially match a string.
         * </p>
         * 
         * @param operator
         *        The condition operator.</p>
         *        <p>
         *        For example, you can use 'Contains' to partially match a string.
         * @see ConditionOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConditionOperator
         */
        Builder operator(ConditionOperator operator);

        /**
         * <p>
         * The value used by the operator.
         * </p>
         * <p>
         * For example, you can specify the value 'financial' for strings in the 'Source_URI' field that partially match
         * or contain this value.
         * </p>
         * 
         * @param conditionOnValue
         *        The value used by the operator.</p>
         *        <p>
         *        For example, you can specify the value 'financial' for strings in the 'Source_URI' field that
         *        partially match or contain this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditionOnValue(DocumentAttributeValue conditionOnValue);

        /**
         * <p>
         * The value used by the operator.
         * </p>
         * <p>
         * For example, you can specify the value 'financial' for strings in the 'Source_URI' field that partially match
         * or contain this value.
         * </p>
         * This is a convenience method that creates an instance of the {@link DocumentAttributeValue.Builder} avoiding
         * the need to create one manually via {@link DocumentAttributeValue#builder()}.
         *
         * When the {@link Consumer} completes, {@link DocumentAttributeValue.Builder#build()} is called immediately and
         * its result is passed to {@link #conditionOnValue(DocumentAttributeValue)}.
         * 
         * @param conditionOnValue
         *        a consumer that will call methods on {@link DocumentAttributeValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #conditionOnValue(DocumentAttributeValue)
         */
        default Builder conditionOnValue(Consumer<DocumentAttributeValue.Builder> conditionOnValue) {
            return conditionOnValue(DocumentAttributeValue.builder().applyMutation(conditionOnValue).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String conditionDocumentAttributeKey;

        private String operator;

        private DocumentAttributeValue conditionOnValue;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentAttributeCondition model) {
            conditionDocumentAttributeKey(model.conditionDocumentAttributeKey);
            operator(model.operator);
            conditionOnValue(model.conditionOnValue);
        }

        public final String getConditionDocumentAttributeKey() {
            return conditionDocumentAttributeKey;
        }

        public final void setConditionDocumentAttributeKey(String conditionDocumentAttributeKey) {
            this.conditionDocumentAttributeKey = conditionDocumentAttributeKey;
        }

        @Override
        public final Builder conditionDocumentAttributeKey(String conditionDocumentAttributeKey) {
            this.conditionDocumentAttributeKey = conditionDocumentAttributeKey;
            return this;
        }

        public final String getOperator() {
            return operator;
        }

        public final void setOperator(String operator) {
            this.operator = operator;
        }

        @Override
        public final Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        @Override
        public final Builder operator(ConditionOperator operator) {
            this.operator(operator == null ? null : operator.toString());
            return this;
        }

        public final DocumentAttributeValue.Builder getConditionOnValue() {
            return conditionOnValue != null ? conditionOnValue.toBuilder() : null;
        }

        public final void setConditionOnValue(DocumentAttributeValue.BuilderImpl conditionOnValue) {
            this.conditionOnValue = conditionOnValue != null ? conditionOnValue.build() : null;
        }

        @Override
        public final Builder conditionOnValue(DocumentAttributeValue conditionOnValue) {
            this.conditionOnValue = conditionOnValue;
            return this;
        }

        @Override
        public DocumentAttributeCondition build() {
            return new DocumentAttributeCondition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
