/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iottwinmaker.endpoints.internal;

import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.iottwinmaker.endpoints.IoTTwinMakerEndpointParams;
import software.amazon.awssdk.services.iottwinmaker.endpoints.IoTTwinMakerEndpointProvider;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class IoTTwinMakerResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return context.request();
        }
        IoTTwinMakerEndpointProvider provider = (IoTTwinMakerEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            Endpoint result = provider.resolveEndpoint(ruleParams(context, executionAttributes)).join();
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        context.request());
                if (hostPrefix.isPresent()) {
                    result = AwsEndpointProviderUtils.addHostPrefix(result, hostPrefix.get());
                }
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, result);
            return context.request();
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    private static IoTTwinMakerEndpointParams ruleParams(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        IoTTwinMakerEndpointParams.Builder builder = IoTTwinMakerEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), context.request());
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        return builder.build();
    }

    private static void setContextParams(IoTTwinMakerEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static void setStaticContextParams(IoTTwinMakerEndpointParams.Builder params, String operationName) {
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        switch (operationName) {
        case "BatchPutPropertyValues": {
            return Optional.of("data.");
        }
        case "CreateComponentType": {
            return Optional.of("api.");
        }
        case "CreateEntity": {
            return Optional.of("api.");
        }
        case "CreateScene": {
            return Optional.of("api.");
        }
        case "CreateSyncJob": {
            return Optional.of("api.");
        }
        case "CreateWorkspace": {
            return Optional.of("api.");
        }
        case "DeleteComponentType": {
            return Optional.of("api.");
        }
        case "DeleteEntity": {
            return Optional.of("api.");
        }
        case "DeleteScene": {
            return Optional.of("api.");
        }
        case "DeleteSyncJob": {
            return Optional.of("api.");
        }
        case "DeleteWorkspace": {
            return Optional.of("api.");
        }
        case "ExecuteQuery": {
            return Optional.of("api.");
        }
        case "GetComponentType": {
            return Optional.of("api.");
        }
        case "GetEntity": {
            return Optional.of("api.");
        }
        case "GetPricingPlan": {
            return Optional.of("api.");
        }
        case "GetPropertyValue": {
            return Optional.of("data.");
        }
        case "GetPropertyValueHistory": {
            return Optional.of("data.");
        }
        case "GetScene": {
            return Optional.of("api.");
        }
        case "GetSyncJob": {
            return Optional.of("api.");
        }
        case "GetWorkspace": {
            return Optional.of("api.");
        }
        case "ListComponentTypes": {
            return Optional.of("api.");
        }
        case "ListEntities": {
            return Optional.of("api.");
        }
        case "ListScenes": {
            return Optional.of("api.");
        }
        case "ListSyncJobs": {
            return Optional.of("api.");
        }
        case "ListSyncResources": {
            return Optional.of("api.");
        }
        case "ListTagsForResource": {
            return Optional.of("api.");
        }
        case "ListWorkspaces": {
            return Optional.of("api.");
        }
        case "TagResource": {
            return Optional.of("api.");
        }
        case "UntagResource": {
            return Optional.of("api.");
        }
        case "UpdateComponentType": {
            return Optional.of("api.");
        }
        case "UpdateEntity": {
            return Optional.of("api.");
        }
        case "UpdatePricingPlan": {
            return Optional.of("api.");
        }
        case "UpdateScene": {
            return Optional.of("api.");
        }
        case "UpdateWorkspace": {
            return Optional.of("api.");
        }
        default:
            return Optional.empty();
        }
    }
}
