/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotanalytics.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a dataset.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Dataset implements SdkPojo, Serializable, ToCopyableBuilder<Dataset.Builder, Dataset> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Dataset::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Dataset::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<List<DatasetAction>> ACTIONS_FIELD = SdkField
            .<List<DatasetAction>> builder(MarshallingType.LIST)
            .memberName("actions")
            .getter(getter(Dataset::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DatasetAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatasetAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DatasetTrigger>> TRIGGERS_FIELD = SdkField
            .<List<DatasetTrigger>> builder(MarshallingType.LIST)
            .memberName("triggers")
            .getter(getter(Dataset::triggers))
            .setter(setter(Builder::triggers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("triggers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DatasetTrigger> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatasetTrigger::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DatasetContentDeliveryRule>> CONTENT_DELIVERY_RULES_FIELD = SdkField
            .<List<DatasetContentDeliveryRule>> builder(MarshallingType.LIST)
            .memberName("contentDeliveryRules")
            .getter(getter(Dataset::contentDeliveryRules))
            .setter(setter(Builder::contentDeliveryRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("contentDeliveryRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DatasetContentDeliveryRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatasetContentDeliveryRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Dataset::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(Dataset::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdateTime").getter(getter(Dataset::lastUpdateTime)).setter(setter(Builder::lastUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdateTime").build()).build();

    private static final SdkField<RetentionPeriod> RETENTION_PERIOD_FIELD = SdkField
            .<RetentionPeriod> builder(MarshallingType.SDK_POJO).memberName("retentionPeriod")
            .getter(getter(Dataset::retentionPeriod)).setter(setter(Builder::retentionPeriod))
            .constructor(RetentionPeriod::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retentionPeriod").build()).build();

    private static final SdkField<VersioningConfiguration> VERSIONING_CONFIGURATION_FIELD = SdkField
            .<VersioningConfiguration> builder(MarshallingType.SDK_POJO).memberName("versioningConfiguration")
            .getter(getter(Dataset::versioningConfiguration)).setter(setter(Builder::versioningConfiguration))
            .constructor(VersioningConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versioningConfiguration").build())
            .build();

    private static final SdkField<List<LateDataRule>> LATE_DATA_RULES_FIELD = SdkField
            .<List<LateDataRule>> builder(MarshallingType.LIST)
            .memberName("lateDataRules")
            .getter(getter(Dataset::lateDataRules))
            .setter(setter(Builder::lateDataRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lateDataRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LateDataRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(LateDataRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            ACTIONS_FIELD, TRIGGERS_FIELD, CONTENT_DELIVERY_RULES_FIELD, STATUS_FIELD, CREATION_TIME_FIELD,
            LAST_UPDATE_TIME_FIELD, RETENTION_PERIOD_FIELD, VERSIONING_CONFIGURATION_FIELD, LATE_DATA_RULES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final List<DatasetAction> actions;

    private final List<DatasetTrigger> triggers;

    private final List<DatasetContentDeliveryRule> contentDeliveryRules;

    private final String status;

    private final Instant creationTime;

    private final Instant lastUpdateTime;

    private final RetentionPeriod retentionPeriod;

    private final VersioningConfiguration versioningConfiguration;

    private final List<LateDataRule> lateDataRules;

    private Dataset(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.actions = builder.actions;
        this.triggers = builder.triggers;
        this.contentDeliveryRules = builder.contentDeliveryRules;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.lastUpdateTime = builder.lastUpdateTime;
        this.retentionPeriod = builder.retentionPeriod;
        this.versioningConfiguration = builder.versioningConfiguration;
        this.lateDataRules = builder.lateDataRules;
    }

    /**
     * <p>
     * The name of the dataset.
     * </p>
     * 
     * @return The name of the dataset.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the dataset.
     * </p>
     * 
     * @return The ARN of the dataset.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>DatasetAction</code> objects that automatically create the dataset contents.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return The <code>DatasetAction</code> objects that automatically create the dataset contents.
     */
    public final List<DatasetAction> actions() {
        return actions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Triggers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTriggers() {
        return triggers != null && !(triggers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTriggers} method.
     * </p>
     * 
     * @return The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
     */
    public final List<DatasetTrigger> triggers() {
        return triggers;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContentDeliveryRules property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContentDeliveryRules() {
        return contentDeliveryRules != null && !(contentDeliveryRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * When dataset contents are created they are delivered to destinations specified here.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContentDeliveryRules} method.
     * </p>
     * 
     * @return When dataset contents are created they are delivered to destinations specified here.
     */
    public final List<DatasetContentDeliveryRule> contentDeliveryRules() {
        return contentDeliveryRules;
    }

    /**
     * <p>
     * The status of the dataset.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DatasetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the dataset.
     * @see DatasetStatus
     */
    public final DatasetStatus status() {
        return DatasetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the dataset.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DatasetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the dataset.
     * @see DatasetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * When the dataset was created.
     * </p>
     * 
     * @return When the dataset was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The last time the dataset was updated.
     * </p>
     * 
     * @return The last time the dataset was updated.
     */
    public final Instant lastUpdateTime() {
        return lastUpdateTime;
    }

    /**
     * <p>
     * Optional. How long, in days, message data is kept for the dataset.
     * </p>
     * 
     * @return Optional. How long, in days, message data is kept for the dataset.
     */
    public final RetentionPeriod retentionPeriod() {
        return retentionPeriod;
    }

    /**
     * <p>
     * Optional. How many versions of dataset contents are kept. If not specified or set to null, only the latest
     * version plus the latest succeeded version (if they are different) are kept for the time period specified by the
     * <code>retentionPeriod</code> parameter. For more information, see <a href=
     * "https://docs.aws.amazon.com/iotanalytics/latest/userguide/getting-started.html#aws-iot-analytics-dataset-versions"
     * > Keeping Multiple Versions of IoT Analytics datasets</a> in the <i>IoT Analytics User Guide</i>.
     * </p>
     * 
     * @return Optional. How many versions of dataset contents are kept. If not specified or set to null, only the
     *         latest version plus the latest succeeded version (if they are different) are kept for the time period
     *         specified by the <code>retentionPeriod</code> parameter. For more information, see <a href=
     *         "https://docs.aws.amazon.com/iotanalytics/latest/userguide/getting-started.html#aws-iot-analytics-dataset-versions"
     *         > Keeping Multiple Versions of IoT Analytics datasets</a> in the <i>IoT Analytics User Guide</i>.
     */
    public final VersioningConfiguration versioningConfiguration() {
        return versioningConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the LateDataRules property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLateDataRules() {
        return lateDataRules != null && !(lateDataRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
     * <code>lateDataRules</code>, the dataset must use a <a
     * href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a> filter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLateDataRules} method.
     * </p>
     * 
     * @return A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
     *         <code>lateDataRules</code>, the dataset must use a <a
     *         href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a>
     *         filter.
     */
    public final List<LateDataRule> lateDataRules() {
        return lateDataRules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTriggers() ? triggers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasContentDeliveryRules() ? contentDeliveryRules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTime());
        hashCode = 31 * hashCode + Objects.hashCode(retentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(versioningConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasLateDataRules() ? lateDataRules() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Dataset)) {
            return false;
        }
        Dataset other = (Dataset) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn()) && hasActions() == other.hasActions()
                && Objects.equals(actions(), other.actions()) && hasTriggers() == other.hasTriggers()
                && Objects.equals(triggers(), other.triggers()) && hasContentDeliveryRules() == other.hasContentDeliveryRules()
                && Objects.equals(contentDeliveryRules(), other.contentDeliveryRules())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdateTime(), other.lastUpdateTime())
                && Objects.equals(retentionPeriod(), other.retentionPeriod())
                && Objects.equals(versioningConfiguration(), other.versioningConfiguration())
                && hasLateDataRules() == other.hasLateDataRules() && Objects.equals(lateDataRules(), other.lateDataRules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Dataset").add("Name", name()).add("Arn", arn()).add("Actions", hasActions() ? actions() : null)
                .add("Triggers", hasTriggers() ? triggers() : null)
                .add("ContentDeliveryRules", hasContentDeliveryRules() ? contentDeliveryRules() : null)
                .add("Status", statusAsString()).add("CreationTime", creationTime()).add("LastUpdateTime", lastUpdateTime())
                .add("RetentionPeriod", retentionPeriod()).add("VersioningConfiguration", versioningConfiguration())
                .add("LateDataRules", hasLateDataRules() ? lateDataRules() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "actions":
            return Optional.ofNullable(clazz.cast(actions()));
        case "triggers":
            return Optional.ofNullable(clazz.cast(triggers()));
        case "contentDeliveryRules":
            return Optional.ofNullable(clazz.cast(contentDeliveryRules()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateTime()));
        case "retentionPeriod":
            return Optional.ofNullable(clazz.cast(retentionPeriod()));
        case "versioningConfiguration":
            return Optional.ofNullable(clazz.cast(versioningConfiguration()));
        case "lateDataRules":
            return Optional.ofNullable(clazz.cast(lateDataRules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("actions", ACTIONS_FIELD);
        map.put("triggers", TRIGGERS_FIELD);
        map.put("contentDeliveryRules", CONTENT_DELIVERY_RULES_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("creationTime", CREATION_TIME_FIELD);
        map.put("lastUpdateTime", LAST_UPDATE_TIME_FIELD);
        map.put("retentionPeriod", RETENTION_PERIOD_FIELD);
        map.put("versioningConfiguration", VERSIONING_CONFIGURATION_FIELD);
        map.put("lateDataRules", LATE_DATA_RULES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Dataset, T> g) {
        return obj -> g.apply((Dataset) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Dataset> {
        /**
         * <p>
         * The name of the dataset.
         * </p>
         * 
         * @param name
         *        The name of the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the dataset.
         * </p>
         * 
         * @param arn
         *        The ARN of the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The <code>DatasetAction</code> objects that automatically create the dataset contents.
         * </p>
         * 
         * @param actions
         *        The <code>DatasetAction</code> objects that automatically create the dataset contents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<DatasetAction> actions);

        /**
         * <p>
         * The <code>DatasetAction</code> objects that automatically create the dataset contents.
         * </p>
         * 
         * @param actions
         *        The <code>DatasetAction</code> objects that automatically create the dataset contents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(DatasetAction... actions);

        /**
         * <p>
         * The <code>DatasetAction</code> objects that automatically create the dataset contents.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetAction.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.iotanalytics.model.DatasetAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetAction.Builder#build()} is called
         * immediately and its result is passed to {@link #actions(List<DatasetAction>)}.
         * 
         * @param actions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iotanalytics.model.DatasetAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(java.util.Collection<DatasetAction>)
         */
        Builder actions(Consumer<DatasetAction.Builder>... actions);

        /**
         * <p>
         * The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
         * </p>
         * 
         * @param triggers
         *        The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggers(Collection<DatasetTrigger> triggers);

        /**
         * <p>
         * The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
         * </p>
         * 
         * @param triggers
         *        The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggers(DatasetTrigger... triggers);

        /**
         * <p>
         * The <code>DatasetTrigger</code> objects that specify when the dataset is automatically updated.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetTrigger.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.iotanalytics.model.DatasetTrigger#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetTrigger.Builder#build()} is called
         * immediately and its result is passed to {@link #triggers(List<DatasetTrigger>)}.
         * 
         * @param triggers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iotanalytics.model.DatasetTrigger.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #triggers(java.util.Collection<DatasetTrigger>)
         */
        Builder triggers(Consumer<DatasetTrigger.Builder>... triggers);

        /**
         * <p>
         * When dataset contents are created they are delivered to destinations specified here.
         * </p>
         * 
         * @param contentDeliveryRules
         *        When dataset contents are created they are delivered to destinations specified here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentDeliveryRules(Collection<DatasetContentDeliveryRule> contentDeliveryRules);

        /**
         * <p>
         * When dataset contents are created they are delivered to destinations specified here.
         * </p>
         * 
         * @param contentDeliveryRules
         *        When dataset contents are created they are delivered to destinations specified here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentDeliveryRules(DatasetContentDeliveryRule... contentDeliveryRules);

        /**
         * <p>
         * When dataset contents are created they are delivered to destinations specified here.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetContentDeliveryRule.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetContentDeliveryRule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iotanalytics.model.DatasetContentDeliveryRule.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #contentDeliveryRules(List<DatasetContentDeliveryRule>)}.
         * 
         * @param contentDeliveryRules
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iotanalytics.model.DatasetContentDeliveryRule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentDeliveryRules(java.util.Collection<DatasetContentDeliveryRule>)
         */
        Builder contentDeliveryRules(Consumer<DatasetContentDeliveryRule.Builder>... contentDeliveryRules);

        /**
         * <p>
         * The status of the dataset.
         * </p>
         * 
         * @param status
         *        The status of the dataset.
         * @see DatasetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatasetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the dataset.
         * </p>
         * 
         * @param status
         *        The status of the dataset.
         * @see DatasetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatasetStatus
         */
        Builder status(DatasetStatus status);

        /**
         * <p>
         * When the dataset was created.
         * </p>
         * 
         * @param creationTime
         *        When the dataset was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The last time the dataset was updated.
         * </p>
         * 
         * @param lastUpdateTime
         *        The last time the dataset was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTime(Instant lastUpdateTime);

        /**
         * <p>
         * Optional. How long, in days, message data is kept for the dataset.
         * </p>
         * 
         * @param retentionPeriod
         *        Optional. How long, in days, message data is kept for the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retentionPeriod(RetentionPeriod retentionPeriod);

        /**
         * <p>
         * Optional. How long, in days, message data is kept for the dataset.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetentionPeriod.Builder} avoiding the
         * need to create one manually via {@link RetentionPeriod#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetentionPeriod.Builder#build()} is called immediately and its
         * result is passed to {@link #retentionPeriod(RetentionPeriod)}.
         * 
         * @param retentionPeriod
         *        a consumer that will call methods on {@link RetentionPeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retentionPeriod(RetentionPeriod)
         */
        default Builder retentionPeriod(Consumer<RetentionPeriod.Builder> retentionPeriod) {
            return retentionPeriod(RetentionPeriod.builder().applyMutation(retentionPeriod).build());
        }

        /**
         * <p>
         * Optional. How many versions of dataset contents are kept. If not specified or set to null, only the latest
         * version plus the latest succeeded version (if they are different) are kept for the time period specified by
         * the <code>retentionPeriod</code> parameter. For more information, see <a href=
         * "https://docs.aws.amazon.com/iotanalytics/latest/userguide/getting-started.html#aws-iot-analytics-dataset-versions"
         * > Keeping Multiple Versions of IoT Analytics datasets</a> in the <i>IoT Analytics User Guide</i>.
         * </p>
         * 
         * @param versioningConfiguration
         *        Optional. How many versions of dataset contents are kept. If not specified or set to null, only the
         *        latest version plus the latest succeeded version (if they are different) are kept for the time period
         *        specified by the <code>retentionPeriod</code> parameter. For more information, see <a href=
         *        "https://docs.aws.amazon.com/iotanalytics/latest/userguide/getting-started.html#aws-iot-analytics-dataset-versions"
         *        > Keeping Multiple Versions of IoT Analytics datasets</a> in the <i>IoT Analytics User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versioningConfiguration(VersioningConfiguration versioningConfiguration);

        /**
         * <p>
         * Optional. How many versions of dataset contents are kept. If not specified or set to null, only the latest
         * version plus the latest succeeded version (if they are different) are kept for the time period specified by
         * the <code>retentionPeriod</code> parameter. For more information, see <a href=
         * "https://docs.aws.amazon.com/iotanalytics/latest/userguide/getting-started.html#aws-iot-analytics-dataset-versions"
         * > Keeping Multiple Versions of IoT Analytics datasets</a> in the <i>IoT Analytics User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link VersioningConfiguration.Builder} avoiding
         * the need to create one manually via {@link VersioningConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VersioningConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #versioningConfiguration(VersioningConfiguration)}.
         * 
         * @param versioningConfiguration
         *        a consumer that will call methods on {@link VersioningConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #versioningConfiguration(VersioningConfiguration)
         */
        default Builder versioningConfiguration(Consumer<VersioningConfiguration.Builder> versioningConfiguration) {
            return versioningConfiguration(VersioningConfiguration.builder().applyMutation(versioningConfiguration).build());
        }

        /**
         * <p>
         * A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
         * <code>lateDataRules</code>, the dataset must use a <a
         * href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a> filter.
         * </p>
         * 
         * @param lateDataRules
         *        A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
         *        <code>lateDataRules</code>, the dataset must use a <a
         *        href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a>
         *        filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lateDataRules(Collection<LateDataRule> lateDataRules);

        /**
         * <p>
         * A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
         * <code>lateDataRules</code>, the dataset must use a <a
         * href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a> filter.
         * </p>
         * 
         * @param lateDataRules
         *        A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
         *        <code>lateDataRules</code>, the dataset must use a <a
         *        href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a>
         *        filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lateDataRules(LateDataRule... lateDataRules);

        /**
         * <p>
         * A list of data rules that send notifications to CloudWatch, when data arrives late. To specify
         * <code>lateDataRules</code>, the dataset must use a <a
         * href="https://docs.aws.amazon.com/iotanalytics/latest/APIReference/API_DeltaTime.html">DeltaTimer</a> filter.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iotanalytics.model.LateDataRule.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.iotanalytics.model.LateDataRule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iotanalytics.model.LateDataRule.Builder#build()} is called immediately
         * and its result is passed to {@link #lateDataRules(List<LateDataRule>)}.
         * 
         * @param lateDataRules
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iotanalytics.model.LateDataRule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lateDataRules(java.util.Collection<LateDataRule>)
         */
        Builder lateDataRules(Consumer<LateDataRule.Builder>... lateDataRules);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private List<DatasetAction> actions = DefaultSdkAutoConstructList.getInstance();

        private List<DatasetTrigger> triggers = DefaultSdkAutoConstructList.getInstance();

        private List<DatasetContentDeliveryRule> contentDeliveryRules = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private Instant creationTime;

        private Instant lastUpdateTime;

        private RetentionPeriod retentionPeriod;

        private VersioningConfiguration versioningConfiguration;

        private List<LateDataRule> lateDataRules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Dataset model) {
            name(model.name);
            arn(model.arn);
            actions(model.actions);
            triggers(model.triggers);
            contentDeliveryRules(model.contentDeliveryRules);
            status(model.status);
            creationTime(model.creationTime);
            lastUpdateTime(model.lastUpdateTime);
            retentionPeriod(model.retentionPeriod);
            versioningConfiguration(model.versioningConfiguration);
            lateDataRules(model.lateDataRules);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final List<DatasetAction.Builder> getActions() {
            List<DatasetAction.Builder> result = DatasetActionsCopier.copyToBuilder(this.actions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActions(Collection<DatasetAction.BuilderImpl> actions) {
            this.actions = DatasetActionsCopier.copyFromBuilder(actions);
        }

        @Override
        public final Builder actions(Collection<DatasetAction> actions) {
            this.actions = DatasetActionsCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(DatasetAction... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<DatasetAction.Builder>... actions) {
            actions(Stream.of(actions).map(c -> DatasetAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DatasetTrigger.Builder> getTriggers() {
            List<DatasetTrigger.Builder> result = DatasetTriggersCopier.copyToBuilder(this.triggers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTriggers(Collection<DatasetTrigger.BuilderImpl> triggers) {
            this.triggers = DatasetTriggersCopier.copyFromBuilder(triggers);
        }

        @Override
        public final Builder triggers(Collection<DatasetTrigger> triggers) {
            this.triggers = DatasetTriggersCopier.copy(triggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggers(DatasetTrigger... triggers) {
            triggers(Arrays.asList(triggers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggers(Consumer<DatasetTrigger.Builder>... triggers) {
            triggers(Stream.of(triggers).map(c -> DatasetTrigger.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DatasetContentDeliveryRule.Builder> getContentDeliveryRules() {
            List<DatasetContentDeliveryRule.Builder> result = DatasetContentDeliveryRulesCopier
                    .copyToBuilder(this.contentDeliveryRules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setContentDeliveryRules(Collection<DatasetContentDeliveryRule.BuilderImpl> contentDeliveryRules) {
            this.contentDeliveryRules = DatasetContentDeliveryRulesCopier.copyFromBuilder(contentDeliveryRules);
        }

        @Override
        public final Builder contentDeliveryRules(Collection<DatasetContentDeliveryRule> contentDeliveryRules) {
            this.contentDeliveryRules = DatasetContentDeliveryRulesCopier.copy(contentDeliveryRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder contentDeliveryRules(DatasetContentDeliveryRule... contentDeliveryRules) {
            contentDeliveryRules(Arrays.asList(contentDeliveryRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder contentDeliveryRules(Consumer<DatasetContentDeliveryRule.Builder>... contentDeliveryRules) {
            contentDeliveryRules(Stream.of(contentDeliveryRules)
                    .map(c -> DatasetContentDeliveryRule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DatasetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdateTime() {
            return lastUpdateTime;
        }

        public final void setLastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
        }

        @Override
        public final Builder lastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
            return this;
        }

        public final RetentionPeriod.Builder getRetentionPeriod() {
            return retentionPeriod != null ? retentionPeriod.toBuilder() : null;
        }

        public final void setRetentionPeriod(RetentionPeriod.BuilderImpl retentionPeriod) {
            this.retentionPeriod = retentionPeriod != null ? retentionPeriod.build() : null;
        }

        @Override
        public final Builder retentionPeriod(RetentionPeriod retentionPeriod) {
            this.retentionPeriod = retentionPeriod;
            return this;
        }

        public final VersioningConfiguration.Builder getVersioningConfiguration() {
            return versioningConfiguration != null ? versioningConfiguration.toBuilder() : null;
        }

        public final void setVersioningConfiguration(VersioningConfiguration.BuilderImpl versioningConfiguration) {
            this.versioningConfiguration = versioningConfiguration != null ? versioningConfiguration.build() : null;
        }

        @Override
        public final Builder versioningConfiguration(VersioningConfiguration versioningConfiguration) {
            this.versioningConfiguration = versioningConfiguration;
            return this;
        }

        public final List<LateDataRule.Builder> getLateDataRules() {
            List<LateDataRule.Builder> result = LateDataRulesCopier.copyToBuilder(this.lateDataRules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLateDataRules(Collection<LateDataRule.BuilderImpl> lateDataRules) {
            this.lateDataRules = LateDataRulesCopier.copyFromBuilder(lateDataRules);
        }

        @Override
        public final Builder lateDataRules(Collection<LateDataRule> lateDataRules) {
            this.lateDataRules = LateDataRulesCopier.copy(lateDataRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lateDataRules(LateDataRule... lateDataRules) {
            lateDataRules(Arrays.asList(lateDataRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lateDataRules(Consumer<LateDataRule.Builder>... lateDataRules) {
            lateDataRules(Stream.of(lateDataRules).map(c -> LateDataRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Dataset build() {
            return new Dataset(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
