/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.internetmonitor.model.CreateMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.CreateMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.DeleteMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.DeleteMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetHealthEventRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetHealthEventResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.TagResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.TagResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.UntagResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.UntagResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.UpdateMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.UpdateMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsPublisher;
import software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsPublisher;

/**
 * Service client for accessing Amazon CloudWatch Internet Monitor asynchronously. This can be created using the static
 * {@link #builder()} method.
 *
 * <p>
 * Amazon CloudWatch Internet Monitor provides visibility into how internet issues impact the performance and
 * availability between your applications hosted on Amazon Web Services and your end users. It reduces the time it takes
 * for you to diagnose internet issues from days to minutes. Internet Monitor uses the connectivity data that Amazon Web
 * Services captures from its global networking footprint to calculate a baseline of performance and availability for
 * internet traffic. This is the same data that Amazon Web Services uses to monitor internet uptime and availability.
 * With those measurements as a baseline, Internet Monitor raises awareness for you when there are significant problems
 * for your end users in the different geographic locations where your application runs.
 * </p>
 * <p>
 * Internet Monitor publishes internet measurements to CloudWatch Logs and CloudWatch Metrics, to easily support using
 * CloudWatch tools with health information for geographies and networks specific to your application. Internet Monitor
 * sends health events to Amazon EventBridge so that you can set up notifications. If an issue is caused by the Amazon
 * Web Services network, you also automatically receive an Amazon Web Services Health Dashboard notification with the
 * steps that Amazon Web Services is taking to mitigate the problem.
 * </p>
 * <p>
 * To use Internet Monitor, you create a <i>monitor</i> and associate your application's resources with it, VPCs,
 * CloudFront distributions, or WorkSpaces directories, to enable Internet Monitor to know where your application's
 * internet traffic is. Internet Monitor then provides internet measurements from Amazon Web Services that are specific
 * to the locations and networks that communicate with your application.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-InternetMonitor.html">Using Amazon
 * CloudWatch Internet Monitor</a> in the <i>Amazon CloudWatch User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface InternetMonitorAsyncClient extends AwsClient {
    String SERVICE_NAME = "internetmonitor";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "internetmonitor";

    /**
     * <p>
     * Creates a monitor in Amazon CloudWatch Internet Monitor. A monitor is built based on information from the
     * application resources that you add: Amazon Virtual Private Clouds (VPCs), Amazon CloudFront distributions, and
     * WorkSpaces directories. Internet Monitor then publishes internet measurements from Amazon Web Services that are
     * specific to the <i>city-networks</i>, that is, the locations and ASNs (typically internet service providers or
     * ISPs), where clients access your application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-InternetMonitor.html">Using
     * Amazon CloudWatch Internet Monitor</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * When you create a monitor, you set a maximum limit for the number of city-networks where client traffic is
     * monitored. The city-network maximum that you choose is the limit, but you only pay for the number of
     * city-networks that are actually monitored. You can change the maximum at any time by updating your monitor. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param createMonitorRequest
     * @return A Java Future containing the result of the CreateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>LimitExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/CreateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateMonitorResponse> createMonitor(CreateMonitorRequest createMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a monitor in Amazon CloudWatch Internet Monitor. A monitor is built based on information from the
     * application resources that you add: Amazon Virtual Private Clouds (VPCs), Amazon CloudFront distributions, and
     * WorkSpaces directories. Internet Monitor then publishes internet measurements from Amazon Web Services that are
     * specific to the <i>city-networks</i>, that is, the locations and ASNs (typically internet service providers or
     * ISPs), where clients access your application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-InternetMonitor.html">Using
     * Amazon CloudWatch Internet Monitor</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * When you create a monitor, you set a maximum limit for the number of city-networks where client traffic is
     * monitored. The city-network maximum that you choose is the limit, but you only pay for the number of
     * city-networks that are actually monitored. You can change the maximum at any time by updating your monitor. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link CreateMonitorRequest#builder()}
     * </p>
     *
     * @param createMonitorRequest
     *        A {@link Consumer} that will call methods on {@link CreateMonitorInput.Builder} to create a request.
     * @return A Java Future containing the result of the CreateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException The requested resource is in use.</li>
     *         <li>LimitExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/CreateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateMonitorResponse> createMonitor(Consumer<CreateMonitorRequest.Builder> createMonitorRequest) {
        return createMonitor(CreateMonitorRequest.builder().applyMutation(createMonitorRequest).build());
    }

    /**
     * <p>
     * Deletes a monitor in Amazon CloudWatch Internet Monitor.
     * </p>
     *
     * @param deleteMonitorRequest
     * @return A Java Future containing the result of the DeleteMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/DeleteMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteMonitorResponse> deleteMonitor(DeleteMonitorRequest deleteMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a monitor in Amazon CloudWatch Internet Monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteMonitorRequest#builder()}
     * </p>
     *
     * @param deleteMonitorRequest
     *        A {@link Consumer} that will call methods on {@link DeleteMonitorInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/DeleteMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteMonitorResponse> deleteMonitor(Consumer<DeleteMonitorRequest.Builder> deleteMonitorRequest) {
        return deleteMonitor(DeleteMonitorRequest.builder().applyMutation(deleteMonitorRequest).build());
    }

    /**
     * <p>
     * Gets information the Amazon CloudWatch Internet Monitor has created and stored about a health event for a
     * specified monitor. This information includes the impacted locations, and all of the information related to the
     * event by location.
     * </p>
     * <p>
     * The information returned includes the performance, availability, and round-trip time impact, information about
     * the network providers, the event type, and so on.
     * </p>
     * <p>
     * Information rolled up at the global traffic level is also returned, including the impact type and total traffic
     * impact.
     * </p>
     *
     * @param getHealthEventRequest
     * @return A Java Future containing the result of the GetHealthEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.GetHealthEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetHealthEvent"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetHealthEventResponse> getHealthEvent(GetHealthEventRequest getHealthEventRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information the Amazon CloudWatch Internet Monitor has created and stored about a health event for a
     * specified monitor. This information includes the impacted locations, and all of the information related to the
     * event by location.
     * </p>
     * <p>
     * The information returned includes the performance, availability, and round-trip time impact, information about
     * the network providers, the event type, and so on.
     * </p>
     * <p>
     * Information rolled up at the global traffic level is also returned, including the impact type and total traffic
     * impact.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetHealthEventRequest.Builder} avoiding the need to
     * create one manually via {@link GetHealthEventRequest#builder()}
     * </p>
     *
     * @param getHealthEventRequest
     *        A {@link Consumer} that will call methods on {@link GetHealthEventInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetHealthEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.GetHealthEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetHealthEvent"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetHealthEventResponse> getHealthEvent(Consumer<GetHealthEventRequest.Builder> getHealthEventRequest) {
        return getHealthEvent(GetHealthEventRequest.builder().applyMutation(getHealthEventRequest).build());
    }

    /**
     * <p>
     * Gets information about a monitor in Amazon CloudWatch Internet Monitor based on a monitor name. The information
     * returned includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor,
     * and status information.
     * </p>
     *
     * @param getMonitorRequest
     * @return A Java Future containing the result of the GetMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetMonitorResponse> getMonitor(GetMonitorRequest getMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about a monitor in Amazon CloudWatch Internet Monitor based on a monitor name. The information
     * returned includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor,
     * and status information.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link GetMonitorRequest#builder()}
     * </p>
     *
     * @param getMonitorRequest
     *        A {@link Consumer} that will call methods on {@link GetMonitorInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetMonitorResponse> getMonitor(Consumer<GetMonitorRequest.Builder> getMonitorRequest) {
        return getMonitor(GetMonitorRequest.builder().applyMutation(getMonitorRequest).build());
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note>
     *
     * @param listHealthEventsRequest
     * @return A Java Future containing the result of the ListHealthEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListHealthEventsResponse> listHealthEvents(ListHealthEventsRequest listHealthEventsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListHealthEventsRequest.Builder} avoiding the need
     * to create one manually via {@link ListHealthEventsRequest#builder()}
     * </p>
     *
     * @param listHealthEventsRequest
     *        A {@link Consumer} that will call methods on {@link ListHealthEventsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListHealthEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListHealthEventsResponse> listHealthEvents(
            Consumer<ListHealthEventsRequest.Builder> listHealthEventsRequest) {
        return listHealthEvents(ListHealthEventsRequest.builder().applyMutation(listHealthEventsRequest).build());
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsPublisher publisher = client.listHealthEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsPublisher publisher = client.listHealthEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHealthEventsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHealthEventsPublisher listHealthEventsPaginator(ListHealthEventsRequest listHealthEventsRequest) {
        return new ListHealthEventsPublisher(this, listHealthEventsRequest);
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsPublisher publisher = client.listHealthEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsPublisher publisher = client.listHealthEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListHealthEventsRequest.Builder} avoiding the need
     * to create one manually via {@link ListHealthEventsRequest#builder()}
     * </p>
     *
     * @param listHealthEventsRequest
     *        A {@link Consumer} that will call methods on {@link ListHealthEventsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHealthEventsPublisher listHealthEventsPaginator(Consumer<ListHealthEventsRequest.Builder> listHealthEventsRequest) {
        return listHealthEventsPaginator(ListHealthEventsRequest.builder().applyMutation(listHealthEventsRequest).build());
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     *
     * @param listMonitorsRequest
     * @return A Java Future containing the result of the ListMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListMonitorsResponse> listMonitors(ListMonitorsRequest listMonitorsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMonitorsRequest.Builder} avoiding the need to
     * create one manually via {@link ListMonitorsRequest#builder()}
     * </p>
     *
     * @param listMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListMonitorsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListMonitorsResponse> listMonitors(Consumer<ListMonitorsRequest.Builder> listMonitorsRequest) {
        return listMonitors(ListMonitorsRequest.builder().applyMutation(listMonitorsRequest).build());
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsPublisher publisher = client.listMonitorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsPublisher publisher = client.listMonitorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation.</b>
     * </p>
     *
     * @param listMonitorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default ListMonitorsPublisher listMonitorsPaginator(ListMonitorsRequest listMonitorsRequest) {
        return new ListMonitorsPublisher(this, listMonitorsRequest);
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsPublisher publisher = client.listMonitorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsPublisher publisher = client.listMonitorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMonitorsRequest.Builder} avoiding the need to
     * create one manually via {@link ListMonitorsRequest#builder()}
     * </p>
     *
     * @param listMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListMonitorsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default ListMonitorsPublisher listMonitorsPaginator(Consumer<ListMonitorsRequest.Builder> listMonitorsRequest) {
        return listMonitorsPaginator(ListMonitorsRequest.builder().applyMutation(listMonitorsRequest).build());
    }

    /**
     * <p>
     * Lists the tags for a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException There were too many requests.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>NotFoundException The request specifies something that doesn't exist.</li>
     *         <li>BadRequestException A bad request was received.</li>
     *         <li>InternalServerErrorException There was an internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags for a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException There were too many requests.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>NotFoundException The request specifies something that doesn't exist.</li>
     *         <li>BadRequestException A bad request was received.</li>
     *         <li>InternalServerErrorException There was an internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds a tag to a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor. You can add
     * a maximum of 50 tags in Internet Monitor.
     * </p>
     * <p>
     * A minimum of one tag is required for this call. It returns an error if you use the <code>TagResource</code>
     * request with 0 tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException There were too many requests.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>NotFoundException The request specifies something that doesn't exist.</li>
     *         <li>BadRequestException A bad request was received.</li>
     *         <li>InternalServerErrorException There was an internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds a tag to a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor. You can add
     * a maximum of 50 tags in Internet Monitor.
     * </p>
     * <p>
     * A minimum of one tag is required for this call. It returns an error if you use the <code>TagResource</code>
     * request with 0 tags.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException There were too many requests.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>NotFoundException The request specifies something that doesn't exist.</li>
     *         <li>BadRequestException A bad request was received.</li>
     *         <li>InternalServerErrorException There was an internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException There were too many requests.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>NotFoundException The request specifies something that doesn't exist.</li>
     *         <li>BadRequestException A bad request was received.</li>
     *         <li>InternalServerErrorException There was an internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException There were too many requests.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>NotFoundException The request specifies something that doesn't exist.</li>
     *         <li>BadRequestException A bad request was received.</li>
     *         <li>InternalServerErrorException There was an internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a monitor. You can update a monitor to change the maximum number of city-networks (locations and ASNs or
     * internet service providers), to add or remove resources, or to change the status of the monitor. Note that you
     * can't change the name of a monitor.
     * </p>
     * <p>
     * The city-network maximum that you choose is the limit, but you only pay for the number of city-networks that are
     * actually monitored. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param updateMonitorRequest
     * @return A Java Future containing the result of the UpdateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UpdateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateMonitorResponse> updateMonitor(UpdateMonitorRequest updateMonitorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a monitor. You can update a monitor to change the maximum number of city-networks (locations and ASNs or
     * internet service providers), to add or remove resources, or to change the status of the monitor. Note that you
     * can't change the name of a monitor.
     * </p>
     * <p>
     * The city-network maximum that you choose is the limit, but you only pay for the number of city-networks that are
     * actually monitored. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateMonitorRequest#builder()}
     * </p>
     *
     * @param updateMonitorRequest
     *        A {@link Consumer} that will call methods on {@link UpdateMonitorInput.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>ResourceNotFoundException The request specifies a resource that doesn't exist.</li>
     *         <li>AccessDeniedException You don't have sufficient permission to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeded a service quota.</li>
     *         <li>ValidationException Invalid request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>InternetMonitorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample InternetMonitorAsyncClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UpdateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateMonitorResponse> updateMonitor(Consumer<UpdateMonitorRequest.Builder> updateMonitorRequest) {
        return updateMonitor(UpdateMonitorRequest.builder().applyMutation(updateMonitorRequest).build());
    }

    @Override
    default InternetMonitorServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link InternetMonitorAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static InternetMonitorAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link InternetMonitorAsyncClient}.
     */
    static InternetMonitorAsyncClientBuilder builder() {
        return new DefaultInternetMonitorAsyncClientBuilder();
    }
}
