/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.imds.internal;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.exception.RetryableException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.internal.http.loader.DefaultSdkHttpClientBuilder;
import software.amazon.awssdk.core.retry.RetryPolicyContext;
import software.amazon.awssdk.http.AbortableInputStream;
import software.amazon.awssdk.http.HttpExecuteRequest;
import software.amazon.awssdk.http.HttpExecuteResponse;
import software.amazon.awssdk.http.HttpStatusFamily;
import software.amazon.awssdk.http.SdkHttpClient;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.imds.Ec2MetadataClient;
import software.amazon.awssdk.imds.Ec2MetadataResponse;
import software.amazon.awssdk.imds.Ec2MetadataRetryPolicy;
import software.amazon.awssdk.imds.EndpointMode;
import software.amazon.awssdk.imds.internal.BaseEc2MetadataClient;
import software.amazon.awssdk.imds.internal.Token;
import software.amazon.awssdk.utils.Either;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.cache.CachedSupplier;
import software.amazon.awssdk.utils.cache.RefreshResult;

@SdkInternalApi
@Immutable
@ThreadSafe
public final class DefaultEc2MetadataClient
extends BaseEc2MetadataClient
implements Ec2MetadataClient {
    private static final Logger log = Logger.loggerFor(DefaultEc2MetadataClient.class);
    private final SdkHttpClient httpClient;
    private final Supplier<Token> tokenCache;
    private final boolean httpClientIsInternal;

    private DefaultEc2MetadataClient(Ec2MetadataBuilder builder) {
        super(builder);
        Validate.isTrue((builder.httpClient == null || builder.httpClientBuilder == null ? 1 : 0) != 0, (String)"The httpClient and the httpClientBuilder can't both be configured.", (Object[])new Object[0]);
        this.httpClient = Either.fromNullable((Object)builder.httpClient, (Object)builder.httpClientBuilder).map(e -> (SdkHttpClient)e.map(Function.identity(), SdkHttpClient.Builder::build)).orElseGet(() -> new DefaultSdkHttpClientBuilder().buildWithDefaults(this.imdsHttpDefaults()));
        this.httpClientIsInternal = builder.httpClient == null;
        this.tokenCache = CachedSupplier.builder(() -> RefreshResult.builder((Object)this.getToken()).staleTime(Instant.now().plus(this.tokenTtl)).build()).cachedValueName(this.toString()).build();
    }

    public String toString() {
        return ToString.create((String)"Ec2MetadataClient");
    }

    public void close() {
        if (this.httpClientIsInternal) {
            this.httpClient.close();
        }
    }

    public static Ec2MetadataBuilder builder() {
        return new Ec2MetadataBuilder();
    }

    @Override
    public Ec2MetadataResponse get(String path) {
        Throwable lastCause = null;
        Token token = null;
        for (int attempt = 0; attempt < this.retryPolicy.numRetries() + 1; ++attempt) {
            int currentTry;
            RetryPolicyContext retryPolicyContext = RetryPolicyContext.builder().retriesAttempted(attempt).build();
            try {
                if (token == null || token.isExpired()) {
                    token = this.tokenCache.get();
                }
                return this.sendRequest(path, token.value());
            }
            catch (UncheckedIOException | RetryableException e) {
                lastCause = e;
                currentTry = attempt;
                log.debug(() -> "Error while executing EC2Metadata request, attempting retry. Current attempt: " + currentTry);
            }
            catch (SdkClientException sdkClientException) {
                int totalTries = attempt + 1;
                log.debug(() -> String.format("Error while executing EC2Metadata request. Total attempts: %d. %s", totalTries, sdkClientException.getMessage()));
                throw sdkClientException;
            }
            catch (IOException ioe) {
                lastCause = new UncheckedIOException(ioe);
                currentTry = attempt;
                log.debug(() -> "Error while executing EC2Metadata request, attempting retry. Current attempt: " + currentTry);
            }
            this.pauseBeforeRetryIfNeeded(retryPolicyContext);
        }
        SdkClientException.Builder sdkClientExceptionBuilder = SdkClientException.builder().message("Exceeded maximum number of retries. Total retry attempts: " + this.retryPolicy.numRetries());
        if (lastCause != null) {
            String msg = sdkClientExceptionBuilder.message();
            sdkClientExceptionBuilder.cause(lastCause).message(msg);
        }
        throw sdkClientExceptionBuilder.build();
    }

    private Ec2MetadataResponse sendRequest(String path, String token) throws IOException {
        HttpExecuteRequest httpExecuteRequest = HttpExecuteRequest.builder().request((SdkHttpRequest)this.requestMarshaller.createDataRequest(path, token, this.tokenTtl)).build();
        HttpExecuteResponse response = this.httpClient.prepareRequest(httpExecuteRequest).call();
        int statusCode = response.httpResponse().statusCode();
        Optional responseBody = response.responseBody();
        if (HttpStatusFamily.of((int)statusCode).isOneOf(new HttpStatusFamily[]{HttpStatusFamily.SERVER_ERROR})) {
            responseBody.map(BaseEc2MetadataClient::uncheckedInputStreamToUtf8).ifPresent(str -> log.debug(() -> "Metadata request response body: " + str));
            throw RetryableException.builder().message(String.format("The requested metadata at path '%s' returned Http code %s", path, statusCode)).build();
        }
        if (!HttpStatusFamily.of((int)statusCode).isOneOf(new HttpStatusFamily[]{HttpStatusFamily.SUCCESSFUL})) {
            responseBody.map(BaseEc2MetadataClient::uncheckedInputStreamToUtf8).ifPresent(str -> log.debug(() -> "Metadata request response body: " + str));
            throw SdkClientException.builder().message(String.format("The requested metadata at path '%s' returned Http code %s", path, statusCode)).build();
        }
        AbortableInputStream abortableInputStream = (AbortableInputStream)responseBody.orElseThrow(() -> ((SdkClientException.Builder)SdkClientException.builder().message("Response body empty with Status Code " + statusCode)).build());
        String data = DefaultEc2MetadataClient.uncheckedInputStreamToUtf8(abortableInputStream);
        return Ec2MetadataResponse.create(data);
    }

    private void pauseBeforeRetryIfNeeded(RetryPolicyContext retryPolicyContext) {
        long backoffTimeMillis = this.retryPolicy.backoffStrategy().computeDelayBeforeNextRetry(retryPolicyContext).toMillis();
        if (backoffTimeMillis > 0L) {
            try {
                TimeUnit.MILLISECONDS.sleep(backoffTimeMillis);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw SdkClientException.builder().message("Thread interrupted while trying to sleep").cause((Throwable)e).build();
            }
        }
    }

    private Token getToken() {
        Duration ttlDuration;
        HttpExecuteRequest httpExecuteRequest = HttpExecuteRequest.builder().request((SdkHttpRequest)this.requestMarshaller.createTokenRequest(this.tokenTtl)).build();
        HttpExecuteResponse response = null;
        try {
            response = this.httpClient.prepareRequest(httpExecuteRequest).call();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        int statusCode = response.httpResponse().statusCode();
        if (HttpStatusFamily.of((int)statusCode).isOneOf(new HttpStatusFamily[]{HttpStatusFamily.SERVER_ERROR})) {
            response.responseBody().map(BaseEc2MetadataClient::uncheckedInputStreamToUtf8).ifPresent(str -> log.debug(() -> "Metadata request response body: " + str));
            throw RetryableException.builder().message("Could not retrieve token, " + statusCode + " error occurred").build();
        }
        if (!HttpStatusFamily.of((int)statusCode).isOneOf(new HttpStatusFamily[]{HttpStatusFamily.SUCCESSFUL})) {
            response.responseBody().map(BaseEc2MetadataClient::uncheckedInputStreamToUtf8).ifPresent(body -> log.debug(() -> "Token request response body: " + body));
            throw SdkClientException.builder().message("Could not retrieve token, " + statusCode + " error occurred.").build();
        }
        String ttl = (String)response.httpResponse().firstMatchingHeader("x-aws-ec2-metadata-token-ttl-seconds").orElseThrow(() -> SdkClientException.builder().message("x-aws-ec2-metadata-token-ttl-seconds header not found in token response").build());
        try {
            ttlDuration = Duration.ofSeconds(Long.parseLong(ttl));
        }
        catch (NumberFormatException nfe) {
            throw SdkClientException.create((String)"Invalid token format received from IMDS server", (Throwable)nfe);
        }
        AbortableInputStream abortableInputStream = (AbortableInputStream)response.responseBody().orElseThrow(() -> ((SdkClientException.Builder)SdkClientException.builder().message("Empty response body")).build());
        String value = DefaultEc2MetadataClient.uncheckedInputStreamToUtf8(abortableInputStream);
        return new Token(value, ttlDuration);
    }

    protected static final class Ec2MetadataBuilder
    implements Ec2MetadataClient.Builder {
        private Ec2MetadataRetryPolicy retryPolicy;
        private URI endpoint;
        private Duration tokenTtl;
        private EndpointMode endpointMode;
        private SdkHttpClient httpClient;
        private SdkHttpClient.Builder<?> httpClientBuilder;

        private Ec2MetadataBuilder() {
        }

        @Override
        public Ec2MetadataBuilder retryPolicy(Ec2MetadataRetryPolicy retryPolicy) {
            this.retryPolicy = retryPolicy;
            return this;
        }

        @Override
        public Ec2MetadataClient.Builder retryPolicy(Consumer<Ec2MetadataRetryPolicy.Builder> builderConsumer) {
            Validate.notNull(builderConsumer, (String)"builderConsumer must not be null", (Object[])new Object[0]);
            Ec2MetadataRetryPolicy.Builder builder = Ec2MetadataRetryPolicy.builder();
            builderConsumer.accept(builder);
            return this.retryPolicy((Ec2MetadataRetryPolicy)builder.build());
        }

        @Override
        public Ec2MetadataBuilder endpoint(URI endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        @Override
        public Ec2MetadataBuilder tokenTtl(Duration tokenTtl) {
            this.tokenTtl = tokenTtl;
            return this;
        }

        @Override
        public Ec2MetadataBuilder endpointMode(EndpointMode endpointMode) {
            this.endpointMode = endpointMode;
            return this;
        }

        @Override
        public Ec2MetadataBuilder httpClient(SdkHttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }

        @Override
        public Ec2MetadataClient.Builder httpClient(SdkHttpClient.Builder<?> builder) {
            this.httpClientBuilder = builder;
            return this;
        }

        public Ec2MetadataRetryPolicy getRetryPolicy() {
            return this.retryPolicy;
        }

        public URI getEndpoint() {
            return this.endpoint;
        }

        public Duration getTokenTtl() {
            return this.tokenTtl;
        }

        public EndpointMode getEndpointMode() {
            return this.endpointMode;
        }

        public Ec2MetadataClient build() {
            return new DefaultEc2MetadataClient(this);
        }
    }
}

