/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the results of a simulation.
 * </p>
 * <p>
 * This data type is used by the return parameter of <code> <a>SimulateCustomPolicy</a> </code> and
 * <code> <a>SimulatePrincipalPolicy</a> </code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EvaluationResult implements SdkPojo, Serializable,
        ToCopyableBuilder<EvaluationResult.Builder, EvaluationResult> {
    private static final SdkField<String> EVAL_ACTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EvaluationResult::evalActionName)).setter(setter(Builder::evalActionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalActionName").build()).build();

    private static final SdkField<String> EVAL_RESOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EvaluationResult::evalResourceName)).setter(setter(Builder::evalResourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalResourceName").build()).build();

    private static final SdkField<String> EVAL_DECISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EvaluationResult::evalDecisionAsString)).setter(setter(Builder::evalDecision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalDecision").build()).build();

    private static final SdkField<List<Statement>> MATCHED_STATEMENTS_FIELD = SdkField
            .<List<Statement>> builder(MarshallingType.LIST)
            .getter(getter(EvaluationResult::matchedStatements))
            .setter(setter(Builder::matchedStatements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchedStatements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Statement> builder(MarshallingType.SDK_POJO)
                                            .constructor(Statement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> MISSING_CONTEXT_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(EvaluationResult::missingContextValues))
            .setter(setter(Builder::missingContextValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MissingContextValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<OrganizationsDecisionDetail> ORGANIZATIONS_DECISION_DETAIL_FIELD = SdkField
            .<OrganizationsDecisionDetail> builder(MarshallingType.SDK_POJO)
            .getter(getter(EvaluationResult::organizationsDecisionDetail))
            .setter(setter(Builder::organizationsDecisionDetail))
            .constructor(OrganizationsDecisionDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationsDecisionDetail")
                    .build()).build();

    private static final SdkField<PermissionsBoundaryDecisionDetail> PERMISSIONS_BOUNDARY_DECISION_DETAIL_FIELD = SdkField
            .<PermissionsBoundaryDecisionDetail> builder(MarshallingType.SDK_POJO)
            .getter(getter(EvaluationResult::permissionsBoundaryDecisionDetail))
            .setter(setter(Builder::permissionsBoundaryDecisionDetail))
            .constructor(PermissionsBoundaryDecisionDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionsBoundaryDecisionDetail")
                    .build()).build();

    private static final SdkField<Map<String, String>> EVAL_DECISION_DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(EvaluationResult::evalDecisionDetailsAsStrings))
            .setter(setter(Builder::evalDecisionDetailsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalDecisionDetails").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<ResourceSpecificResult>> RESOURCE_SPECIFIC_RESULTS_FIELD = SdkField
            .<List<ResourceSpecificResult>> builder(MarshallingType.LIST)
            .getter(getter(EvaluationResult::resourceSpecificResults))
            .setter(setter(Builder::resourceSpecificResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceSpecificResults").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceSpecificResult> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceSpecificResult::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVAL_ACTION_NAME_FIELD,
            EVAL_RESOURCE_NAME_FIELD, EVAL_DECISION_FIELD, MATCHED_STATEMENTS_FIELD, MISSING_CONTEXT_VALUES_FIELD,
            ORGANIZATIONS_DECISION_DETAIL_FIELD, PERMISSIONS_BOUNDARY_DECISION_DETAIL_FIELD, EVAL_DECISION_DETAILS_FIELD,
            RESOURCE_SPECIFIC_RESULTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String evalActionName;

    private final String evalResourceName;

    private final String evalDecision;

    private final List<Statement> matchedStatements;

    private final List<String> missingContextValues;

    private final OrganizationsDecisionDetail organizationsDecisionDetail;

    private final PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail;

    private final Map<String, String> evalDecisionDetails;

    private final List<ResourceSpecificResult> resourceSpecificResults;

    private EvaluationResult(BuilderImpl builder) {
        this.evalActionName = builder.evalActionName;
        this.evalResourceName = builder.evalResourceName;
        this.evalDecision = builder.evalDecision;
        this.matchedStatements = builder.matchedStatements;
        this.missingContextValues = builder.missingContextValues;
        this.organizationsDecisionDetail = builder.organizationsDecisionDetail;
        this.permissionsBoundaryDecisionDetail = builder.permissionsBoundaryDecisionDetail;
        this.evalDecisionDetails = builder.evalDecisionDetails;
        this.resourceSpecificResults = builder.resourceSpecificResults;
    }

    /**
     * <p>
     * The name of the API operation tested on the indicated resource.
     * </p>
     * 
     * @return The name of the API operation tested on the indicated resource.
     */
    public String evalActionName() {
        return evalActionName;
    }

    /**
     * <p>
     * The ARN of the resource that the indicated API operation was tested on.
     * </p>
     * 
     * @return The ARN of the resource that the indicated API operation was tested on.
     */
    public String evalResourceName() {
        return evalResourceName;
    }

    /**
     * <p>
     * The result of the simulation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #evalDecision} will
     * return {@link PolicyEvaluationDecisionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #evalDecisionAsString}.
     * </p>
     * 
     * @return The result of the simulation.
     * @see PolicyEvaluationDecisionType
     */
    public PolicyEvaluationDecisionType evalDecision() {
        return PolicyEvaluationDecisionType.fromValue(evalDecision);
    }

    /**
     * <p>
     * The result of the simulation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #evalDecision} will
     * return {@link PolicyEvaluationDecisionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #evalDecisionAsString}.
     * </p>
     * 
     * @return The result of the simulation.
     * @see PolicyEvaluationDecisionType
     */
    public String evalDecisionAsString() {
        return evalDecision;
    }

    /**
     * Returns true if the MatchedStatements property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMatchedStatements() {
        return matchedStatements != null && !(matchedStatements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the statements in the input policies that determine the result for this scenario. Remember that even if
     * multiple statements allow the operation on the resource, if only one statement denies that operation, then the
     * explicit deny overrides any allow. In addition, the deny statement is the only entry included in the result.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMatchedStatements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the statements in the input policies that determine the result for this scenario. Remember that
     *         even if multiple statements allow the operation on the resource, if only one statement denies that
     *         operation, then the explicit deny overrides any allow. In addition, the deny statement is the only entry
     *         included in the result.
     */
    public List<Statement> matchedStatements() {
        return matchedStatements;
    }

    /**
     * Returns true if the MissingContextValues property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMissingContextValues() {
        return missingContextValues != null && !(missingContextValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of context keys that are required by the included input policies but that were not provided by one of the
     * input parameters. This list is used when the resource in a simulation is "*", either explicitly, or when the
     * <code>ResourceArns</code> parameter blank. If you include a list of resources, then any missing context values
     * are instead included under the <code>ResourceSpecificResults</code> section. To discover the context keys used by
     * a set of policies, you can call <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMissingContextValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of context keys that are required by the included input policies but that were not provided by one
     *         of the input parameters. This list is used when the resource in a simulation is "*", either explicitly,
     *         or when the <code>ResourceArns</code> parameter blank. If you include a list of resources, then any
     *         missing context values are instead included under the <code>ResourceSpecificResults</code> section. To
     *         discover the context keys used by a set of policies, you can call <a>GetContextKeysForCustomPolicy</a> or
     *         <a>GetContextKeysForPrincipalPolicy</a>.
     */
    public List<String> missingContextValues() {
        return missingContextValues;
    }

    /**
     * <p>
     * A structure that details how Organizations and its service control policies affect the results of the simulation.
     * Only applies if the simulated user's account is part of an organization.
     * </p>
     * 
     * @return A structure that details how Organizations and its service control policies affect the results of the
     *         simulation. Only applies if the simulated user's account is part of an organization.
     */
    public OrganizationsDecisionDetail organizationsDecisionDetail() {
        return organizationsDecisionDetail;
    }

    /**
     * <p>
     * Contains information about the effect that a permissions boundary has on a policy simulation when the boundary is
     * applied to an IAM entity.
     * </p>
     * 
     * @return Contains information about the effect that a permissions boundary has on a policy simulation when the
     *         boundary is applied to an IAM entity.
     */
    public PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail() {
        return permissionsBoundaryDecisionDetail;
    }

    /**
     * <p>
     * Additional details about the results of the cross-account evaluation decision. This parameter is populated for
     * only cross-account simulations. It contains a brief summary of how each policy type contributes to the final
     * evaluation decision.
     * </p>
     * <p>
     * If the simulation evaluates policies within the same account and includes a resource ARN, then the parameter is
     * present but the response is empty. If the simulation evaluates policies within the same account and specifies all
     * resources (<code>*</code>), then the parameter is not returned.
     * </p>
     * <p>
     * When you make a cross-account request, AWS evaluates the request in the trusting account and the trusted account.
     * The request is allowed only if both evaluations return <code>true</code>. For more information about how policies
     * are evaluated, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
     * >Evaluating Policies Within a Single Account</a>.
     * </p>
     * <p>
     * If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this case, policy
     * evaluation does not proceed any further and this parameter is not returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEvalDecisionDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Additional details about the results of the cross-account evaluation decision. This parameter is
     *         populated for only cross-account simulations. It contains a brief summary of how each policy type
     *         contributes to the final evaluation decision.</p>
     *         <p>
     *         If the simulation evaluates policies within the same account and includes a resource ARN, then the
     *         parameter is present but the response is empty. If the simulation evaluates policies within the same
     *         account and specifies all resources (<code>*</code>), then the parameter is not returned.
     *         </p>
     *         <p>
     *         When you make a cross-account request, AWS evaluates the request in the trusting account and the trusted
     *         account. The request is allowed only if both evaluations return <code>true</code>. For more information
     *         about how policies are evaluated, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
     *         >Evaluating Policies Within a Single Account</a>.
     *         </p>
     *         <p>
     *         If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this case,
     *         policy evaluation does not proceed any further and this parameter is not returned.
     */
    public Map<String, PolicyEvaluationDecisionType> evalDecisionDetails() {
        return EvalDecisionDetailsTypeCopier.copyStringToEnum(evalDecisionDetails);
    }

    /**
     * Returns true if the EvalDecisionDetails property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEvalDecisionDetails() {
        return evalDecisionDetails != null && !(evalDecisionDetails instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional details about the results of the cross-account evaluation decision. This parameter is populated for
     * only cross-account simulations. It contains a brief summary of how each policy type contributes to the final
     * evaluation decision.
     * </p>
     * <p>
     * If the simulation evaluates policies within the same account and includes a resource ARN, then the parameter is
     * present but the response is empty. If the simulation evaluates policies within the same account and specifies all
     * resources (<code>*</code>), then the parameter is not returned.
     * </p>
     * <p>
     * When you make a cross-account request, AWS evaluates the request in the trusting account and the trusted account.
     * The request is allowed only if both evaluations return <code>true</code>. For more information about how policies
     * are evaluated, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
     * >Evaluating Policies Within a Single Account</a>.
     * </p>
     * <p>
     * If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this case, policy
     * evaluation does not proceed any further and this parameter is not returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEvalDecisionDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Additional details about the results of the cross-account evaluation decision. This parameter is
     *         populated for only cross-account simulations. It contains a brief summary of how each policy type
     *         contributes to the final evaluation decision.</p>
     *         <p>
     *         If the simulation evaluates policies within the same account and includes a resource ARN, then the
     *         parameter is present but the response is empty. If the simulation evaluates policies within the same
     *         account and specifies all resources (<code>*</code>), then the parameter is not returned.
     *         </p>
     *         <p>
     *         When you make a cross-account request, AWS evaluates the request in the trusting account and the trusted
     *         account. The request is allowed only if both evaluations return <code>true</code>. For more information
     *         about how policies are evaluated, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
     *         >Evaluating Policies Within a Single Account</a>.
     *         </p>
     *         <p>
     *         If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this case,
     *         policy evaluation does not proceed any further and this parameter is not returned.
     */
    public Map<String, String> evalDecisionDetailsAsStrings() {
        return evalDecisionDetails;
    }

    /**
     * Returns true if the ResourceSpecificResults property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasResourceSpecificResults() {
        return resourceSpecificResults != null && !(resourceSpecificResults instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The individual results of the simulation of the API operation specified in EvalActionName on each resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceSpecificResults()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The individual results of the simulation of the API operation specified in EvalActionName on each
     *         resource.
     */
    public List<ResourceSpecificResult> resourceSpecificResults() {
        return resourceSpecificResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evalActionName());
        hashCode = 31 * hashCode + Objects.hashCode(evalResourceName());
        hashCode = 31 * hashCode + Objects.hashCode(evalDecisionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(matchedStatements());
        hashCode = 31 * hashCode + Objects.hashCode(missingContextValues());
        hashCode = 31 * hashCode + Objects.hashCode(organizationsDecisionDetail());
        hashCode = 31 * hashCode + Objects.hashCode(permissionsBoundaryDecisionDetail());
        hashCode = 31 * hashCode + Objects.hashCode(evalDecisionDetailsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(resourceSpecificResults());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EvaluationResult)) {
            return false;
        }
        EvaluationResult other = (EvaluationResult) obj;
        return Objects.equals(evalActionName(), other.evalActionName())
                && Objects.equals(evalResourceName(), other.evalResourceName())
                && Objects.equals(evalDecisionAsString(), other.evalDecisionAsString())
                && Objects.equals(matchedStatements(), other.matchedStatements())
                && Objects.equals(missingContextValues(), other.missingContextValues())
                && Objects.equals(organizationsDecisionDetail(), other.organizationsDecisionDetail())
                && Objects.equals(permissionsBoundaryDecisionDetail(), other.permissionsBoundaryDecisionDetail())
                && Objects.equals(evalDecisionDetailsAsStrings(), other.evalDecisionDetailsAsStrings())
                && Objects.equals(resourceSpecificResults(), other.resourceSpecificResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EvaluationResult").add("EvalActionName", evalActionName())
                .add("EvalResourceName", evalResourceName()).add("EvalDecision", evalDecisionAsString())
                .add("MatchedStatements", matchedStatements()).add("MissingContextValues", missingContextValues())
                .add("OrganizationsDecisionDetail", organizationsDecisionDetail())
                .add("PermissionsBoundaryDecisionDetail", permissionsBoundaryDecisionDetail())
                .add("EvalDecisionDetails", evalDecisionDetailsAsStrings())
                .add("ResourceSpecificResults", resourceSpecificResults()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EvalActionName":
            return Optional.ofNullable(clazz.cast(evalActionName()));
        case "EvalResourceName":
            return Optional.ofNullable(clazz.cast(evalResourceName()));
        case "EvalDecision":
            return Optional.ofNullable(clazz.cast(evalDecisionAsString()));
        case "MatchedStatements":
            return Optional.ofNullable(clazz.cast(matchedStatements()));
        case "MissingContextValues":
            return Optional.ofNullable(clazz.cast(missingContextValues()));
        case "OrganizationsDecisionDetail":
            return Optional.ofNullable(clazz.cast(organizationsDecisionDetail()));
        case "PermissionsBoundaryDecisionDetail":
            return Optional.ofNullable(clazz.cast(permissionsBoundaryDecisionDetail()));
        case "EvalDecisionDetails":
            return Optional.ofNullable(clazz.cast(evalDecisionDetailsAsStrings()));
        case "ResourceSpecificResults":
            return Optional.ofNullable(clazz.cast(resourceSpecificResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EvaluationResult, T> g) {
        return obj -> g.apply((EvaluationResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EvaluationResult> {
        /**
         * <p>
         * The name of the API operation tested on the indicated resource.
         * </p>
         * 
         * @param evalActionName
         *        The name of the API operation tested on the indicated resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalActionName(String evalActionName);

        /**
         * <p>
         * The ARN of the resource that the indicated API operation was tested on.
         * </p>
         * 
         * @param evalResourceName
         *        The ARN of the resource that the indicated API operation was tested on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalResourceName(String evalResourceName);

        /**
         * <p>
         * The result of the simulation.
         * </p>
         * 
         * @param evalDecision
         *        The result of the simulation.
         * @see PolicyEvaluationDecisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyEvaluationDecisionType
         */
        Builder evalDecision(String evalDecision);

        /**
         * <p>
         * The result of the simulation.
         * </p>
         * 
         * @param evalDecision
         *        The result of the simulation.
         * @see PolicyEvaluationDecisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyEvaluationDecisionType
         */
        Builder evalDecision(PolicyEvaluationDecisionType evalDecision);

        /**
         * <p>
         * A list of the statements in the input policies that determine the result for this scenario. Remember that
         * even if multiple statements allow the operation on the resource, if only one statement denies that operation,
         * then the explicit deny overrides any allow. In addition, the deny statement is the only entry included in the
         * result.
         * </p>
         * 
         * @param matchedStatements
         *        A list of the statements in the input policies that determine the result for this scenario. Remember
         *        that even if multiple statements allow the operation on the resource, if only one statement denies
         *        that operation, then the explicit deny overrides any allow. In addition, the deny statement is the
         *        only entry included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchedStatements(Collection<Statement> matchedStatements);

        /**
         * <p>
         * A list of the statements in the input policies that determine the result for this scenario. Remember that
         * even if multiple statements allow the operation on the resource, if only one statement denies that operation,
         * then the explicit deny overrides any allow. In addition, the deny statement is the only entry included in the
         * result.
         * </p>
         * 
         * @param matchedStatements
         *        A list of the statements in the input policies that determine the result for this scenario. Remember
         *        that even if multiple statements allow the operation on the resource, if only one statement denies
         *        that operation, then the explicit deny overrides any allow. In addition, the deny statement is the
         *        only entry included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchedStatements(Statement... matchedStatements);

        /**
         * <p>
         * A list of the statements in the input policies that determine the result for this scenario. Remember that
         * even if multiple statements allow the operation on the resource, if only one statement denies that operation,
         * then the explicit deny overrides any allow. In addition, the deny statement is the only entry included in the
         * result.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Statement>.Builder} avoiding the need to
         * create one manually via {@link List<Statement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Statement>.Builder#build()} is called immediately and its
         * result is passed to {@link #matchedStatements(List<Statement>)}.
         * 
         * @param matchedStatements
         *        a consumer that will call methods on {@link List<Statement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matchedStatements(List<Statement>)
         */
        Builder matchedStatements(Consumer<Statement.Builder>... matchedStatements);

        /**
         * <p>
         * A list of context keys that are required by the included input policies but that were not provided by one of
         * the input parameters. This list is used when the resource in a simulation is "*", either explicitly, or when
         * the <code>ResourceArns</code> parameter blank. If you include a list of resources, then any missing context
         * values are instead included under the <code>ResourceSpecificResults</code> section. To discover the context
         * keys used by a set of policies, you can call <a>GetContextKeysForCustomPolicy</a> or
         * <a>GetContextKeysForPrincipalPolicy</a>.
         * </p>
         * 
         * @param missingContextValues
         *        A list of context keys that are required by the included input policies but that were not provided by
         *        one of the input parameters. This list is used when the resource in a simulation is "*", either
         *        explicitly, or when the <code>ResourceArns</code> parameter blank. If you include a list of resources,
         *        then any missing context values are instead included under the <code>ResourceSpecificResults</code>
         *        section. To discover the context keys used by a set of policies, you can call
         *        <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingContextValues(Collection<String> missingContextValues);

        /**
         * <p>
         * A list of context keys that are required by the included input policies but that were not provided by one of
         * the input parameters. This list is used when the resource in a simulation is "*", either explicitly, or when
         * the <code>ResourceArns</code> parameter blank. If you include a list of resources, then any missing context
         * values are instead included under the <code>ResourceSpecificResults</code> section. To discover the context
         * keys used by a set of policies, you can call <a>GetContextKeysForCustomPolicy</a> or
         * <a>GetContextKeysForPrincipalPolicy</a>.
         * </p>
         * 
         * @param missingContextValues
         *        A list of context keys that are required by the included input policies but that were not provided by
         *        one of the input parameters. This list is used when the resource in a simulation is "*", either
         *        explicitly, or when the <code>ResourceArns</code> parameter blank. If you include a list of resources,
         *        then any missing context values are instead included under the <code>ResourceSpecificResults</code>
         *        section. To discover the context keys used by a set of policies, you can call
         *        <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingContextValues(String... missingContextValues);

        /**
         * <p>
         * A structure that details how Organizations and its service control policies affect the results of the
         * simulation. Only applies if the simulated user's account is part of an organization.
         * </p>
         * 
         * @param organizationsDecisionDetail
         *        A structure that details how Organizations and its service control policies affect the results of the
         *        simulation. Only applies if the simulated user's account is part of an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationsDecisionDetail(OrganizationsDecisionDetail organizationsDecisionDetail);

        /**
         * <p>
         * A structure that details how Organizations and its service control policies affect the results of the
         * simulation. Only applies if the simulated user's account is part of an organization.
         * </p>
         * This is a convenience that creates an instance of the {@link OrganizationsDecisionDetail.Builder} avoiding
         * the need to create one manually via {@link OrganizationsDecisionDetail#builder()}.
         *
         * When the {@link Consumer} completes, {@link OrganizationsDecisionDetail.Builder#build()} is called
         * immediately and its result is passed to {@link #organizationsDecisionDetail(OrganizationsDecisionDetail)}.
         * 
         * @param organizationsDecisionDetail
         *        a consumer that will call methods on {@link OrganizationsDecisionDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #organizationsDecisionDetail(OrganizationsDecisionDetail)
         */
        default Builder organizationsDecisionDetail(Consumer<OrganizationsDecisionDetail.Builder> organizationsDecisionDetail) {
            return organizationsDecisionDetail(OrganizationsDecisionDetail.builder().applyMutation(organizationsDecisionDetail)
                    .build());
        }

        /**
         * <p>
         * Contains information about the effect that a permissions boundary has on a policy simulation when the
         * boundary is applied to an IAM entity.
         * </p>
         * 
         * @param permissionsBoundaryDecisionDetail
         *        Contains information about the effect that a permissions boundary has on a policy simulation when the
         *        boundary is applied to an IAM entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail);

        /**
         * <p>
         * Contains information about the effect that a permissions boundary has on a policy simulation when the
         * boundary is applied to an IAM entity.
         * </p>
         * This is a convenience that creates an instance of the {@link PermissionsBoundaryDecisionDetail.Builder}
         * avoiding the need to create one manually via {@link PermissionsBoundaryDecisionDetail#builder()}.
         *
         * When the {@link Consumer} completes, {@link PermissionsBoundaryDecisionDetail.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail)}.
         * 
         * @param permissionsBoundaryDecisionDetail
         *        a consumer that will call methods on {@link PermissionsBoundaryDecisionDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail)
         */
        default Builder permissionsBoundaryDecisionDetail(
                Consumer<PermissionsBoundaryDecisionDetail.Builder> permissionsBoundaryDecisionDetail) {
            return permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail.builder()
                    .applyMutation(permissionsBoundaryDecisionDetail).build());
        }

        /**
         * <p>
         * Additional details about the results of the cross-account evaluation decision. This parameter is populated
         * for only cross-account simulations. It contains a brief summary of how each policy type contributes to the
         * final evaluation decision.
         * </p>
         * <p>
         * If the simulation evaluates policies within the same account and includes a resource ARN, then the parameter
         * is present but the response is empty. If the simulation evaluates policies within the same account and
         * specifies all resources (<code>*</code>), then the parameter is not returned.
         * </p>
         * <p>
         * When you make a cross-account request, AWS evaluates the request in the trusting account and the trusted
         * account. The request is allowed only if both evaluations return <code>true</code>. For more information about
         * how policies are evaluated, see <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
         * >Evaluating Policies Within a Single Account</a>.
         * </p>
         * <p>
         * If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this case,
         * policy evaluation does not proceed any further and this parameter is not returned.
         * </p>
         * 
         * @param evalDecisionDetails
         *        Additional details about the results of the cross-account evaluation decision. This parameter is
         *        populated for only cross-account simulations. It contains a brief summary of how each policy type
         *        contributes to the final evaluation decision.</p>
         *        <p>
         *        If the simulation evaluates policies within the same account and includes a resource ARN, then the
         *        parameter is present but the response is empty. If the simulation evaluates policies within the same
         *        account and specifies all resources (<code>*</code>), then the parameter is not returned.
         *        </p>
         *        <p>
         *        When you make a cross-account request, AWS evaluates the request in the trusting account and the
         *        trusted account. The request is allowed only if both evaluations return <code>true</code>. For more
         *        information about how policies are evaluated, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
         *        >Evaluating Policies Within a Single Account</a>.
         *        </p>
         *        <p>
         *        If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this
         *        case, policy evaluation does not proceed any further and this parameter is not returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalDecisionDetailsWithStrings(Map<String, String> evalDecisionDetails);

        /**
         * <p>
         * Additional details about the results of the cross-account evaluation decision. This parameter is populated
         * for only cross-account simulations. It contains a brief summary of how each policy type contributes to the
         * final evaluation decision.
         * </p>
         * <p>
         * If the simulation evaluates policies within the same account and includes a resource ARN, then the parameter
         * is present but the response is empty. If the simulation evaluates policies within the same account and
         * specifies all resources (<code>*</code>), then the parameter is not returned.
         * </p>
         * <p>
         * When you make a cross-account request, AWS evaluates the request in the trusting account and the trusted
         * account. The request is allowed only if both evaluations return <code>true</code>. For more information about
         * how policies are evaluated, see <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
         * >Evaluating Policies Within a Single Account</a>.
         * </p>
         * <p>
         * If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this case,
         * policy evaluation does not proceed any further and this parameter is not returned.
         * </p>
         * 
         * @param evalDecisionDetails
         *        Additional details about the results of the cross-account evaluation decision. This parameter is
         *        populated for only cross-account simulations. It contains a brief summary of how each policy type
         *        contributes to the final evaluation decision.</p>
         *        <p>
         *        If the simulation evaluates policies within the same account and includes a resource ARN, then the
         *        parameter is present but the response is empty. If the simulation evaluates policies within the same
         *        account and specifies all resources (<code>*</code>), then the parameter is not returned.
         *        </p>
         *        <p>
         *        When you make a cross-account request, AWS evaluates the request in the trusting account and the
         *        trusted account. The request is allowed only if both evaluations return <code>true</code>. For more
         *        information about how policies are evaluated, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html#policy-eval-basics"
         *        >Evaluating Policies Within a Single Account</a>.
         *        </p>
         *        <p>
         *        If an AWS Organizations SCP included in the evaluation denies access, the simulation ends. In this
         *        case, policy evaluation does not proceed any further and this parameter is not returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalDecisionDetails(Map<String, PolicyEvaluationDecisionType> evalDecisionDetails);

        /**
         * <p>
         * The individual results of the simulation of the API operation specified in EvalActionName on each resource.
         * </p>
         * 
         * @param resourceSpecificResults
         *        The individual results of the simulation of the API operation specified in EvalActionName on each
         *        resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceSpecificResults(Collection<ResourceSpecificResult> resourceSpecificResults);

        /**
         * <p>
         * The individual results of the simulation of the API operation specified in EvalActionName on each resource.
         * </p>
         * 
         * @param resourceSpecificResults
         *        The individual results of the simulation of the API operation specified in EvalActionName on each
         *        resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceSpecificResults(ResourceSpecificResult... resourceSpecificResults);

        /**
         * <p>
         * The individual results of the simulation of the API operation specified in EvalActionName on each resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceSpecificResult>.Builder} avoiding
         * the need to create one manually via {@link List<ResourceSpecificResult>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceSpecificResult>.Builder#build()} is called
         * immediately and its result is passed to {@link #resourceSpecificResults(List<ResourceSpecificResult>)}.
         * 
         * @param resourceSpecificResults
         *        a consumer that will call methods on {@link List<ResourceSpecificResult>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceSpecificResults(List<ResourceSpecificResult>)
         */
        Builder resourceSpecificResults(Consumer<ResourceSpecificResult.Builder>... resourceSpecificResults);
    }

    static final class BuilderImpl implements Builder {
        private String evalActionName;

        private String evalResourceName;

        private String evalDecision;

        private List<Statement> matchedStatements = DefaultSdkAutoConstructList.getInstance();

        private List<String> missingContextValues = DefaultSdkAutoConstructList.getInstance();

        private OrganizationsDecisionDetail organizationsDecisionDetail;

        private PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail;

        private Map<String, String> evalDecisionDetails = DefaultSdkAutoConstructMap.getInstance();

        private List<ResourceSpecificResult> resourceSpecificResults = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EvaluationResult model) {
            evalActionName(model.evalActionName);
            evalResourceName(model.evalResourceName);
            evalDecision(model.evalDecision);
            matchedStatements(model.matchedStatements);
            missingContextValues(model.missingContextValues);
            organizationsDecisionDetail(model.organizationsDecisionDetail);
            permissionsBoundaryDecisionDetail(model.permissionsBoundaryDecisionDetail);
            evalDecisionDetailsWithStrings(model.evalDecisionDetails);
            resourceSpecificResults(model.resourceSpecificResults);
        }

        public final String getEvalActionName() {
            return evalActionName;
        }

        @Override
        public final Builder evalActionName(String evalActionName) {
            this.evalActionName = evalActionName;
            return this;
        }

        public final void setEvalActionName(String evalActionName) {
            this.evalActionName = evalActionName;
        }

        public final String getEvalResourceName() {
            return evalResourceName;
        }

        @Override
        public final Builder evalResourceName(String evalResourceName) {
            this.evalResourceName = evalResourceName;
            return this;
        }

        public final void setEvalResourceName(String evalResourceName) {
            this.evalResourceName = evalResourceName;
        }

        public final String getEvalDecision() {
            return evalDecision;
        }

        @Override
        public final Builder evalDecision(String evalDecision) {
            this.evalDecision = evalDecision;
            return this;
        }

        @Override
        public final Builder evalDecision(PolicyEvaluationDecisionType evalDecision) {
            this.evalDecision(evalDecision == null ? null : evalDecision.toString());
            return this;
        }

        public final void setEvalDecision(String evalDecision) {
            this.evalDecision = evalDecision;
        }

        public final Collection<Statement.Builder> getMatchedStatements() {
            return matchedStatements != null ? matchedStatements.stream().map(Statement::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder matchedStatements(Collection<Statement> matchedStatements) {
            this.matchedStatements = StatementListTypeCopier.copy(matchedStatements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchedStatements(Statement... matchedStatements) {
            matchedStatements(Arrays.asList(matchedStatements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchedStatements(Consumer<Statement.Builder>... matchedStatements) {
            matchedStatements(Stream.of(matchedStatements).map(c -> Statement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMatchedStatements(Collection<Statement.BuilderImpl> matchedStatements) {
            this.matchedStatements = StatementListTypeCopier.copyFromBuilder(matchedStatements);
        }

        public final Collection<String> getMissingContextValues() {
            return missingContextValues;
        }

        @Override
        public final Builder missingContextValues(Collection<String> missingContextValues) {
            this.missingContextValues = ContextKeyNamesResultListTypeCopier.copy(missingContextValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder missingContextValues(String... missingContextValues) {
            missingContextValues(Arrays.asList(missingContextValues));
            return this;
        }

        public final void setMissingContextValues(Collection<String> missingContextValues) {
            this.missingContextValues = ContextKeyNamesResultListTypeCopier.copy(missingContextValues);
        }

        public final OrganizationsDecisionDetail.Builder getOrganizationsDecisionDetail() {
            return organizationsDecisionDetail != null ? organizationsDecisionDetail.toBuilder() : null;
        }

        @Override
        public final Builder organizationsDecisionDetail(OrganizationsDecisionDetail organizationsDecisionDetail) {
            this.organizationsDecisionDetail = organizationsDecisionDetail;
            return this;
        }

        public final void setOrganizationsDecisionDetail(OrganizationsDecisionDetail.BuilderImpl organizationsDecisionDetail) {
            this.organizationsDecisionDetail = organizationsDecisionDetail != null ? organizationsDecisionDetail.build() : null;
        }

        public final PermissionsBoundaryDecisionDetail.Builder getPermissionsBoundaryDecisionDetail() {
            return permissionsBoundaryDecisionDetail != null ? permissionsBoundaryDecisionDetail.toBuilder() : null;
        }

        @Override
        public final Builder permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail) {
            this.permissionsBoundaryDecisionDetail = permissionsBoundaryDecisionDetail;
            return this;
        }

        public final void setPermissionsBoundaryDecisionDetail(
                PermissionsBoundaryDecisionDetail.BuilderImpl permissionsBoundaryDecisionDetail) {
            this.permissionsBoundaryDecisionDetail = permissionsBoundaryDecisionDetail != null ? permissionsBoundaryDecisionDetail
                    .build() : null;
        }

        public final Map<String, String> getEvalDecisionDetails() {
            return evalDecisionDetails;
        }

        @Override
        public final Builder evalDecisionDetailsWithStrings(Map<String, String> evalDecisionDetails) {
            this.evalDecisionDetails = EvalDecisionDetailsTypeCopier.copy(evalDecisionDetails);
            return this;
        }

        @Override
        public final Builder evalDecisionDetails(Map<String, PolicyEvaluationDecisionType> evalDecisionDetails) {
            this.evalDecisionDetails = EvalDecisionDetailsTypeCopier.copyEnumToString(evalDecisionDetails);
            return this;
        }

        public final void setEvalDecisionDetails(Map<String, String> evalDecisionDetails) {
            this.evalDecisionDetails = EvalDecisionDetailsTypeCopier.copy(evalDecisionDetails);
        }

        public final Collection<ResourceSpecificResult.Builder> getResourceSpecificResults() {
            return resourceSpecificResults != null ? resourceSpecificResults.stream().map(ResourceSpecificResult::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resourceSpecificResults(Collection<ResourceSpecificResult> resourceSpecificResults) {
            this.resourceSpecificResults = ResourceSpecificResultListTypeCopier.copy(resourceSpecificResults);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceSpecificResults(ResourceSpecificResult... resourceSpecificResults) {
            resourceSpecificResults(Arrays.asList(resourceSpecificResults));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceSpecificResults(Consumer<ResourceSpecificResult.Builder>... resourceSpecificResults) {
            resourceSpecificResults(Stream.of(resourceSpecificResults)
                    .map(c -> ResourceSpecificResult.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setResourceSpecificResults(Collection<ResourceSpecificResult.BuilderImpl> resourceSpecificResults) {
            this.resourceSpecificResults = ResourceSpecificResultListTypeCopier.copyFromBuilder(resourceSpecificResults);
        }

        @Override
        public EvaluationResult build() {
            return new EvaluationResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
