/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an instance profile.
 * </p>
 * <p>
 * This data type is used as a response element in the following operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateInstanceProfile</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetInstanceProfile</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListInstanceProfiles</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListInstanceProfilesForRole</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceProfile implements SdkPojo, Serializable, ToCopyableBuilder<InstanceProfile.Builder, InstanceProfile> {
    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceProfile::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<String> INSTANCE_PROFILE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceProfile::instanceProfileName)).setter(setter(Builder::instanceProfileName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceProfileName").build())
            .build();

    private static final SdkField<String> INSTANCE_PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceProfile::instanceProfileId)).setter(setter(Builder::instanceProfileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceProfileId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceProfile::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<Instant> CREATE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstanceProfile::createDate)).setter(setter(Builder::createDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateDate").build()).build();

    private static final SdkField<List<Role>> ROLES_FIELD = SdkField
            .<List<Role>> builder(MarshallingType.LIST)
            .getter(getter(InstanceProfile::roles))
            .setter(setter(Builder::roles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Roles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Role> builder(MarshallingType.SDK_POJO)
                                            .constructor(Role::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATH_FIELD,
            INSTANCE_PROFILE_NAME_FIELD, INSTANCE_PROFILE_ID_FIELD, ARN_FIELD, CREATE_DATE_FIELD, ROLES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String path;

    private final String instanceProfileName;

    private final String instanceProfileId;

    private final String arn;

    private final Instant createDate;

    private final List<Role> roles;

    private InstanceProfile(BuilderImpl builder) {
        this.path = builder.path;
        this.instanceProfileName = builder.instanceProfileName;
        this.instanceProfileId = builder.instanceProfileId;
        this.arn = builder.arn;
        this.createDate = builder.createDate;
        this.roles = builder.roles;
    }

    /**
     * <p>
     * The path to the instance profile. For more information about paths, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the <i>IAM
     * User Guide</i>.
     * </p>
     * 
     * @return The path to the instance profile. For more information about paths, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the
     *         <i>IAM User Guide</i>.
     */
    public String path() {
        return path;
    }

    /**
     * <p>
     * The name identifying the instance profile.
     * </p>
     * 
     * @return The name identifying the instance profile.
     */
    public String instanceProfileName() {
        return instanceProfileName;
    }

    /**
     * <p>
     * The stable and unique string identifying the instance profile. For more information about IDs, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the <i>IAM
     * User Guide</i>.
     * </p>
     * 
     * @return The stable and unique string identifying the instance profile. For more information about IDs, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the
     *         <i>IAM User Guide</i>.
     */
    public String instanceProfileId() {
        return instanceProfileId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) specifying the instance profile. For more information about ARNs and how to use
     * them in policies, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM
     * Identifiers</a> in the <i>IAM User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) specifying the instance profile. For more information about ARNs and how
     *         to use them in policies, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the
     *         <i>IAM User Guide</i>.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The date when the instance profile was created.
     * </p>
     * 
     * @return The date when the instance profile was created.
     */
    public Instant createDate() {
        return createDate;
    }

    /**
     * Returns true if the Roles property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRoles() {
        return roles != null && !(roles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The role associated with the instance profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRoles()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The role associated with the instance profile.
     */
    public List<Role> roles() {
        return roles;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(instanceProfileName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceProfileId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createDate());
        hashCode = 31 * hashCode + Objects.hashCode(roles());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceProfile)) {
            return false;
        }
        InstanceProfile other = (InstanceProfile) obj;
        return Objects.equals(path(), other.path()) && Objects.equals(instanceProfileName(), other.instanceProfileName())
                && Objects.equals(instanceProfileId(), other.instanceProfileId()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createDate(), other.createDate()) && Objects.equals(roles(), other.roles());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceProfile").add("Path", path()).add("InstanceProfileName", instanceProfileName())
                .add("InstanceProfileId", instanceProfileId()).add("Arn", arn()).add("CreateDate", createDate())
                .add("Roles", roles()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "InstanceProfileName":
            return Optional.ofNullable(clazz.cast(instanceProfileName()));
        case "InstanceProfileId":
            return Optional.ofNullable(clazz.cast(instanceProfileId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "CreateDate":
            return Optional.ofNullable(clazz.cast(createDate()));
        case "Roles":
            return Optional.ofNullable(clazz.cast(roles()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceProfile, T> g) {
        return obj -> g.apply((InstanceProfile) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceProfile> {
        /**
         * <p>
         * The path to the instance profile. For more information about paths, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the
         * <i>IAM User Guide</i>.
         * </p>
         * 
         * @param path
         *        The path to the instance profile. For more information about paths, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in
         *        the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * The name identifying the instance profile.
         * </p>
         * 
         * @param instanceProfileName
         *        The name identifying the instance profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceProfileName(String instanceProfileName);

        /**
         * <p>
         * The stable and unique string identifying the instance profile. For more information about IDs, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the
         * <i>IAM User Guide</i>.
         * </p>
         * 
         * @param instanceProfileId
         *        The stable and unique string identifying the instance profile. For more information about IDs, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in
         *        the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceProfileId(String instanceProfileId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) specifying the instance profile. For more information about ARNs and how to
         * use them in policies, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in the
         * <i>IAM User Guide</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) specifying the instance profile. For more information about ARNs and
         *        how to use them in policies, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM Identifiers</a> in
         *        the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date when the instance profile was created.
         * </p>
         * 
         * @param createDate
         *        The date when the instance profile was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDate(Instant createDate);

        /**
         * <p>
         * The role associated with the instance profile.
         * </p>
         * 
         * @param roles
         *        The role associated with the instance profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Collection<Role> roles);

        /**
         * <p>
         * The role associated with the instance profile.
         * </p>
         * 
         * @param roles
         *        The role associated with the instance profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Role... roles);

        /**
         * <p>
         * The role associated with the instance profile.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Role>.Builder} avoiding the need to create
         * one manually via {@link List<Role>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Role>.Builder#build()} is called immediately and its result
         * is passed to {@link #roles(List<Role>)}.
         * 
         * @param roles
         *        a consumer that will call methods on {@link List<Role>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #roles(List<Role>)
         */
        Builder roles(Consumer<Role.Builder>... roles);
    }

    static final class BuilderImpl implements Builder {
        private String path;

        private String instanceProfileName;

        private String instanceProfileId;

        private String arn;

        private Instant createDate;

        private List<Role> roles = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceProfile model) {
            path(model.path);
            instanceProfileName(model.instanceProfileName);
            instanceProfileId(model.instanceProfileId);
            arn(model.arn);
            createDate(model.createDate);
            roles(model.roles);
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final String getInstanceProfileName() {
            return instanceProfileName;
        }

        @Override
        public final Builder instanceProfileName(String instanceProfileName) {
            this.instanceProfileName = instanceProfileName;
            return this;
        }

        public final void setInstanceProfileName(String instanceProfileName) {
            this.instanceProfileName = instanceProfileName;
        }

        public final String getInstanceProfileId() {
            return instanceProfileId;
        }

        @Override
        public final Builder instanceProfileId(String instanceProfileId) {
            this.instanceProfileId = instanceProfileId;
            return this;
        }

        public final void setInstanceProfileId(String instanceProfileId) {
            this.instanceProfileId = instanceProfileId;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getCreateDate() {
            return createDate;
        }

        @Override
        public final Builder createDate(Instant createDate) {
            this.createDate = createDate;
            return this;
        }

        public final void setCreateDate(Instant createDate) {
            this.createDate = createDate;
        }

        public final Collection<Role.Builder> getRoles() {
            return roles != null ? roles.stream().map(Role::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder roles(Collection<Role> roles) {
            this.roles = _roleListTypeCopier.copy(roles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Role... roles) {
            roles(Arrays.asList(roles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Consumer<Role.Builder>... roles) {
            roles(Stream.of(roles).map(c -> Role.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRoles(Collection<Role.BuilderImpl> roles) {
            this.roles = _roleListTypeCopier.copyFromBuilder(roles);
        }

        @Override
        public InstanceProfile build() {
            return new InstanceProfile(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
