/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.greengrassv2.internal.GreengrassV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.greengrassv2.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.greengrassv2.model.AccessDeniedException;
import software.amazon.awssdk.services.greengrassv2.model.AssociateServiceRoleToAccountRequest;
import software.amazon.awssdk.services.greengrassv2.model.AssociateServiceRoleToAccountResponse;
import software.amazon.awssdk.services.greengrassv2.model.BatchAssociateClientDeviceWithCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.BatchAssociateClientDeviceWithCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.BatchDisassociateClientDeviceFromCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.BatchDisassociateClientDeviceFromCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.CancelDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.CancelDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.ConflictException;
import software.amazon.awssdk.services.greengrassv2.model.CreateComponentVersionRequest;
import software.amazon.awssdk.services.greengrassv2.model.CreateComponentVersionResponse;
import software.amazon.awssdk.services.greengrassv2.model.CreateDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.CreateDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.DeleteComponentRequest;
import software.amazon.awssdk.services.greengrassv2.model.DeleteComponentResponse;
import software.amazon.awssdk.services.greengrassv2.model.DeleteCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.DeleteCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.DeleteDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.DeleteDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.DescribeComponentRequest;
import software.amazon.awssdk.services.greengrassv2.model.DescribeComponentResponse;
import software.amazon.awssdk.services.greengrassv2.model.DisassociateServiceRoleFromAccountRequest;
import software.amazon.awssdk.services.greengrassv2.model.DisassociateServiceRoleFromAccountResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentVersionArtifactRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentVersionArtifactResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetConnectivityInfoRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetConnectivityInfoResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetServiceRoleForAccountRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetServiceRoleForAccountResponse;
import software.amazon.awssdk.services.greengrassv2.model.GreengrassV2Exception;
import software.amazon.awssdk.services.greengrassv2.model.InternalServerException;
import software.amazon.awssdk.services.greengrassv2.model.ListClientDevicesAssociatedWithCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListClientDevicesAssociatedWithCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.greengrassv2.model.RequestAlreadyInProgressException;
import software.amazon.awssdk.services.greengrassv2.model.ResolveComponentCandidatesRequest;
import software.amazon.awssdk.services.greengrassv2.model.ResolveComponentCandidatesResponse;
import software.amazon.awssdk.services.greengrassv2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.greengrassv2.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.greengrassv2.model.TagResourceRequest;
import software.amazon.awssdk.services.greengrassv2.model.TagResourceResponse;
import software.amazon.awssdk.services.greengrassv2.model.ThrottlingException;
import software.amazon.awssdk.services.greengrassv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.greengrassv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.greengrassv2.model.UpdateConnectivityInfoRequest;
import software.amazon.awssdk.services.greengrassv2.model.UpdateConnectivityInfoResponse;
import software.amazon.awssdk.services.greengrassv2.model.ValidationException;
import software.amazon.awssdk.services.greengrassv2.transform.AssociateServiceRoleToAccountRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.BatchAssociateClientDeviceWithCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.BatchDisassociateClientDeviceFromCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.CancelDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.CreateComponentVersionRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.CreateDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DeleteComponentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DeleteCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DeleteDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DescribeComponentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DisassociateServiceRoleFromAccountRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetComponentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetComponentVersionArtifactRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetConnectivityInfoRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetServiceRoleForAccountRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListClientDevicesAssociatedWithCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListComponentVersionsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListCoreDevicesRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListEffectiveDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListInstalledComponentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ResolveComponentCandidatesRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.UpdateConnectivityInfoRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GreengrassV2AsyncClient}.
 *
 * @see GreengrassV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGreengrassV2AsyncClient implements GreengrassV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGreengrassV2AsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGreengrassV2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "GreengrassV2" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a Greengrass service role with IoT Greengrass for your Amazon Web Services account in this Amazon Web
     * Services Region. IoT Greengrass uses this role to verify the identity of client devices and manage core device
     * connectivity information. The role must include the <a href=
     * "https://console.aws.amazon.com/iam/home#/policies/arn:awsiam::aws:policy/service-role/AWSGreengrassResourceAccessRolePolicy"
     * >AWSGreengrassResourceAccessRolePolicy</a> managed policy or a custom policy that defines equivalent permissions
     * for the IoT Greengrass features that you use. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-service-role.html">Greengrass service
     * role</a> in the <i>IoT Greengrass Version 2 Developer Guide</i>.
     * </p>
     *
     * @param associateServiceRoleToAccountRequest
     * @return A Java Future containing the result of the AssociateServiceRoleToAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.AssociateServiceRoleToAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/AssociateServiceRoleToAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateServiceRoleToAccountResponse> associateServiceRoleToAccount(
            AssociateServiceRoleToAccountRequest associateServiceRoleToAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateServiceRoleToAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateServiceRoleToAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateServiceRoleToAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateServiceRoleToAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateServiceRoleToAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateServiceRoleToAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateServiceRoleToAccountRequest, AssociateServiceRoleToAccountResponse>()
                            .withOperationName("AssociateServiceRoleToAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateServiceRoleToAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateServiceRoleToAccountRequest));
            CompletableFuture<AssociateServiceRoleToAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a list of client devices with a core device. Use this API operation to specify which client devices
     * can discover a core device through cloud discovery. With cloud discovery, client devices connect to IoT
     * Greengrass to retrieve associated core devices' connectivity information and certificates. For more information,
     * see <a href="https://docs.aws.amazon.com/greengrass/v2/developerguide/configure-cloud-discovery.html">Configure
     * cloud discovery</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Client devices are local IoT devices that connect to and communicate with an IoT Greengrass core device over
     * MQTT. You can connect client devices to a core device to sync MQTT messages and data to Amazon Web Services IoT
     * Core and interact with client devices in Greengrass components. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/interact-with-local-iot-devices.html">Interact
     * with local IoT devices</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param batchAssociateClientDeviceWithCoreDeviceRequest
     * @return A Java Future containing the result of the BatchAssociateClientDeviceWithCoreDevice operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.BatchAssociateClientDeviceWithCoreDevice
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/BatchAssociateClientDeviceWithCoreDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchAssociateClientDeviceWithCoreDeviceResponse> batchAssociateClientDeviceWithCoreDevice(
            BatchAssociateClientDeviceWithCoreDeviceRequest batchAssociateClientDeviceWithCoreDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchAssociateClientDeviceWithCoreDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchAssociateClientDeviceWithCoreDeviceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchAssociateClientDeviceWithCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchAssociateClientDeviceWithCoreDeviceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchAssociateClientDeviceWithCoreDeviceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchAssociateClientDeviceWithCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchAssociateClientDeviceWithCoreDeviceRequest, BatchAssociateClientDeviceWithCoreDeviceResponse>()
                            .withOperationName("BatchAssociateClientDeviceWithCoreDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchAssociateClientDeviceWithCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchAssociateClientDeviceWithCoreDeviceRequest));
            CompletableFuture<BatchAssociateClientDeviceWithCoreDeviceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a list of client devices from a core device. After you disassociate a client device from a core
     * device, the client device won't be able to use cloud discovery to retrieve the core device's connectivity
     * information and certificates.
     * </p>
     *
     * @param batchDisassociateClientDeviceFromCoreDeviceRequest
     * @return A Java Future containing the result of the BatchDisassociateClientDeviceFromCoreDevice operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.BatchDisassociateClientDeviceFromCoreDevice
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/BatchDisassociateClientDeviceFromCoreDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDisassociateClientDeviceFromCoreDeviceResponse> batchDisassociateClientDeviceFromCoreDevice(
            BatchDisassociateClientDeviceFromCoreDeviceRequest batchDisassociateClientDeviceFromCoreDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchDisassociateClientDeviceFromCoreDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchDisassociateClientDeviceFromCoreDeviceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDisassociateClientDeviceFromCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDisassociateClientDeviceFromCoreDeviceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchDisassociateClientDeviceFromCoreDeviceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchDisassociateClientDeviceFromCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDisassociateClientDeviceFromCoreDeviceRequest, BatchDisassociateClientDeviceFromCoreDeviceResponse>()
                            .withOperationName("BatchDisassociateClientDeviceFromCoreDevice")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDisassociateClientDeviceFromCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDisassociateClientDeviceFromCoreDeviceRequest));
            CompletableFuture<BatchDisassociateClientDeviceFromCoreDeviceResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a deployment. This operation cancels the deployment for devices that haven't yet received it. If a device
     * already received the deployment, this operation doesn't change anything for that device.
     * </p>
     *
     * @param cancelDeploymentRequest
     * @return A Java Future containing the result of the CancelDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.CancelDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/CancelDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelDeploymentResponse> cancelDeployment(CancelDeploymentRequest cancelDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelDeploymentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CancelDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelDeploymentRequest, CancelDeploymentResponse>()
                            .withOperationName("CancelDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelDeploymentRequest));
            CompletableFuture<CancelDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a component. Components are software that run on Greengrass core devices. After you develop and test a
     * component on your core device, you can use this operation to upload your component to IoT Greengrass. Then, you
     * can deploy the component to other core devices.
     * </p>
     * <p>
     * You can use this operation to do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Create components from recipes</b>
     * </p>
     * <p>
     * Create a component from a recipe, which is a file that defines the component's metadata, parameters,
     * dependencies, lifecycle, artifacts, and platform capability. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/component-recipe-reference.html">IoT Greengrass
     * component recipe reference</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * <p>
     * To create a component from a recipe, specify <code>inlineRecipe</code> when you call this operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Create components from Lambda functions</b>
     * </p>
     * <p>
     * Create a component from an Lambda function that runs on IoT Greengrass. This creates a recipe and artifacts from
     * the Lambda function's deployment package. You can use this operation to migrate Lambda functions from IoT
     * Greengrass V1 to IoT Greengrass V2.
     * </p>
     * <p>
     * This function accepts Lambda functions in all supported versions of Python, Node.js, and Java runtimes. IoT
     * Greengrass doesn't apply any additional restrictions on deprecated Lambda runtime versions.
     * </p>
     * <p>
     * To create a component from a Lambda function, specify <code>lambdaFunction</code> when you call this operation.
     * </p>
     * <note>
     * <p>
     * IoT Greengrass currently supports Lambda functions on only Linux core devices.
     * </p>
     * </note></li>
     * </ul>
     *
     * @param createComponentVersionRequest
     * @return A Java Future containing the result of the CreateComponentVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds a service quota. For example, you might have the
     *         maximum number of components that you can create.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>RequestAlreadyInProgressException The request is already in progress. This exception occurs when you
     *         use a client token for multiple requests while IoT Greengrass is still processing an earlier request that
     *         uses the same client token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.CreateComponentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/CreateComponentVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComponentVersionResponse> createComponentVersion(
            CreateComponentVersionRequest createComponentVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createComponentVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComponentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComponentVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComponentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComponentVersionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateComponentVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComponentVersionRequest, CreateComponentVersionResponse>()
                            .withOperationName("CreateComponentVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateComponentVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createComponentVersionRequest));
            CompletableFuture<CreateComponentVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a continuous deployment for a target, which is a Greengrass core device or group of core devices. When
     * you add a new core device to a group of core devices that has a deployment, IoT Greengrass deploys that group's
     * deployment to the new device.
     * </p>
     * <p>
     * You can define one deployment for each target. When you create a new deployment for a target that has an existing
     * deployment, you replace the previous deployment. IoT Greengrass applies the new deployment to the target devices.
     * </p>
     * <p>
     * Every deployment has a revision number that indicates how many deployment revisions you define for a target. Use
     * this operation to create a new revision of an existing deployment.
     * </p>
     * <p>
     * For more information, see the <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create deployments</a> in
     * the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     *
     * @param createDeploymentRequest
     * @return A Java Future containing the result of the CreateDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>RequestAlreadyInProgressException The request is already in progress. This exception occurs when you
     *         use a client token for multiple requests while IoT Greengrass is still processing an earlier request that
     *         uses the same client token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.CreateDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/CreateDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeploymentResponse> createDeployment(CreateDeploymentRequest createDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeploymentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentRequest, CreateDeploymentResponse>()
                            .withOperationName("CreateDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDeploymentRequest));
            CompletableFuture<CreateDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a version of a component from IoT Greengrass.
     * </p>
     * <note>
     * <p>
     * This operation deletes the component's recipe and artifacts. As a result, deployments that refer to this
     * component version will fail. If you have deployments that use this component version, you can remove the
     * component from the deployment or update the deployment to use a valid version.
     * </p>
     * </note>
     *
     * @param deleteComponentRequest
     * @return A Java Future containing the result of the DeleteComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DeleteComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DeleteComponent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComponentResponse> deleteComponent(DeleteComponentRequest deleteComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteComponentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComponentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComponentRequest, DeleteComponentResponse>()
                            .withOperationName("DeleteComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteComponentRequest));
            CompletableFuture<DeleteComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Greengrass core device, which is an IoT thing. This operation removes the core device from the list of
     * core devices. This operation doesn't delete the IoT thing. For more information about how to delete the IoT
     * thing, see <a href="https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html">DeleteThing</a> in
     * the <i>IoT API Reference</i>.
     * </p>
     *
     * @param deleteCoreDeviceRequest
     * @return A Java Future containing the result of the DeleteCoreDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DeleteCoreDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DeleteCoreDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCoreDeviceResponse> deleteCoreDevice(DeleteCoreDeviceRequest deleteCoreDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCoreDeviceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCoreDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCoreDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCoreDeviceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCoreDeviceRequest, DeleteCoreDeviceResponse>()
                            .withOperationName("DeleteCoreDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCoreDeviceRequest));
            CompletableFuture<DeleteCoreDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a deployment. To delete an active deployment, you must first cancel it. For more information, see <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_CancelDeployment.html">CancelDeployment</a>.
     * </p>
     * <p>
     * Deleting a deployment doesn't affect core devices that run that deployment, because core devices store the
     * deployment's configuration on the device. Additionally, core devices can roll back to a previous deployment that
     * has been deleted.
     * </p>
     *
     * @param deleteDeploymentRequest
     * @return A Java Future containing the result of the DeleteDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DeleteDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DeleteDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDeploymentResponse> deleteDeployment(DeleteDeploymentRequest deleteDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDeploymentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDeploymentRequest, DeleteDeploymentResponse>()
                            .withOperationName("DeleteDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDeploymentRequest));
            CompletableFuture<DeleteDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves metadata for a version of a component.
     * </p>
     *
     * @param describeComponentRequest
     * @return A Java Future containing the result of the DescribeComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DescribeComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DescribeComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComponentResponse> describeComponent(DescribeComponentRequest describeComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeComponentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeComponentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComponentRequest, DescribeComponentResponse>()
                            .withOperationName("DescribeComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeComponentRequest));
            CompletableFuture<DescribeComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the Greengrass service role from IoT Greengrass for your Amazon Web Services account in this Amazon
     * Web Services Region. Without a service role, IoT Greengrass can't verify the identity of client devices or manage
     * core device connectivity information. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-service-role.html">Greengrass service
     * role</a> in the <i>IoT Greengrass Version 2 Developer Guide</i>.
     * </p>
     *
     * @param disassociateServiceRoleFromAccountRequest
     * @return A Java Future containing the result of the DisassociateServiceRoleFromAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DisassociateServiceRoleFromAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DisassociateServiceRoleFromAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateServiceRoleFromAccountResponse> disassociateServiceRoleFromAccount(
            DisassociateServiceRoleFromAccountRequest disassociateServiceRoleFromAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateServiceRoleFromAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateServiceRoleFromAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateServiceRoleFromAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateServiceRoleFromAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateServiceRoleFromAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateServiceRoleFromAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateServiceRoleFromAccountRequest, DisassociateServiceRoleFromAccountResponse>()
                            .withOperationName("DisassociateServiceRoleFromAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateServiceRoleFromAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateServiceRoleFromAccountRequest));
            CompletableFuture<DisassociateServiceRoleFromAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the recipe for a version of a component.
     * </p>
     *
     * @param getComponentRequest
     * @return A Java Future containing the result of the GetComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetComponent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetComponentResponse> getComponent(GetComponentRequest getComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComponentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComponentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetComponentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComponentRequest, GetComponentResponse>()
                            .withOperationName("GetComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getComponentRequest));
            CompletableFuture<GetComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the pre-signed URL to download a public or a Lambda component artifact. Core devices call this operation to
     * identify the URL that they can use to download an artifact to install.
     * </p>
     *
     * @param getComponentVersionArtifactRequest
     * @return A Java Future containing the result of the GetComponentVersionArtifact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetComponentVersionArtifact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetComponentVersionArtifact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetComponentVersionArtifactResponse> getComponentVersionArtifact(
            GetComponentVersionArtifactRequest getComponentVersionArtifactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComponentVersionArtifactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentVersionArtifactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponentVersionArtifact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComponentVersionArtifactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetComponentVersionArtifactResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetComponentVersionArtifactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComponentVersionArtifactRequest, GetComponentVersionArtifactResponse>()
                            .withOperationName("GetComponentVersionArtifact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetComponentVersionArtifactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getComponentVersionArtifactRequest));
            CompletableFuture<GetComponentVersionArtifactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves connectivity information for a Greengrass core device.
     * </p>
     * <p>
     * Connectivity information includes endpoints and ports where client devices can connect to an MQTT broker on the
     * core device. When a client device calls the <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-discover-api.html">IoT Greengrass
     * discovery API</a>, IoT Greengrass returns connectivity information for all of the core devices where the client
     * device can connect. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/connect-client-devices.html">Connect client
     * devices to core devices</a> in the <i>IoT Greengrass Version 2 Developer Guide</i>.
     * </p>
     *
     * @param getConnectivityInfoRequest
     * @return A Java Future containing the result of the GetConnectivityInfo operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetConnectivityInfo
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetConnectivityInfo"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConnectivityInfoResponse> getConnectivityInfo(
            GetConnectivityInfoRequest getConnectivityInfoRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectivityInfoRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectivityInfoRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnectivityInfo");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConnectivityInfoResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConnectivityInfoResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetConnectivityInfoResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConnectivityInfoRequest, GetConnectivityInfoResponse>()
                            .withOperationName("GetConnectivityInfo").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConnectivityInfoRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConnectivityInfoRequest));
            CompletableFuture<GetConnectivityInfoResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves metadata for a Greengrass core device.
     * </p>
     * <note>
     * <p>
     * IoT Greengrass relies on individual devices to send status updates to the Amazon Web Services Cloud. If the IoT
     * Greengrass Core software isn't running on the device, or if device isn't connected to the Amazon Web Services
     * Cloud, then the reported status of that device might not reflect its current status. The status timestamp
     * indicates when the device status was last updated.
     * </p>
     * <p>
     * Core devices send status updates at the following times:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When the IoT Greengrass Core software starts
     * </p>
     * </li>
     * <li>
     * <p>
     * When the core device receives a deployment from the Amazon Web Services Cloud
     * </p>
     * </li>
     * <li>
     * <p>
     * When the status of any component on the core device becomes <code>BROKEN</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * At a <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html#greengrass-nucleus-component-configuration-fss"
     * >regular interval that you can configure</a>, which defaults to 24 hours
     * </p>
     * </li>
     * <li>
     * <p>
     * For IoT Greengrass Core v2.7.0, the core device sends status updates upon local deployment and cloud deployment
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param getCoreDeviceRequest
     * @return A Java Future containing the result of the GetCoreDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetCoreDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetCoreDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCoreDeviceResponse> getCoreDevice(GetCoreDeviceRequest getCoreDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCoreDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCoreDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCoreDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCoreDeviceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCoreDeviceRequest, GetCoreDeviceResponse>()
                            .withOperationName("GetCoreDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCoreDeviceRequest));
            CompletableFuture<GetCoreDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a deployment. Deployments define the components that run on Greengrass core devices.
     * </p>
     *
     * @param getDeploymentRequest
     * @return A Java Future containing the result of the GetDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeploymentResponse> getDeployment(GetDeploymentRequest getDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeploymentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDeploymentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                            .withOperationName("GetDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDeploymentRequest));
            CompletableFuture<GetDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the service role associated with IoT Greengrass for your Amazon Web Services account in this Amazon Web
     * Services Region. IoT Greengrass uses this role to verify the identity of client devices and manage core device
     * connectivity information. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-service-role.html">Greengrass service
     * role</a> in the <i>IoT Greengrass Version 2 Developer Guide</i>.
     * </p>
     *
     * @param getServiceRoleForAccountRequest
     * @return A Java Future containing the result of the GetServiceRoleForAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetServiceRoleForAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetServiceRoleForAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceRoleForAccountResponse> getServiceRoleForAccount(
            GetServiceRoleForAccountRequest getServiceRoleForAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceRoleForAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceRoleForAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceRoleForAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceRoleForAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceRoleForAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetServiceRoleForAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceRoleForAccountRequest, GetServiceRoleForAccountResponse>()
                            .withOperationName("GetServiceRoleForAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceRoleForAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceRoleForAccountRequest));
            CompletableFuture<GetServiceRoleForAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of client devices that are associated with a core device.
     * </p>
     *
     * @param listClientDevicesAssociatedWithCoreDeviceRequest
     * @return A Java Future containing the result of the ListClientDevicesAssociatedWithCoreDevice operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListClientDevicesAssociatedWithCoreDevice
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListClientDevicesAssociatedWithCoreDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListClientDevicesAssociatedWithCoreDeviceResponse> listClientDevicesAssociatedWithCoreDevice(
            ListClientDevicesAssociatedWithCoreDeviceRequest listClientDevicesAssociatedWithCoreDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listClientDevicesAssociatedWithCoreDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listClientDevicesAssociatedWithCoreDeviceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClientDevicesAssociatedWithCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClientDevicesAssociatedWithCoreDeviceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListClientDevicesAssociatedWithCoreDeviceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListClientDevicesAssociatedWithCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClientDevicesAssociatedWithCoreDeviceRequest, ListClientDevicesAssociatedWithCoreDeviceResponse>()
                            .withOperationName("ListClientDevicesAssociatedWithCoreDevice")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListClientDevicesAssociatedWithCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listClientDevicesAssociatedWithCoreDeviceRequest));
            CompletableFuture<ListClientDevicesAssociatedWithCoreDeviceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of all versions for a component. Greater versions are listed first.
     * </p>
     *
     * @param listComponentVersionsRequest
     * @return A Java Future containing the result of the ListComponentVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListComponentVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListComponentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentVersionsResponse> listComponentVersions(
            ListComponentVersionsRequest listComponentVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComponentVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponentVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentVersionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListComponentVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentVersionsRequest, ListComponentVersionsResponse>()
                            .withOperationName("ListComponentVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListComponentVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listComponentVersionsRequest));
            CompletableFuture<ListComponentVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of component summaries. This list includes components that you have permission to
     * view.
     * </p>
     *
     * @param listComponentsRequest
     * @return A Java Future containing the result of the ListComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListComponents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentsResponse> listComponents(ListComponentsRequest listComponentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComponentsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                            .withOperationName("ListComponents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listComponentsRequest));
            CompletableFuture<ListComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of Greengrass core devices.
     * </p>
     * <note>
     * <p>
     * IoT Greengrass relies on individual devices to send status updates to the Amazon Web Services Cloud. If the IoT
     * Greengrass Core software isn't running on the device, or if device isn't connected to the Amazon Web Services
     * Cloud, then the reported status of that device might not reflect its current status. The status timestamp
     * indicates when the device status was last updated.
     * </p>
     * <p>
     * Core devices send status updates at the following times:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When the IoT Greengrass Core software starts
     * </p>
     * </li>
     * <li>
     * <p>
     * When the core device receives a deployment from the Amazon Web Services Cloud
     * </p>
     * </li>
     * <li>
     * <p>
     * For Greengrass nucleus 2.12.2 and earlier, the core device sends status updates when the status of any component
     * on the core device becomes <code>ERRORED</code> or <code>BROKEN</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Greengrass nucleus 2.12.3 and later, the core device sends status updates when the status of any component on
     * the core device becomes <code>ERRORED</code>, <code>BROKEN</code>, <code>RUNNING</code>, or <code>FINISHED</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * At a <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html#greengrass-nucleus-component-configuration-fss"
     * >regular interval that you can configure</a>, which defaults to 24 hours
     * </p>
     * </li>
     * <li>
     * <p>
     * For IoT Greengrass Core v2.7.0, the core device sends status updates upon local deployment and cloud deployment
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param listCoreDevicesRequest
     * @return A Java Future containing the result of the ListCoreDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListCoreDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListCoreDevices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCoreDevicesResponse> listCoreDevices(ListCoreDevicesRequest listCoreDevicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCoreDevicesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCoreDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCoreDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCoreDevicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCoreDevicesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListCoreDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCoreDevicesRequest, ListCoreDevicesResponse>()
                            .withOperationName("ListCoreDevices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCoreDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCoreDevicesRequest));
            CompletableFuture<ListCoreDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of deployments.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A Java Future containing the result of the ListDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListDeployments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeploymentsResponse> listDeployments(ListDeploymentsRequest listDeploymentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeploymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeploymentsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                            .withOperationName("ListDeployments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDeploymentsRequest));
            CompletableFuture<ListDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of deployment jobs that IoT Greengrass sends to Greengrass core devices.
     * </p>
     *
     * @param listEffectiveDeploymentsRequest
     * @return A Java Future containing the result of the ListEffectiveDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListEffectiveDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListEffectiveDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEffectiveDeploymentsResponse> listEffectiveDeployments(
            ListEffectiveDeploymentsRequest listEffectiveDeploymentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEffectiveDeploymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEffectiveDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEffectiveDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEffectiveDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEffectiveDeploymentsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListEffectiveDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEffectiveDeploymentsRequest, ListEffectiveDeploymentsResponse>()
                            .withOperationName("ListEffectiveDeployments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEffectiveDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEffectiveDeploymentsRequest));
            CompletableFuture<ListEffectiveDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of the components that a Greengrass core device runs. By default, this list doesn't
     * include components that are deployed as dependencies of other components. To include dependencies in the
     * response, set the <code>topologyFilter</code> parameter to <code>ALL</code>.
     * </p>
     * <note>
     * <p>
     * IoT Greengrass relies on individual devices to send status updates to the Amazon Web Services Cloud. If the IoT
     * Greengrass Core software isn't running on the device, or if device isn't connected to the Amazon Web Services
     * Cloud, then the reported status of that device might not reflect its current status. The status timestamp
     * indicates when the device status was last updated.
     * </p>
     * <p>
     * Core devices send status updates at the following times:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When the IoT Greengrass Core software starts
     * </p>
     * </li>
     * <li>
     * <p>
     * When the core device receives a deployment from the Amazon Web Services Cloud
     * </p>
     * </li>
     * <li>
     * <p>
     * When the status of any component on the core device becomes <code>BROKEN</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * At a <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html#greengrass-nucleus-component-configuration-fss"
     * >regular interval that you can configure</a>, which defaults to 24 hours
     * </p>
     * </li>
     * <li>
     * <p>
     * For IoT Greengrass Core v2.7.0, the core device sends status updates upon local deployment and cloud deployment
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param listInstalledComponentsRequest
     * @return A Java Future containing the result of the ListInstalledComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListInstalledComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListInstalledComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInstalledComponentsResponse> listInstalledComponents(
            ListInstalledComponentsRequest listInstalledComponentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstalledComponentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstalledComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstalledComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstalledComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInstalledComponentsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListInstalledComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInstalledComponentsRequest, ListInstalledComponentsResponse>()
                            .withOperationName("ListInstalledComponents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInstalledComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInstalledComponentsRequest));
            CompletableFuture<ListInstalledComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of tags for an IoT Greengrass resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of components that meet the component, version, and platform requirements of a deployment.
     * Greengrass core devices call this operation when they receive a deployment to identify the components to install.
     * </p>
     * <p>
     * This operation identifies components that meet all dependency requirements for a deployment. If the requirements
     * conflict, then this operation returns an error and the deployment fails. For example, this occurs if component
     * <code>A</code> requires version <code>&gt;2.0.0</code> and component <code>B</code> requires version
     * <code>&lt;2.0.0</code> of a component dependency.
     * </p>
     * <p>
     * When you specify the component candidates to resolve, IoT Greengrass compares each component's digest from the
     * core device with the component's digest in the Amazon Web Services Cloud. If the digests don't match, then IoT
     * Greengrass specifies to use the version from the Amazon Web Services Cloud.
     * </p>
     * <important>
     * <p>
     * To use this operation, you must use the data plane API endpoint and authenticate with an IoT device certificate.
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/greengrass.html">IoT Greengrass
     * endpoints and quotas</a>.
     * </p>
     * </important>
     *
     * @param resolveComponentCandidatesRequest
     * @return A Java Future containing the result of the ResolveComponentCandidates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ResolveComponentCandidates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ResolveComponentCandidates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ResolveComponentCandidatesResponse> resolveComponentCandidates(
            ResolveComponentCandidatesRequest resolveComponentCandidatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resolveComponentCandidatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveComponentCandidatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveComponentCandidates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResolveComponentCandidatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResolveComponentCandidatesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ResolveComponentCandidatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResolveComponentCandidatesRequest, ResolveComponentCandidatesResponse>()
                            .withOperationName("ResolveComponentCandidates").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResolveComponentCandidatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resolveComponentCandidatesRequest));
            CompletableFuture<ResolveComponentCandidatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to an IoT Greengrass resource. If a tag already exists for the resource, this operation updates the
     * tag's value.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from an IoT Greengrass resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates connectivity information for a Greengrass core device.
     * </p>
     * <p>
     * Connectivity information includes endpoints and ports where client devices can connect to an MQTT broker on the
     * core device. When a client device calls the <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-discover-api.html">IoT Greengrass
     * discovery API</a>, IoT Greengrass returns connectivity information for all of the core devices where the client
     * device can connect. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/connect-client-devices.html">Connect client
     * devices to core devices</a> in the <i>IoT Greengrass Version 2 Developer Guide</i>.
     * </p>
     *
     * @param updateConnectivityInfoRequest
     * @return A Java Future containing the result of the UpdateConnectivityInfo operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>InternalServerException IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.UpdateConnectivityInfo
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/UpdateConnectivityInfo"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConnectivityInfoResponse> updateConnectivityInfo(
            UpdateConnectivityInfoRequest updateConnectivityInfoRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConnectivityInfoRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectivityInfoRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnectivityInfo");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConnectivityInfoResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConnectivityInfoResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "RequestAlreadyInProgressException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestAlreadyInProgressException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RequestAlreadyInProgressException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateConnectivityInfoResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectivityInfoRequest, UpdateConnectivityInfoResponse>()
                            .withOperationName("UpdateConnectivityInfo").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConnectivityInfoRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConnectivityInfoRequest));
            CompletableFuture<UpdateConnectivityInfoResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final GreengrassV2ServiceClientConfiguration serviceClientConfiguration() {
        return new GreengrassV2ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(GreengrassV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        GreengrassV2ServiceClientConfigurationBuilder serviceConfigBuilder = new GreengrassV2ServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
