/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the schema structure for an Iceberg table, including field definitions, data types, and schema metadata.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IcebergSchema implements SdkPojo, Serializable, ToCopyableBuilder<IcebergSchema.Builder, IcebergSchema> {
    private static final SdkField<Integer> SCHEMA_ID_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SchemaId").getter(getter(IcebergSchema::schemaId)).setter(setter(Builder::schemaId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaId").build()).build();

    private static final SdkField<List<Integer>> IDENTIFIER_FIELD_IDS_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("IdentifierFieldIds")
            .getter(getter(IcebergSchema::identifierFieldIds))
            .setter(setter(Builder::identifierFieldIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifierFieldIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(IcebergSchema::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<IcebergStructField>> FIELDS_FIELD = SdkField
            .<List<IcebergStructField>> builder(MarshallingType.LIST)
            .memberName("Fields")
            .getter(getter(IcebergSchema::fields))
            .setter(setter(Builder::fields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Fields").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IcebergStructField> builder(MarshallingType.SDK_POJO)
                                            .constructor(IcebergStructField::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCHEMA_ID_FIELD,
            IDENTIFIER_FIELD_IDS_FIELD, TYPE_FIELD, FIELDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer schemaId;

    private final List<Integer> identifierFieldIds;

    private final String type;

    private final List<IcebergStructField> fields;

    private IcebergSchema(BuilderImpl builder) {
        this.schemaId = builder.schemaId;
        this.identifierFieldIds = builder.identifierFieldIds;
        this.type = builder.type;
        this.fields = builder.fields;
    }

    /**
     * <p>
     * The unique identifier for this schema version within the Iceberg table's schema evolution history.
     * </p>
     * 
     * @return The unique identifier for this schema version within the Iceberg table's schema evolution history.
     */
    public final Integer schemaId() {
        return schemaId;
    }

    /**
     * For responses, this returns true if the service returned a value for the IdentifierFieldIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIdentifierFieldIds() {
        return identifierFieldIds != null && !(identifierFieldIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of field identifiers that uniquely identify records in the table, used for row-level operations and
     * deduplication.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdentifierFieldIds} method.
     * </p>
     * 
     * @return The list of field identifiers that uniquely identify records in the table, used for row-level operations
     *         and deduplication.
     */
    public final List<Integer> identifierFieldIds() {
        return identifierFieldIds;
    }

    /**
     * <p>
     * The root type of the schema structure, typically "struct" for Iceberg table schemas.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IcebergStructTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The root type of the schema structure, typically "struct" for Iceberg table schemas.
     * @see IcebergStructTypeEnum
     */
    public final IcebergStructTypeEnum type() {
        return IcebergStructTypeEnum.fromValue(type);
    }

    /**
     * <p>
     * The root type of the schema structure, typically "struct" for Iceberg table schemas.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IcebergStructTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The root type of the schema structure, typically "struct" for Iceberg table schemas.
     * @see IcebergStructTypeEnum
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * For responses, this returns true if the service returned a value for the Fields property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFields() {
        return fields != null && !(fields instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of field definitions that make up the table schema, including field names, types, and metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFields} method.
     * </p>
     * 
     * @return The list of field definitions that make up the table schema, including field names, types, and metadata.
     */
    public final List<IcebergStructField> fields() {
        return fields;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(schemaId());
        hashCode = 31 * hashCode + Objects.hashCode(hasIdentifierFieldIds() ? identifierFieldIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFields() ? fields() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IcebergSchema)) {
            return false;
        }
        IcebergSchema other = (IcebergSchema) obj;
        return Objects.equals(schemaId(), other.schemaId()) && hasIdentifierFieldIds() == other.hasIdentifierFieldIds()
                && Objects.equals(identifierFieldIds(), other.identifierFieldIds())
                && Objects.equals(typeAsString(), other.typeAsString()) && hasFields() == other.hasFields()
                && Objects.equals(fields(), other.fields());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IcebergSchema").add("SchemaId", schemaId())
                .add("IdentifierFieldIds", hasIdentifierFieldIds() ? identifierFieldIds() : null).add("Type", typeAsString())
                .add("Fields", hasFields() ? fields() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SchemaId":
            return Optional.ofNullable(clazz.cast(schemaId()));
        case "IdentifierFieldIds":
            return Optional.ofNullable(clazz.cast(identifierFieldIds()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Fields":
            return Optional.ofNullable(clazz.cast(fields()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SchemaId", SCHEMA_ID_FIELD);
        map.put("IdentifierFieldIds", IDENTIFIER_FIELD_IDS_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("Fields", FIELDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IcebergSchema, T> g) {
        return obj -> g.apply((IcebergSchema) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IcebergSchema> {
        /**
         * <p>
         * The unique identifier for this schema version within the Iceberg table's schema evolution history.
         * </p>
         * 
         * @param schemaId
         *        The unique identifier for this schema version within the Iceberg table's schema evolution history.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaId(Integer schemaId);

        /**
         * <p>
         * The list of field identifiers that uniquely identify records in the table, used for row-level operations and
         * deduplication.
         * </p>
         * 
         * @param identifierFieldIds
         *        The list of field identifiers that uniquely identify records in the table, used for row-level
         *        operations and deduplication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifierFieldIds(Collection<Integer> identifierFieldIds);

        /**
         * <p>
         * The list of field identifiers that uniquely identify records in the table, used for row-level operations and
         * deduplication.
         * </p>
         * 
         * @param identifierFieldIds
         *        The list of field identifiers that uniquely identify records in the table, used for row-level
         *        operations and deduplication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifierFieldIds(Integer... identifierFieldIds);

        /**
         * <p>
         * The root type of the schema structure, typically "struct" for Iceberg table schemas.
         * </p>
         * 
         * @param type
         *        The root type of the schema structure, typically "struct" for Iceberg table schemas.
         * @see IcebergStructTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IcebergStructTypeEnum
         */
        Builder type(String type);

        /**
         * <p>
         * The root type of the schema structure, typically "struct" for Iceberg table schemas.
         * </p>
         * 
         * @param type
         *        The root type of the schema structure, typically "struct" for Iceberg table schemas.
         * @see IcebergStructTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IcebergStructTypeEnum
         */
        Builder type(IcebergStructTypeEnum type);

        /**
         * <p>
         * The list of field definitions that make up the table schema, including field names, types, and metadata.
         * </p>
         * 
         * @param fields
         *        The list of field definitions that make up the table schema, including field names, types, and
         *        metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fields(Collection<IcebergStructField> fields);

        /**
         * <p>
         * The list of field definitions that make up the table schema, including field names, types, and metadata.
         * </p>
         * 
         * @param fields
         *        The list of field definitions that make up the table schema, including field names, types, and
         *        metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fields(IcebergStructField... fields);

        /**
         * <p>
         * The list of field definitions that make up the table schema, including field names, types, and metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.IcebergStructField.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.glue.model.IcebergStructField#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.IcebergStructField.Builder#build()} is called immediately
         * and its result is passed to {@link #fields(List<IcebergStructField>)}.
         * 
         * @param fields
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.IcebergStructField.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fields(java.util.Collection<IcebergStructField>)
         */
        Builder fields(Consumer<IcebergStructField.Builder>... fields);
    }

    static final class BuilderImpl implements Builder {
        private Integer schemaId;

        private List<Integer> identifierFieldIds = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private List<IcebergStructField> fields = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IcebergSchema model) {
            schemaId(model.schemaId);
            identifierFieldIds(model.identifierFieldIds);
            type(model.type);
            fields(model.fields);
        }

        public final Integer getSchemaId() {
            return schemaId;
        }

        public final void setSchemaId(Integer schemaId) {
            this.schemaId = schemaId;
        }

        @Override
        public final Builder schemaId(Integer schemaId) {
            this.schemaId = schemaId;
            return this;
        }

        public final Collection<Integer> getIdentifierFieldIds() {
            if (identifierFieldIds instanceof SdkAutoConstructList) {
                return null;
            }
            return identifierFieldIds;
        }

        public final void setIdentifierFieldIds(Collection<Integer> identifierFieldIds) {
            this.identifierFieldIds = IntegerListCopier.copy(identifierFieldIds);
        }

        @Override
        public final Builder identifierFieldIds(Collection<Integer> identifierFieldIds) {
            this.identifierFieldIds = IntegerListCopier.copy(identifierFieldIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identifierFieldIds(Integer... identifierFieldIds) {
            identifierFieldIds(Arrays.asList(identifierFieldIds));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(IcebergStructTypeEnum type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final List<IcebergStructField.Builder> getFields() {
            List<IcebergStructField.Builder> result = IcebergStructFieldListCopier.copyToBuilder(this.fields);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFields(Collection<IcebergStructField.BuilderImpl> fields) {
            this.fields = IcebergStructFieldListCopier.copyFromBuilder(fields);
        }

        @Override
        public final Builder fields(Collection<IcebergStructField> fields) {
            this.fields = IcebergStructFieldListCopier.copy(fields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fields(IcebergStructField... fields) {
            fields(Arrays.asList(fields));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fields(Consumer<IcebergStructField.Builder>... fields) {
            fields(Stream.of(fields).map(c -> IcebergStructField.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public IcebergSchema build() {
            return new IcebergSchema(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
