/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A workflow is a collection of multiple dependent Glue jobs and crawlers that are run to complete a complex ETL task.
 * A workflow manages the execution and monitoring of all its jobs and crawlers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Workflow implements SdkPojo, Serializable, ToCopyableBuilder<Workflow.Builder, Workflow> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Workflow::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Workflow::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Map<String, String>> DEFAULT_RUN_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("DefaultRunProperties")
            .getter(getter(Workflow::defaultRunProperties))
            .setter(setter(Builder::defaultRunProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultRunProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedOn").getter(getter(Workflow::createdOn)).setter(setter(Builder::createdOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedOn").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedOn").getter(getter(Workflow::lastModifiedOn)).setter(setter(Builder::lastModifiedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedOn").build()).build();

    private static final SdkField<WorkflowRun> LAST_RUN_FIELD = SdkField.<WorkflowRun> builder(MarshallingType.SDK_POJO)
            .memberName("LastRun").getter(getter(Workflow::lastRun)).setter(setter(Builder::lastRun))
            .constructor(WorkflowRun::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastRun").build()).build();

    private static final SdkField<WorkflowGraph> GRAPH_FIELD = SdkField.<WorkflowGraph> builder(MarshallingType.SDK_POJO)
            .memberName("Graph").getter(getter(Workflow::graph)).setter(setter(Builder::graph))
            .constructor(WorkflowGraph::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Graph").build()).build();

    private static final SdkField<Integer> MAX_CONCURRENT_RUNS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConcurrentRuns").getter(getter(Workflow::maxConcurrentRuns))
            .setter(setter(Builder::maxConcurrentRuns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrentRuns").build()).build();

    private static final SdkField<BlueprintDetails> BLUEPRINT_DETAILS_FIELD = SdkField
            .<BlueprintDetails> builder(MarshallingType.SDK_POJO).memberName("BlueprintDetails")
            .getter(getter(Workflow::blueprintDetails)).setter(setter(Builder::blueprintDetails))
            .constructor(BlueprintDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlueprintDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            DEFAULT_RUN_PROPERTIES_FIELD, CREATED_ON_FIELD, LAST_MODIFIED_ON_FIELD, LAST_RUN_FIELD, GRAPH_FIELD,
            MAX_CONCURRENT_RUNS_FIELD, BLUEPRINT_DETAILS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final Map<String, String> defaultRunProperties;

    private final Instant createdOn;

    private final Instant lastModifiedOn;

    private final WorkflowRun lastRun;

    private final WorkflowGraph graph;

    private final Integer maxConcurrentRuns;

    private final BlueprintDetails blueprintDetails;

    private Workflow(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.defaultRunProperties = builder.defaultRunProperties;
        this.createdOn = builder.createdOn;
        this.lastModifiedOn = builder.lastModifiedOn;
        this.lastRun = builder.lastRun;
        this.graph = builder.graph;
        this.maxConcurrentRuns = builder.maxConcurrentRuns;
        this.blueprintDetails = builder.blueprintDetails;
    }

    /**
     * <p>
     * The name of the workflow.
     * </p>
     * 
     * @return The name of the workflow.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A description of the workflow.
     * </p>
     * 
     * @return A description of the workflow.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the DefaultRunProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDefaultRunProperties() {
        return defaultRunProperties != null && !(defaultRunProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A collection of properties to be used as part of each execution of the workflow. The run properties are made
     * available to each job in the workflow. A job can modify the properties for the next jobs in the flow.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDefaultRunProperties} method.
     * </p>
     * 
     * @return A collection of properties to be used as part of each execution of the workflow. The run properties are
     *         made available to each job in the workflow. A job can modify the properties for the next jobs in the
     *         flow.
     */
    public final Map<String, String> defaultRunProperties() {
        return defaultRunProperties;
    }

    /**
     * <p>
     * The date and time when the workflow was created.
     * </p>
     * 
     * @return The date and time when the workflow was created.
     */
    public final Instant createdOn() {
        return createdOn;
    }

    /**
     * <p>
     * The date and time when the workflow was last modified.
     * </p>
     * 
     * @return The date and time when the workflow was last modified.
     */
    public final Instant lastModifiedOn() {
        return lastModifiedOn;
    }

    /**
     * <p>
     * The information about the last execution of the workflow.
     * </p>
     * 
     * @return The information about the last execution of the workflow.
     */
    public final WorkflowRun lastRun() {
        return lastRun;
    }

    /**
     * <p>
     * The graph representing all the Glue components that belong to the workflow as nodes and directed connections
     * between them as edges.
     * </p>
     * 
     * @return The graph representing all the Glue components that belong to the workflow as nodes and directed
     *         connections between them as edges.
     */
    public final WorkflowGraph graph() {
        return graph;
    }

    /**
     * <p>
     * You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some cases, to
     * prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you leave this parameter
     * blank, there is no limit to the number of concurrent workflow runs.
     * </p>
     * 
     * @return You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some
     *         cases, to prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you
     *         leave this parameter blank, there is no limit to the number of concurrent workflow runs.
     */
    public final Integer maxConcurrentRuns() {
        return maxConcurrentRuns;
    }

    /**
     * <p>
     * This structure indicates the details of the blueprint that this particular workflow is created from.
     * </p>
     * 
     * @return This structure indicates the details of the blueprint that this particular workflow is created from.
     */
    public final BlueprintDetails blueprintDetails() {
        return blueprintDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasDefaultRunProperties() ? defaultRunProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdOn());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedOn());
        hashCode = 31 * hashCode + Objects.hashCode(lastRun());
        hashCode = 31 * hashCode + Objects.hashCode(graph());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentRuns());
        hashCode = 31 * hashCode + Objects.hashCode(blueprintDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Workflow)) {
            return false;
        }
        Workflow other = (Workflow) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && hasDefaultRunProperties() == other.hasDefaultRunProperties()
                && Objects.equals(defaultRunProperties(), other.defaultRunProperties())
                && Objects.equals(createdOn(), other.createdOn()) && Objects.equals(lastModifiedOn(), other.lastModifiedOn())
                && Objects.equals(lastRun(), other.lastRun()) && Objects.equals(graph(), other.graph())
                && Objects.equals(maxConcurrentRuns(), other.maxConcurrentRuns())
                && Objects.equals(blueprintDetails(), other.blueprintDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Workflow").add("Name", name()).add("Description", description())
                .add("DefaultRunProperties", hasDefaultRunProperties() ? defaultRunProperties() : null)
                .add("CreatedOn", createdOn()).add("LastModifiedOn", lastModifiedOn()).add("LastRun", lastRun())
                .add("Graph", graph()).add("MaxConcurrentRuns", maxConcurrentRuns()).add("BlueprintDetails", blueprintDetails())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DefaultRunProperties":
            return Optional.ofNullable(clazz.cast(defaultRunProperties()));
        case "CreatedOn":
            return Optional.ofNullable(clazz.cast(createdOn()));
        case "LastModifiedOn":
            return Optional.ofNullable(clazz.cast(lastModifiedOn()));
        case "LastRun":
            return Optional.ofNullable(clazz.cast(lastRun()));
        case "Graph":
            return Optional.ofNullable(clazz.cast(graph()));
        case "MaxConcurrentRuns":
            return Optional.ofNullable(clazz.cast(maxConcurrentRuns()));
        case "BlueprintDetails":
            return Optional.ofNullable(clazz.cast(blueprintDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("DefaultRunProperties", DEFAULT_RUN_PROPERTIES_FIELD);
        map.put("CreatedOn", CREATED_ON_FIELD);
        map.put("LastModifiedOn", LAST_MODIFIED_ON_FIELD);
        map.put("LastRun", LAST_RUN_FIELD);
        map.put("Graph", GRAPH_FIELD);
        map.put("MaxConcurrentRuns", MAX_CONCURRENT_RUNS_FIELD);
        map.put("BlueprintDetails", BLUEPRINT_DETAILS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Workflow, T> g) {
        return obj -> g.apply((Workflow) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Workflow> {
        /**
         * <p>
         * The name of the workflow.
         * </p>
         * 
         * @param name
         *        The name of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the workflow.
         * </p>
         * 
         * @param description
         *        A description of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A collection of properties to be used as part of each execution of the workflow. The run properties are made
         * available to each job in the workflow. A job can modify the properties for the next jobs in the flow.
         * </p>
         * 
         * @param defaultRunProperties
         *        A collection of properties to be used as part of each execution of the workflow. The run properties
         *        are made available to each job in the workflow. A job can modify the properties for the next jobs in
         *        the flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultRunProperties(Map<String, String> defaultRunProperties);

        /**
         * <p>
         * The date and time when the workflow was created.
         * </p>
         * 
         * @param createdOn
         *        The date and time when the workflow was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdOn(Instant createdOn);

        /**
         * <p>
         * The date and time when the workflow was last modified.
         * </p>
         * 
         * @param lastModifiedOn
         *        The date and time when the workflow was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedOn(Instant lastModifiedOn);

        /**
         * <p>
         * The information about the last execution of the workflow.
         * </p>
         * 
         * @param lastRun
         *        The information about the last execution of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRun(WorkflowRun lastRun);

        /**
         * <p>
         * The information about the last execution of the workflow.
         * </p>
         * This is a convenience method that creates an instance of the {@link WorkflowRun.Builder} avoiding the need to
         * create one manually via {@link WorkflowRun#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WorkflowRun.Builder#build()} is called immediately and its result
         * is passed to {@link #lastRun(WorkflowRun)}.
         * 
         * @param lastRun
         *        a consumer that will call methods on {@link WorkflowRun.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastRun(WorkflowRun)
         */
        default Builder lastRun(Consumer<WorkflowRun.Builder> lastRun) {
            return lastRun(WorkflowRun.builder().applyMutation(lastRun).build());
        }

        /**
         * <p>
         * The graph representing all the Glue components that belong to the workflow as nodes and directed connections
         * between them as edges.
         * </p>
         * 
         * @param graph
         *        The graph representing all the Glue components that belong to the workflow as nodes and directed
         *        connections between them as edges.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder graph(WorkflowGraph graph);

        /**
         * <p>
         * The graph representing all the Glue components that belong to the workflow as nodes and directed connections
         * between them as edges.
         * </p>
         * This is a convenience method that creates an instance of the {@link WorkflowGraph.Builder} avoiding the need
         * to create one manually via {@link WorkflowGraph#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WorkflowGraph.Builder#build()} is called immediately and its
         * result is passed to {@link #graph(WorkflowGraph)}.
         * 
         * @param graph
         *        a consumer that will call methods on {@link WorkflowGraph.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #graph(WorkflowGraph)
         */
        default Builder graph(Consumer<WorkflowGraph.Builder> graph) {
            return graph(WorkflowGraph.builder().applyMutation(graph).build());
        }

        /**
         * <p>
         * You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some cases,
         * to prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you leave this
         * parameter blank, there is no limit to the number of concurrent workflow runs.
         * </p>
         * 
         * @param maxConcurrentRuns
         *        You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some
         *        cases, to prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you
         *        leave this parameter blank, there is no limit to the number of concurrent workflow runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentRuns(Integer maxConcurrentRuns);

        /**
         * <p>
         * This structure indicates the details of the blueprint that this particular workflow is created from.
         * </p>
         * 
         * @param blueprintDetails
         *        This structure indicates the details of the blueprint that this particular workflow is created from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blueprintDetails(BlueprintDetails blueprintDetails);

        /**
         * <p>
         * This structure indicates the details of the blueprint that this particular workflow is created from.
         * </p>
         * This is a convenience method that creates an instance of the {@link BlueprintDetails.Builder} avoiding the
         * need to create one manually via {@link BlueprintDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BlueprintDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #blueprintDetails(BlueprintDetails)}.
         * 
         * @param blueprintDetails
         *        a consumer that will call methods on {@link BlueprintDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #blueprintDetails(BlueprintDetails)
         */
        default Builder blueprintDetails(Consumer<BlueprintDetails.Builder> blueprintDetails) {
            return blueprintDetails(BlueprintDetails.builder().applyMutation(blueprintDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private Map<String, String> defaultRunProperties = DefaultSdkAutoConstructMap.getInstance();

        private Instant createdOn;

        private Instant lastModifiedOn;

        private WorkflowRun lastRun;

        private WorkflowGraph graph;

        private Integer maxConcurrentRuns;

        private BlueprintDetails blueprintDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(Workflow model) {
            name(model.name);
            description(model.description);
            defaultRunProperties(model.defaultRunProperties);
            createdOn(model.createdOn);
            lastModifiedOn(model.lastModifiedOn);
            lastRun(model.lastRun);
            graph(model.graph);
            maxConcurrentRuns(model.maxConcurrentRuns);
            blueprintDetails(model.blueprintDetails);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Map<String, String> getDefaultRunProperties() {
            if (defaultRunProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return defaultRunProperties;
        }

        public final void setDefaultRunProperties(Map<String, String> defaultRunProperties) {
            this.defaultRunProperties = WorkflowRunPropertiesCopier.copy(defaultRunProperties);
        }

        @Override
        public final Builder defaultRunProperties(Map<String, String> defaultRunProperties) {
            this.defaultRunProperties = WorkflowRunPropertiesCopier.copy(defaultRunProperties);
            return this;
        }

        public final Instant getCreatedOn() {
            return createdOn;
        }

        public final void setCreatedOn(Instant createdOn) {
            this.createdOn = createdOn;
        }

        @Override
        public final Builder createdOn(Instant createdOn) {
            this.createdOn = createdOn;
            return this;
        }

        public final Instant getLastModifiedOn() {
            return lastModifiedOn;
        }

        public final void setLastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
        }

        @Override
        public final Builder lastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
            return this;
        }

        public final WorkflowRun.Builder getLastRun() {
            return lastRun != null ? lastRun.toBuilder() : null;
        }

        public final void setLastRun(WorkflowRun.BuilderImpl lastRun) {
            this.lastRun = lastRun != null ? lastRun.build() : null;
        }

        @Override
        public final Builder lastRun(WorkflowRun lastRun) {
            this.lastRun = lastRun;
            return this;
        }

        public final WorkflowGraph.Builder getGraph() {
            return graph != null ? graph.toBuilder() : null;
        }

        public final void setGraph(WorkflowGraph.BuilderImpl graph) {
            this.graph = graph != null ? graph.build() : null;
        }

        @Override
        public final Builder graph(WorkflowGraph graph) {
            this.graph = graph;
            return this;
        }

        public final Integer getMaxConcurrentRuns() {
            return maxConcurrentRuns;
        }

        public final void setMaxConcurrentRuns(Integer maxConcurrentRuns) {
            this.maxConcurrentRuns = maxConcurrentRuns;
        }

        @Override
        public final Builder maxConcurrentRuns(Integer maxConcurrentRuns) {
            this.maxConcurrentRuns = maxConcurrentRuns;
            return this;
        }

        public final BlueprintDetails.Builder getBlueprintDetails() {
            return blueprintDetails != null ? blueprintDetails.toBuilder() : null;
        }

        public final void setBlueprintDetails(BlueprintDetails.BuilderImpl blueprintDetails) {
            this.blueprintDetails = blueprintDetails != null ? blueprintDetails.build() : null;
        }

        @Override
        public final Builder blueprintDetails(BlueprintDetails blueprintDetails) {
            this.blueprintDetails = blueprintDetails;
            return this;
        }

        @Override
        public Workflow build() {
            return new Workflow(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
