/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a source generated with standard connection options.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConnectorDataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<ConnectorDataSource.Builder, ConnectorDataSource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ConnectorDataSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CONNECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionType").getter(getter(ConnectorDataSource::connectionType))
            .setter(setter(Builder::connectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionType").build()).build();

    private static final SdkField<Map<String, String>> DATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Data")
            .getter(getter(ConnectorDataSource::data))
            .setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Data").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<GlueSchema>> OUTPUT_SCHEMAS_FIELD = SdkField
            .<List<GlueSchema>> builder(MarshallingType.LIST)
            .memberName("OutputSchemas")
            .getter(getter(ConnectorDataSource::outputSchemas))
            .setter(setter(Builder::outputSchemas))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputSchemas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GlueSchema> builder(MarshallingType.SDK_POJO)
                                            .constructor(GlueSchema::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONNECTION_TYPE_FIELD, DATA_FIELD, OUTPUT_SCHEMAS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String connectionType;

    private final Map<String, String> data;

    private final List<GlueSchema> outputSchemas;

    private ConnectorDataSource(BuilderImpl builder) {
        this.name = builder.name;
        this.connectionType = builder.connectionType;
        this.data = builder.data;
        this.outputSchemas = builder.outputSchemas;
    }

    /**
     * <p>
     * The name of this source node.
     * </p>
     * 
     * @return The name of this source node.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The <code>connectionType</code>, as provided to the underlying Glue library. This node type supports the
     * following connection types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>opensearch</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>azuresql</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>azurecosmos</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>bigquery</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>saphana</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>teradata</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vertica</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The <code>connectionType</code>, as provided to the underlying Glue library. This node type supports the
     *         following connection types: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>opensearch</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>azuresql</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>azurecosmos</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>bigquery</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>saphana</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>teradata</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vertica</code>
     *         </p>
     *         </li>
     */
    public final String connectionType() {
        return connectionType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Data property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasData() {
        return data != null && !(data instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map specifying connection options for the node. You can find standard connection options for the corresponding
     * connection type in the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-connect.html"> Connection
     * parameters</a> section of the Glue documentation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasData} method.
     * </p>
     * 
     * @return A map specifying connection options for the node. You can find standard connection options for the
     *         corresponding connection type in the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-connect.html"> Connection
     *         parameters</a> section of the Glue documentation.
     */
    public final Map<String, String> data() {
        return data;
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputSchemas property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputSchemas() {
        return outputSchemas != null && !(outputSchemas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the data schema for this source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputSchemas} method.
     * </p>
     * 
     * @return Specifies the data schema for this source.
     */
    public final List<GlueSchema> outputSchemas() {
        return outputSchemas;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(connectionType());
        hashCode = 31 * hashCode + Objects.hashCode(hasData() ? data() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputSchemas() ? outputSchemas() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConnectorDataSource)) {
            return false;
        }
        ConnectorDataSource other = (ConnectorDataSource) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(connectionType(), other.connectionType())
                && hasData() == other.hasData() && Objects.equals(data(), other.data())
                && hasOutputSchemas() == other.hasOutputSchemas() && Objects.equals(outputSchemas(), other.outputSchemas());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConnectorDataSource").add("Name", name()).add("ConnectionType", connectionType())
                .add("Data", hasData() ? data() : null).add("OutputSchemas", hasOutputSchemas() ? outputSchemas() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ConnectionType":
            return Optional.ofNullable(clazz.cast(connectionType()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "OutputSchemas":
            return Optional.ofNullable(clazz.cast(outputSchemas()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("ConnectionType", CONNECTION_TYPE_FIELD);
        map.put("Data", DATA_FIELD);
        map.put("OutputSchemas", OUTPUT_SCHEMAS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ConnectorDataSource, T> g) {
        return obj -> g.apply((ConnectorDataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConnectorDataSource> {
        /**
         * <p>
         * The name of this source node.
         * </p>
         * 
         * @param name
         *        The name of this source node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The <code>connectionType</code>, as provided to the underlying Glue library. This node type supports the
         * following connection types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>opensearch</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>azuresql</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>azurecosmos</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>bigquery</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>saphana</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>teradata</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vertica</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param connectionType
         *        The <code>connectionType</code>, as provided to the underlying Glue library. This node type supports
         *        the following connection types: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>opensearch</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>azuresql</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>azurecosmos</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>bigquery</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>saphana</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>teradata</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vertica</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionType(String connectionType);

        /**
         * <p>
         * A map specifying connection options for the node. You can find standard connection options for the
         * corresponding connection type in the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-connect.html"> Connection
         * parameters</a> section of the Glue documentation.
         * </p>
         * 
         * @param data
         *        A map specifying connection options for the node. You can find standard connection options for the
         *        corresponding connection type in the <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-connect.html"> Connection
         *        parameters</a> section of the Glue documentation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(Map<String, String> data);

        /**
         * <p>
         * Specifies the data schema for this source.
         * </p>
         * 
         * @param outputSchemas
         *        Specifies the data schema for this source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputSchemas(Collection<GlueSchema> outputSchemas);

        /**
         * <p>
         * Specifies the data schema for this source.
         * </p>
         * 
         * @param outputSchemas
         *        Specifies the data schema for this source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputSchemas(GlueSchema... outputSchemas);

        /**
         * <p>
         * Specifies the data schema for this source.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.GlueSchema.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.glue.model.GlueSchema#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.GlueSchema.Builder#build()} is called immediately and its
         * result is passed to {@link #outputSchemas(List<GlueSchema>)}.
         * 
         * @param outputSchemas
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.GlueSchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputSchemas(java.util.Collection<GlueSchema>)
         */
        Builder outputSchemas(Consumer<GlueSchema.Builder>... outputSchemas);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String connectionType;

        private Map<String, String> data = DefaultSdkAutoConstructMap.getInstance();

        private List<GlueSchema> outputSchemas = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConnectorDataSource model) {
            name(model.name);
            connectionType(model.connectionType);
            data(model.data);
            outputSchemas(model.outputSchemas);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getConnectionType() {
            return connectionType;
        }

        public final void setConnectionType(String connectionType) {
            this.connectionType = connectionType;
        }

        @Override
        public final Builder connectionType(String connectionType) {
            this.connectionType = connectionType;
            return this;
        }

        public final Map<String, String> getData() {
            if (data instanceof SdkAutoConstructMap) {
                return null;
            }
            return data;
        }

        public final void setData(Map<String, String> data) {
            this.data = ConnectorOptionsCopier.copy(data);
        }

        @Override
        public final Builder data(Map<String, String> data) {
            this.data = ConnectorOptionsCopier.copy(data);
            return this;
        }

        public final List<GlueSchema.Builder> getOutputSchemas() {
            List<GlueSchema.Builder> result = GlueSchemasCopier.copyToBuilder(this.outputSchemas);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputSchemas(Collection<GlueSchema.BuilderImpl> outputSchemas) {
            this.outputSchemas = GlueSchemasCopier.copyFromBuilder(outputSchemas);
        }

        @Override
        public final Builder outputSchemas(Collection<GlueSchema> outputSchemas) {
            this.outputSchemas = GlueSchemasCopier.copy(outputSchemas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputSchemas(GlueSchema... outputSchemas) {
            outputSchemas(Arrays.asList(outputSchemas));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputSchemas(Consumer<GlueSchema.Builder>... outputSchemas) {
            outputSchemas(Stream.of(outputSchemas).map(c -> GlueSchema.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ConnectorDataSource build() {
            return new ConnectorDataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
