/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The evaluation metrics for the find matches algorithm. The quality of your machine learning transform is measured by
 * getting your transform to predict some matches and comparing the results to known matches from the same dataset. The
 * quality metrics are based on a subset of your data, so they are not precise.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FindMatchesMetrics implements SdkPojo, Serializable,
        ToCopyableBuilder<FindMatchesMetrics.Builder, FindMatchesMetrics> {
    private static final SdkField<Double> AREA_UNDER_PR_CURVE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(FindMatchesMetrics::areaUnderPRCurve)).setter(setter(Builder::areaUnderPRCurve))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AreaUnderPRCurve").build()).build();

    private static final SdkField<Double> PRECISION_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(FindMatchesMetrics::precision)).setter(setter(Builder::precision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Precision").build()).build();

    private static final SdkField<Double> RECALL_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(FindMatchesMetrics::recall)).setter(setter(Builder::recall))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recall").build()).build();

    private static final SdkField<Double> F1_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(FindMatchesMetrics::f1)).setter(setter(Builder::f1))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("F1").build()).build();

    private static final SdkField<ConfusionMatrix> CONFUSION_MATRIX_FIELD = SdkField
            .<ConfusionMatrix> builder(MarshallingType.SDK_POJO).getter(getter(FindMatchesMetrics::confusionMatrix))
            .setter(setter(Builder::confusionMatrix)).constructor(ConfusionMatrix::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfusionMatrix").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AREA_UNDER_PR_CURVE_FIELD,
            PRECISION_FIELD, RECALL_FIELD, F1_FIELD, CONFUSION_MATRIX_FIELD));

    private static final long serialVersionUID = 1L;

    private final Double areaUnderPRCurve;

    private final Double precision;

    private final Double recall;

    private final Double f1;

    private final ConfusionMatrix confusionMatrix;

    private FindMatchesMetrics(BuilderImpl builder) {
        this.areaUnderPRCurve = builder.areaUnderPRCurve;
        this.precision = builder.precision;
        this.recall = builder.recall;
        this.f1 = builder.f1;
        this.confusionMatrix = builder.confusionMatrix;
    }

    /**
     * <p>
     * The area under the precision/recall curve (AUPRC) is a single number measuring the overall quality of the
     * transform, that is independent of the choice made for precision vs. recall. Higher values indicate that you have
     * a more attractive precision vs. recall tradeoff.
     * </p>
     * <p>
     * For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and recall</a>
     * in Wikipedia.
     * </p>
     * 
     * @return The area under the precision/recall curve (AUPRC) is a single number measuring the overall quality of the
     *         transform, that is independent of the choice made for precision vs. recall. Higher values indicate that
     *         you have a more attractive precision vs. recall tradeoff.</p>
     *         <p>
     *         For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
     *         recall</a> in Wikipedia.
     */
    public Double areaUnderPRCurve() {
        return areaUnderPRCurve;
    }

    /**
     * <p>
     * The precision metric indicates when often your transform is correct when it predicts a match. Specifically, it
     * measures how well the transform finds true positives from the total true positives possible.
     * </p>
     * <p>
     * For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and recall</a>
     * in Wikipedia.
     * </p>
     * 
     * @return The precision metric indicates when often your transform is correct when it predicts a match.
     *         Specifically, it measures how well the transform finds true positives from the total true positives
     *         possible.</p>
     *         <p>
     *         For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
     *         recall</a> in Wikipedia.
     */
    public Double precision() {
        return precision;
    }

    /**
     * <p>
     * The recall metric indicates that for an actual match, how often your transform predicts the match. Specifically,
     * it measures how well the transform finds true positives from the total records in the source data.
     * </p>
     * <p>
     * For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and recall</a>
     * in Wikipedia.
     * </p>
     * 
     * @return The recall metric indicates that for an actual match, how often your transform predicts the match.
     *         Specifically, it measures how well the transform finds true positives from the total records in the
     *         source data.</p>
     *         <p>
     *         For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
     *         recall</a> in Wikipedia.
     */
    public Double recall() {
        return recall;
    }

    /**
     * <p>
     * The maximum F1 metric indicates the transform's accuracy between 0 and 1, where 1 is the best accuracy.
     * </p>
     * <p>
     * For more information, see <a href="https://en.wikipedia.org/wiki/F1_score">F1 score</a> in Wikipedia.
     * </p>
     * 
     * @return The maximum F1 metric indicates the transform's accuracy between 0 and 1, where 1 is the best
     *         accuracy.</p>
     *         <p>
     *         For more information, see <a href="https://en.wikipedia.org/wiki/F1_score">F1 score</a> in Wikipedia.
     */
    public Double f1() {
        return f1;
    }

    /**
     * <p>
     * The confusion matrix shows you what your transform is predicting accurately and what types of errors it is
     * making.
     * </p>
     * <p>
     * For more information, see <a href="https://en.wikipedia.org/wiki/Confusion_matrix">Confusion matrix</a> in
     * Wikipedia.
     * </p>
     * 
     * @return The confusion matrix shows you what your transform is predicting accurately and what types of errors it
     *         is making.</p>
     *         <p>
     *         For more information, see <a href="https://en.wikipedia.org/wiki/Confusion_matrix">Confusion matrix</a>
     *         in Wikipedia.
     */
    public ConfusionMatrix confusionMatrix() {
        return confusionMatrix;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(areaUnderPRCurve());
        hashCode = 31 * hashCode + Objects.hashCode(precision());
        hashCode = 31 * hashCode + Objects.hashCode(recall());
        hashCode = 31 * hashCode + Objects.hashCode(f1());
        hashCode = 31 * hashCode + Objects.hashCode(confusionMatrix());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FindMatchesMetrics)) {
            return false;
        }
        FindMatchesMetrics other = (FindMatchesMetrics) obj;
        return Objects.equals(areaUnderPRCurve(), other.areaUnderPRCurve()) && Objects.equals(precision(), other.precision())
                && Objects.equals(recall(), other.recall()) && Objects.equals(f1(), other.f1())
                && Objects.equals(confusionMatrix(), other.confusionMatrix());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("FindMatchesMetrics").add("AreaUnderPRCurve", areaUnderPRCurve()).add("Precision", precision())
                .add("Recall", recall()).add("F1", f1()).add("ConfusionMatrix", confusionMatrix()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AreaUnderPRCurve":
            return Optional.ofNullable(clazz.cast(areaUnderPRCurve()));
        case "Precision":
            return Optional.ofNullable(clazz.cast(precision()));
        case "Recall":
            return Optional.ofNullable(clazz.cast(recall()));
        case "F1":
            return Optional.ofNullable(clazz.cast(f1()));
        case "ConfusionMatrix":
            return Optional.ofNullable(clazz.cast(confusionMatrix()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FindMatchesMetrics, T> g) {
        return obj -> g.apply((FindMatchesMetrics) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FindMatchesMetrics> {
        /**
         * <p>
         * The area under the precision/recall curve (AUPRC) is a single number measuring the overall quality of the
         * transform, that is independent of the choice made for precision vs. recall. Higher values indicate that you
         * have a more attractive precision vs. recall tradeoff.
         * </p>
         * <p>
         * For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
         * recall</a> in Wikipedia.
         * </p>
         * 
         * @param areaUnderPRCurve
         *        The area under the precision/recall curve (AUPRC) is a single number measuring the overall quality of
         *        the transform, that is independent of the choice made for precision vs. recall. Higher values indicate
         *        that you have a more attractive precision vs. recall tradeoff.</p>
         *        <p>
         *        For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
         *        recall</a> in Wikipedia.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder areaUnderPRCurve(Double areaUnderPRCurve);

        /**
         * <p>
         * The precision metric indicates when often your transform is correct when it predicts a match. Specifically,
         * it measures how well the transform finds true positives from the total true positives possible.
         * </p>
         * <p>
         * For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
         * recall</a> in Wikipedia.
         * </p>
         * 
         * @param precision
         *        The precision metric indicates when often your transform is correct when it predicts a match.
         *        Specifically, it measures how well the transform finds true positives from the total true positives
         *        possible.</p>
         *        <p>
         *        For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
         *        recall</a> in Wikipedia.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder precision(Double precision);

        /**
         * <p>
         * The recall metric indicates that for an actual match, how often your transform predicts the match.
         * Specifically, it measures how well the transform finds true positives from the total records in the source
         * data.
         * </p>
         * <p>
         * For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
         * recall</a> in Wikipedia.
         * </p>
         * 
         * @param recall
         *        The recall metric indicates that for an actual match, how often your transform predicts the match.
         *        Specifically, it measures how well the transform finds true positives from the total records in the
         *        source data.</p>
         *        <p>
         *        For more information, see <a href="https://en.wikipedia.org/wiki/Precision_and_recall">Precision and
         *        recall</a> in Wikipedia.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recall(Double recall);

        /**
         * <p>
         * The maximum F1 metric indicates the transform's accuracy between 0 and 1, where 1 is the best accuracy.
         * </p>
         * <p>
         * For more information, see <a href="https://en.wikipedia.org/wiki/F1_score">F1 score</a> in Wikipedia.
         * </p>
         * 
         * @param f1
         *        The maximum F1 metric indicates the transform's accuracy between 0 and 1, where 1 is the best
         *        accuracy.</p>
         *        <p>
         *        For more information, see <a href="https://en.wikipedia.org/wiki/F1_score">F1 score</a> in Wikipedia.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder f1(Double f1);

        /**
         * <p>
         * The confusion matrix shows you what your transform is predicting accurately and what types of errors it is
         * making.
         * </p>
         * <p>
         * For more information, see <a href="https://en.wikipedia.org/wiki/Confusion_matrix">Confusion matrix</a> in
         * Wikipedia.
         * </p>
         * 
         * @param confusionMatrix
         *        The confusion matrix shows you what your transform is predicting accurately and what types of errors
         *        it is making.</p>
         *        <p>
         *        For more information, see <a href="https://en.wikipedia.org/wiki/Confusion_matrix">Confusion
         *        matrix</a> in Wikipedia.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confusionMatrix(ConfusionMatrix confusionMatrix);

        /**
         * <p>
         * The confusion matrix shows you what your transform is predicting accurately and what types of errors it is
         * making.
         * </p>
         * <p>
         * For more information, see <a href="https://en.wikipedia.org/wiki/Confusion_matrix">Confusion matrix</a> in
         * Wikipedia.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfusionMatrix.Builder} avoiding the need to
         * create one manually via {@link ConfusionMatrix#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfusionMatrix.Builder#build()} is called immediately and its
         * result is passed to {@link #confusionMatrix(ConfusionMatrix)}.
         * 
         * @param confusionMatrix
         *        a consumer that will call methods on {@link ConfusionMatrix.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #confusionMatrix(ConfusionMatrix)
         */
        default Builder confusionMatrix(Consumer<ConfusionMatrix.Builder> confusionMatrix) {
            return confusionMatrix(ConfusionMatrix.builder().applyMutation(confusionMatrix).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Double areaUnderPRCurve;

        private Double precision;

        private Double recall;

        private Double f1;

        private ConfusionMatrix confusionMatrix;

        private BuilderImpl() {
        }

        private BuilderImpl(FindMatchesMetrics model) {
            areaUnderPRCurve(model.areaUnderPRCurve);
            precision(model.precision);
            recall(model.recall);
            f1(model.f1);
            confusionMatrix(model.confusionMatrix);
        }

        public final Double getAreaUnderPRCurve() {
            return areaUnderPRCurve;
        }

        @Override
        public final Builder areaUnderPRCurve(Double areaUnderPRCurve) {
            this.areaUnderPRCurve = areaUnderPRCurve;
            return this;
        }

        public final void setAreaUnderPRCurve(Double areaUnderPRCurve) {
            this.areaUnderPRCurve = areaUnderPRCurve;
        }

        public final Double getPrecision() {
            return precision;
        }

        @Override
        public final Builder precision(Double precision) {
            this.precision = precision;
            return this;
        }

        public final void setPrecision(Double precision) {
            this.precision = precision;
        }

        public final Double getRecall() {
            return recall;
        }

        @Override
        public final Builder recall(Double recall) {
            this.recall = recall;
            return this;
        }

        public final void setRecall(Double recall) {
            this.recall = recall;
        }

        public final Double getF1() {
            return f1;
        }

        @Override
        public final Builder f1(Double f1) {
            this.f1 = f1;
            return this;
        }

        public final void setF1(Double f1) {
            this.f1 = f1;
        }

        public final ConfusionMatrix.Builder getConfusionMatrix() {
            return confusionMatrix != null ? confusionMatrix.toBuilder() : null;
        }

        @Override
        public final Builder confusionMatrix(ConfusionMatrix confusionMatrix) {
            this.confusionMatrix = confusionMatrix;
            return this;
        }

        public final void setConfusionMatrix(ConfusionMatrix.BuilderImpl confusionMatrix) {
            this.confusionMatrix = confusionMatrix != null ? confusionMatrix.build() : null;
        }

        @Override
        public FindMatchesMetrics build() {
            return new FindMatchesMetrics(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
