/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A workflow run is an execution of a workflow providing all the runtime information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkflowRun implements SdkPojo, Serializable, ToCopyableBuilder<WorkflowRun.Builder, WorkflowRun> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkflowRun::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> WORKFLOW_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkflowRun::workflowRunId)).setter(setter(Builder::workflowRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkflowRunId").build()).build();

    private static final SdkField<Map<String, String>> WORKFLOW_RUN_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(WorkflowRun::workflowRunProperties))
            .setter(setter(Builder::workflowRunProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkflowRunProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> STARTED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(WorkflowRun::startedOn)).setter(setter(Builder::startedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedOn").build()).build();

    private static final SdkField<Instant> COMPLETED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(WorkflowRun::completedOn)).setter(setter(Builder::completedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletedOn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkflowRun::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<WorkflowRunStatistics> STATISTICS_FIELD = SdkField
            .<WorkflowRunStatistics> builder(MarshallingType.SDK_POJO).getter(getter(WorkflowRun::statistics))
            .setter(setter(Builder::statistics)).constructor(WorkflowRunStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistics").build()).build();

    private static final SdkField<WorkflowGraph> GRAPH_FIELD = SdkField.<WorkflowGraph> builder(MarshallingType.SDK_POJO)
            .getter(getter(WorkflowRun::graph)).setter(setter(Builder::graph)).constructor(WorkflowGraph::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Graph").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            WORKFLOW_RUN_ID_FIELD, WORKFLOW_RUN_PROPERTIES_FIELD, STARTED_ON_FIELD, COMPLETED_ON_FIELD, STATUS_FIELD,
            STATISTICS_FIELD, GRAPH_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String workflowRunId;

    private final Map<String, String> workflowRunProperties;

    private final Instant startedOn;

    private final Instant completedOn;

    private final String status;

    private final WorkflowRunStatistics statistics;

    private final WorkflowGraph graph;

    private WorkflowRun(BuilderImpl builder) {
        this.name = builder.name;
        this.workflowRunId = builder.workflowRunId;
        this.workflowRunProperties = builder.workflowRunProperties;
        this.startedOn = builder.startedOn;
        this.completedOn = builder.completedOn;
        this.status = builder.status;
        this.statistics = builder.statistics;
        this.graph = builder.graph;
    }

    /**
     * <p>
     * Name of the workflow which was executed.
     * </p>
     * 
     * @return Name of the workflow which was executed.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ID of this workflow run.
     * </p>
     * 
     * @return The ID of this workflow run.
     */
    public String workflowRunId() {
        return workflowRunId;
    }

    /**
     * Returns true if the WorkflowRunProperties property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasWorkflowRunProperties() {
        return workflowRunProperties != null && !(workflowRunProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The workflow run properties which were set during the run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasWorkflowRunProperties()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The workflow run properties which were set during the run.
     */
    public Map<String, String> workflowRunProperties() {
        return workflowRunProperties;
    }

    /**
     * <p>
     * The date and time when the workflow run was started.
     * </p>
     * 
     * @return The date and time when the workflow run was started.
     */
    public Instant startedOn() {
        return startedOn;
    }

    /**
     * <p>
     * The date and time when the workflow run completed.
     * </p>
     * 
     * @return The date and time when the workflow run completed.
     */
    public Instant completedOn() {
        return completedOn;
    }

    /**
     * <p>
     * The status of the workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkflowRunStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the workflow run.
     * @see WorkflowRunStatus
     */
    public WorkflowRunStatus status() {
        return WorkflowRunStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkflowRunStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the workflow run.
     * @see WorkflowRunStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The statistics of the run.
     * </p>
     * 
     * @return The statistics of the run.
     */
    public WorkflowRunStatistics statistics() {
        return statistics;
    }

    /**
     * <p>
     * The graph representing all the AWS Glue components that belong to the workflow as nodes and directed connections
     * between them as edges.
     * </p>
     * 
     * @return The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
     *         connections between them as edges.
     */
    public WorkflowGraph graph() {
        return graph;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(workflowRunId());
        hashCode = 31 * hashCode + Objects.hashCode(workflowRunProperties());
        hashCode = 31 * hashCode + Objects.hashCode(startedOn());
        hashCode = 31 * hashCode + Objects.hashCode(completedOn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statistics());
        hashCode = 31 * hashCode + Objects.hashCode(graph());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkflowRun)) {
            return false;
        }
        WorkflowRun other = (WorkflowRun) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(workflowRunId(), other.workflowRunId())
                && Objects.equals(workflowRunProperties(), other.workflowRunProperties())
                && Objects.equals(startedOn(), other.startedOn()) && Objects.equals(completedOn(), other.completedOn())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(statistics(), other.statistics())
                && Objects.equals(graph(), other.graph());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("WorkflowRun").add("Name", name()).add("WorkflowRunId", workflowRunId())
                .add("WorkflowRunProperties", workflowRunProperties()).add("StartedOn", startedOn())
                .add("CompletedOn", completedOn()).add("Status", statusAsString()).add("Statistics", statistics())
                .add("Graph", graph()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "WorkflowRunId":
            return Optional.ofNullable(clazz.cast(workflowRunId()));
        case "WorkflowRunProperties":
            return Optional.ofNullable(clazz.cast(workflowRunProperties()));
        case "StartedOn":
            return Optional.ofNullable(clazz.cast(startedOn()));
        case "CompletedOn":
            return Optional.ofNullable(clazz.cast(completedOn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Statistics":
            return Optional.ofNullable(clazz.cast(statistics()));
        case "Graph":
            return Optional.ofNullable(clazz.cast(graph()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkflowRun, T> g) {
        return obj -> g.apply((WorkflowRun) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkflowRun> {
        /**
         * <p>
         * Name of the workflow which was executed.
         * </p>
         * 
         * @param name
         *        Name of the workflow which was executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of this workflow run.
         * </p>
         * 
         * @param workflowRunId
         *        The ID of this workflow run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowRunId(String workflowRunId);

        /**
         * <p>
         * The workflow run properties which were set during the run.
         * </p>
         * 
         * @param workflowRunProperties
         *        The workflow run properties which were set during the run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowRunProperties(Map<String, String> workflowRunProperties);

        /**
         * <p>
         * The date and time when the workflow run was started.
         * </p>
         * 
         * @param startedOn
         *        The date and time when the workflow run was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedOn(Instant startedOn);

        /**
         * <p>
         * The date and time when the workflow run completed.
         * </p>
         * 
         * @param completedOn
         *        The date and time when the workflow run completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedOn(Instant completedOn);

        /**
         * <p>
         * The status of the workflow run.
         * </p>
         * 
         * @param status
         *        The status of the workflow run.
         * @see WorkflowRunStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkflowRunStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the workflow run.
         * </p>
         * 
         * @param status
         *        The status of the workflow run.
         * @see WorkflowRunStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkflowRunStatus
         */
        Builder status(WorkflowRunStatus status);

        /**
         * <p>
         * The statistics of the run.
         * </p>
         * 
         * @param statistics
         *        The statistics of the run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(WorkflowRunStatistics statistics);

        /**
         * <p>
         * The statistics of the run.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowRunStatistics.Builder} avoiding the need
         * to create one manually via {@link WorkflowRunStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowRunStatistics.Builder#build()} is called immediately and
         * its result is passed to {@link #statistics(WorkflowRunStatistics)}.
         * 
         * @param statistics
         *        a consumer that will call methods on {@link WorkflowRunStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statistics(WorkflowRunStatistics)
         */
        default Builder statistics(Consumer<WorkflowRunStatistics.Builder> statistics) {
            return statistics(WorkflowRunStatistics.builder().applyMutation(statistics).build());
        }

        /**
         * <p>
         * The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
         * connections between them as edges.
         * </p>
         * 
         * @param graph
         *        The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
         *        connections between them as edges.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder graph(WorkflowGraph graph);

        /**
         * <p>
         * The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
         * connections between them as edges.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowGraph.Builder} avoiding the need to
         * create one manually via {@link WorkflowGraph#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowGraph.Builder#build()} is called immediately and its
         * result is passed to {@link #graph(WorkflowGraph)}.
         * 
         * @param graph
         *        a consumer that will call methods on {@link WorkflowGraph.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #graph(WorkflowGraph)
         */
        default Builder graph(Consumer<WorkflowGraph.Builder> graph) {
            return graph(WorkflowGraph.builder().applyMutation(graph).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String workflowRunId;

        private Map<String, String> workflowRunProperties = DefaultSdkAutoConstructMap.getInstance();

        private Instant startedOn;

        private Instant completedOn;

        private String status;

        private WorkflowRunStatistics statistics;

        private WorkflowGraph graph;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkflowRun model) {
            name(model.name);
            workflowRunId(model.workflowRunId);
            workflowRunProperties(model.workflowRunProperties);
            startedOn(model.startedOn);
            completedOn(model.completedOn);
            status(model.status);
            statistics(model.statistics);
            graph(model.graph);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getWorkflowRunId() {
            return workflowRunId;
        }

        @Override
        public final Builder workflowRunId(String workflowRunId) {
            this.workflowRunId = workflowRunId;
            return this;
        }

        public final void setWorkflowRunId(String workflowRunId) {
            this.workflowRunId = workflowRunId;
        }

        public final Map<String, String> getWorkflowRunProperties() {
            return workflowRunProperties;
        }

        @Override
        public final Builder workflowRunProperties(Map<String, String> workflowRunProperties) {
            this.workflowRunProperties = WorkflowRunPropertiesCopier.copy(workflowRunProperties);
            return this;
        }

        public final void setWorkflowRunProperties(Map<String, String> workflowRunProperties) {
            this.workflowRunProperties = WorkflowRunPropertiesCopier.copy(workflowRunProperties);
        }

        public final Instant getStartedOn() {
            return startedOn;
        }

        @Override
        public final Builder startedOn(Instant startedOn) {
            this.startedOn = startedOn;
            return this;
        }

        public final void setStartedOn(Instant startedOn) {
            this.startedOn = startedOn;
        }

        public final Instant getCompletedOn() {
            return completedOn;
        }

        @Override
        public final Builder completedOn(Instant completedOn) {
            this.completedOn = completedOn;
            return this;
        }

        public final void setCompletedOn(Instant completedOn) {
            this.completedOn = completedOn;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(WorkflowRunStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final WorkflowRunStatistics.Builder getStatistics() {
            return statistics != null ? statistics.toBuilder() : null;
        }

        @Override
        public final Builder statistics(WorkflowRunStatistics statistics) {
            this.statistics = statistics;
            return this;
        }

        public final void setStatistics(WorkflowRunStatistics.BuilderImpl statistics) {
            this.statistics = statistics != null ? statistics.build() : null;
        }

        public final WorkflowGraph.Builder getGraph() {
            return graph != null ? graph.toBuilder() : null;
        }

        @Override
        public final Builder graph(WorkflowGraph graph) {
            this.graph = graph;
            return this;
        }

        public final void setGraph(WorkflowGraph.BuilderImpl graph) {
            this.graph = graph != null ? graph.build() : null;
        }

        @Override
        public WorkflowRun build() {
            return new WorkflowRun(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
