/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glacier.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the location in Amazon S3 where the select job results are stored.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Location implements SdkPojo, Serializable, ToCopyableBuilder<S3Location.Builder, S3Location> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Location::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Location::prefix)).setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefix").build()).build();

    private static final SdkField<Encryption> ENCRYPTION_FIELD = SdkField.<Encryption> builder(MarshallingType.SDK_POJO)
            .getter(getter(S3Location::encryption)).setter(setter(Builder::encryption)).constructor(Encryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encryption").build()).build();

    private static final SdkField<String> CANNED_ACL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Location::cannedACLAsString)).setter(setter(Builder::cannedACL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CannedACL").build()).build();

    private static final SdkField<List<Grant>> ACCESS_CONTROL_LIST_FIELD = SdkField
            .<List<Grant>> builder(MarshallingType.LIST)
            .getter(getter(S3Location::accessControlList))
            .setter(setter(Builder::accessControlList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Grant> builder(MarshallingType.SDK_POJO)
                                            .constructor(Grant::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGGING_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(S3Location::tagging))
            .setter(setter(Builder::tagging))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tagging").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> USER_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(S3Location::userMetadata))
            .setter(setter(Builder::userMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STORAGE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Location::storageClassAsString)).setter(setter(Builder::storageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageClass").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            PREFIX_FIELD, ENCRYPTION_FIELD, CANNED_ACL_FIELD, ACCESS_CONTROL_LIST_FIELD, TAGGING_FIELD, USER_METADATA_FIELD,
            STORAGE_CLASS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketName;

    private final String prefix;

    private final Encryption encryption;

    private final String cannedACL;

    private final List<Grant> accessControlList;

    private final Map<String, String> tagging;

    private final Map<String, String> userMetadata;

    private final String storageClass;

    private S3Location(BuilderImpl builder) {
        this.bucketName = builder.bucketName;
        this.prefix = builder.prefix;
        this.encryption = builder.encryption;
        this.cannedACL = builder.cannedACL;
        this.accessControlList = builder.accessControlList;
        this.tagging = builder.tagging;
        this.userMetadata = builder.userMetadata;
        this.storageClass = builder.storageClass;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket where the job results are stored.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket where the job results are stored.
     */
    public String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The prefix that is prepended to the results for this request.
     * </p>
     * 
     * @return The prefix that is prepended to the results for this request.
     */
    public String prefix() {
        return prefix;
    }

    /**
     * <p>
     * Contains information about the encryption used to store the job results in Amazon S3.
     * </p>
     * 
     * @return Contains information about the encryption used to store the job results in Amazon S3.
     */
    public Encryption encryption() {
        return encryption;
    }

    /**
     * <p>
     * The canned access control list (ACL) to apply to the job results.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #cannedACL} will
     * return {@link CannedACL#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #cannedACLAsString}.
     * </p>
     * 
     * @return The canned access control list (ACL) to apply to the job results.
     * @see CannedACL
     */
    public CannedACL cannedACL() {
        return CannedACL.fromValue(cannedACL);
    }

    /**
     * <p>
     * The canned access control list (ACL) to apply to the job results.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #cannedACL} will
     * return {@link CannedACL#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #cannedACLAsString}.
     * </p>
     * 
     * @return The canned access control list (ACL) to apply to the job results.
     * @see CannedACL
     */
    public String cannedACLAsString() {
        return cannedACL;
    }

    /**
     * Returns true if the AccessControlList property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAccessControlList() {
        return accessControlList != null && !(accessControlList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of grants that control access to the staged results.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAccessControlList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of grants that control access to the staged results.
     */
    public List<Grant> accessControlList() {
        return accessControlList;
    }

    /**
     * Returns true if the Tagging property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTagging() {
        return tagging != null && !(tagging instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tag-set that is applied to the job results.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTagging()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tag-set that is applied to the job results.
     */
    public Map<String, String> tagging() {
        return tagging;
    }

    /**
     * Returns true if the UserMetadata property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasUserMetadata() {
        return userMetadata != null && !(userMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of metadata to store with the job results in Amazon S3.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUserMetadata()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of metadata to store with the job results in Amazon S3.
     */
    public Map<String, String> userMetadata() {
        return userMetadata;
    }

    /**
     * <p>
     * The storage class used to store the job results.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return The storage class used to store the job results.
     * @see StorageClass
     */
    public StorageClass storageClass() {
        return StorageClass.fromValue(storageClass);
    }

    /**
     * <p>
     * The storage class used to store the job results.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return The storage class used to store the job results.
     * @see StorageClass
     */
    public String storageClassAsString() {
        return storageClass;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(encryption());
        hashCode = 31 * hashCode + Objects.hashCode(cannedACLAsString());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlList());
        hashCode = 31 * hashCode + Objects.hashCode(tagging());
        hashCode = 31 * hashCode + Objects.hashCode(userMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(storageClassAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Location)) {
            return false;
        }
        S3Location other = (S3Location) obj;
        return Objects.equals(bucketName(), other.bucketName()) && Objects.equals(prefix(), other.prefix())
                && Objects.equals(encryption(), other.encryption())
                && Objects.equals(cannedACLAsString(), other.cannedACLAsString())
                && Objects.equals(accessControlList(), other.accessControlList()) && Objects.equals(tagging(), other.tagging())
                && Objects.equals(userMetadata(), other.userMetadata())
                && Objects.equals(storageClassAsString(), other.storageClassAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3Location").add("BucketName", bucketName()).add("Prefix", prefix())
                .add("Encryption", encryption()).add("CannedACL", cannedACLAsString())
                .add("AccessControlList", accessControlList()).add("Tagging", tagging()).add("UserMetadata", userMetadata())
                .add("StorageClass", storageClassAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "Prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "Encryption":
            return Optional.ofNullable(clazz.cast(encryption()));
        case "CannedACL":
            return Optional.ofNullable(clazz.cast(cannedACLAsString()));
        case "AccessControlList":
            return Optional.ofNullable(clazz.cast(accessControlList()));
        case "Tagging":
            return Optional.ofNullable(clazz.cast(tagging()));
        case "UserMetadata":
            return Optional.ofNullable(clazz.cast(userMetadata()));
        case "StorageClass":
            return Optional.ofNullable(clazz.cast(storageClassAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Location, T> g) {
        return obj -> g.apply((S3Location) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Location> {
        /**
         * <p>
         * The name of the Amazon S3 bucket where the job results are stored.
         * </p>
         * 
         * @param bucketName
         *        The name of the Amazon S3 bucket where the job results are stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The prefix that is prepended to the results for this request.
         * </p>
         * 
         * @param prefix
         *        The prefix that is prepended to the results for this request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * Contains information about the encryption used to store the job results in Amazon S3.
         * </p>
         * 
         * @param encryption
         *        Contains information about the encryption used to store the job results in Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryption(Encryption encryption);

        /**
         * <p>
         * Contains information about the encryption used to store the job results in Amazon S3.
         * </p>
         * This is a convenience that creates an instance of the {@link Encryption.Builder} avoiding the need to create
         * one manually via {@link Encryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link Encryption.Builder#build()} is called immediately and its result
         * is passed to {@link #encryption(Encryption)}.
         * 
         * @param encryption
         *        a consumer that will call methods on {@link Encryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryption(Encryption)
         */
        default Builder encryption(Consumer<Encryption.Builder> encryption) {
            return encryption(Encryption.builder().applyMutation(encryption).build());
        }

        /**
         * <p>
         * The canned access control list (ACL) to apply to the job results.
         * </p>
         * 
         * @param cannedACL
         *        The canned access control list (ACL) to apply to the job results.
         * @see CannedACL
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CannedACL
         */
        Builder cannedACL(String cannedACL);

        /**
         * <p>
         * The canned access control list (ACL) to apply to the job results.
         * </p>
         * 
         * @param cannedACL
         *        The canned access control list (ACL) to apply to the job results.
         * @see CannedACL
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CannedACL
         */
        Builder cannedACL(CannedACL cannedACL);

        /**
         * <p>
         * A list of grants that control access to the staged results.
         * </p>
         * 
         * @param accessControlList
         *        A list of grants that control access to the staged results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlList(Collection<Grant> accessControlList);

        /**
         * <p>
         * A list of grants that control access to the staged results.
         * </p>
         * 
         * @param accessControlList
         *        A list of grants that control access to the staged results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlList(Grant... accessControlList);

        /**
         * <p>
         * A list of grants that control access to the staged results.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Grant>.Builder} avoiding the need to create
         * one manually via {@link List<Grant>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Grant>.Builder#build()} is called immediately and its result
         * is passed to {@link #accessControlList(List<Grant>)}.
         * 
         * @param accessControlList
         *        a consumer that will call methods on {@link List<Grant>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlList(List<Grant>)
         */
        Builder accessControlList(Consumer<Grant.Builder>... accessControlList);

        /**
         * <p>
         * The tag-set that is applied to the job results.
         * </p>
         * 
         * @param tagging
         *        The tag-set that is applied to the job results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagging(Map<String, String> tagging);

        /**
         * <p>
         * A map of metadata to store with the job results in Amazon S3.
         * </p>
         * 
         * @param userMetadata
         *        A map of metadata to store with the job results in Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userMetadata(Map<String, String> userMetadata);

        /**
         * <p>
         * The storage class used to store the job results.
         * </p>
         * 
         * @param storageClass
         *        The storage class used to store the job results.
         * @see StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageClass
         */
        Builder storageClass(String storageClass);

        /**
         * <p>
         * The storage class used to store the job results.
         * </p>
         * 
         * @param storageClass
         *        The storage class used to store the job results.
         * @see StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageClass
         */
        Builder storageClass(StorageClass storageClass);
    }

    static final class BuilderImpl implements Builder {
        private String bucketName;

        private String prefix;

        private Encryption encryption;

        private String cannedACL;

        private List<Grant> accessControlList = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tagging = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> userMetadata = DefaultSdkAutoConstructMap.getInstance();

        private String storageClass;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Location model) {
            bucketName(model.bucketName);
            prefix(model.prefix);
            encryption(model.encryption);
            cannedACL(model.cannedACL);
            accessControlList(model.accessControlList);
            tagging(model.tagging);
            userMetadata(model.userMetadata);
            storageClass(model.storageClass);
        }

        public final String getBucketName() {
            return bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        public final String getPrefix() {
            return prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        public final Encryption.Builder getEncryption() {
            return encryption != null ? encryption.toBuilder() : null;
        }

        @Override
        public final Builder encryption(Encryption encryption) {
            this.encryption = encryption;
            return this;
        }

        public final void setEncryption(Encryption.BuilderImpl encryption) {
            this.encryption = encryption != null ? encryption.build() : null;
        }

        public final String getCannedACL() {
            return cannedACL;
        }

        @Override
        public final Builder cannedACL(String cannedACL) {
            this.cannedACL = cannedACL;
            return this;
        }

        @Override
        public final Builder cannedACL(CannedACL cannedACL) {
            this.cannedACL(cannedACL == null ? null : cannedACL.toString());
            return this;
        }

        public final void setCannedACL(String cannedACL) {
            this.cannedACL = cannedACL;
        }

        public final Collection<Grant.Builder> getAccessControlList() {
            return accessControlList != null ? accessControlList.stream().map(Grant::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder accessControlList(Collection<Grant> accessControlList) {
            this.accessControlList = AccessControlPolicyListCopier.copy(accessControlList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessControlList(Grant... accessControlList) {
            accessControlList(Arrays.asList(accessControlList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessControlList(Consumer<Grant.Builder>... accessControlList) {
            accessControlList(Stream.of(accessControlList).map(c -> Grant.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAccessControlList(Collection<Grant.BuilderImpl> accessControlList) {
            this.accessControlList = AccessControlPolicyListCopier.copyFromBuilder(accessControlList);
        }

        public final Map<String, String> getTagging() {
            return tagging;
        }

        @Override
        public final Builder tagging(Map<String, String> tagging) {
            this.tagging = _hashmapCopier.copy(tagging);
            return this;
        }

        public final void setTagging(Map<String, String> tagging) {
            this.tagging = _hashmapCopier.copy(tagging);
        }

        public final Map<String, String> getUserMetadata() {
            return userMetadata;
        }

        @Override
        public final Builder userMetadata(Map<String, String> userMetadata) {
            this.userMetadata = _hashmapCopier.copy(userMetadata);
            return this;
        }

        public final void setUserMetadata(Map<String, String> userMetadata) {
            this.userMetadata = _hashmapCopier.copy(userMetadata);
        }

        public final String getStorageClass() {
            return storageClass;
        }

        @Override
        public final Builder storageClass(String storageClass) {
            this.storageClass = storageClass;
            return this;
        }

        @Override
        public final Builder storageClass(StorageClass storageClass) {
            this.storageClass(storageClass == null ? null : storageClass.toString());
            return this;
        }

        public final void setStorageClass(String storageClass) {
            this.storageClass = storageClass;
        }

        @Override
        public S3Location build() {
            return new S3Location(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
