/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container's dependency on another container in the same container group. The dependency impacts how the dependent
 * container is able to start or shut down based the status of the other container.
 * </p>
 * <p>
 * For example, <i>ContainerA</i> is configured with the following dependency: a <code>START</code> dependency on
 * <i>ContainerB</i>. This means that <i>ContainerA</i> can't start until <i>ContainerB</i> has started. It also means
 * that <i>ContainerA</i> must shut down before <i>ContainerB</i>.
 * </p>
 * <p>
 * <b>Part of:</b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_GameServerContainerDefinition.html"
 * >GameServerContainerDefinition</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_GameServerContainerDefinitionInput.html"
 * >GameServerContainerDefinitionInput</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_SupportContainerDefinition.html"
 * >SupportContainerDefinition</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_SupportContainerDefinitionInput.html"
 * >SupportContainerDefinitionInput</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDependency implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerDependency.Builder, ContainerDependency> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerName").getter(getter(ContainerDependency::containerName))
            .setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerName").build()).build();

    private static final SdkField<String> CONDITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Condition").getter(getter(ContainerDependency::conditionAsString)).setter(setter(Builder::condition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Condition").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_NAME_FIELD,
            CONDITION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String containerName;

    private final String condition;

    private ContainerDependency(BuilderImpl builder) {
        this.containerName = builder.containerName;
        this.condition = builder.condition;
    }

    /**
     * <p>
     * A descriptive label for the container definition that this container depends on.
     * </p>
     * 
     * @return A descriptive label for the container definition that this container depends on.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * <p>
     * The condition that the dependency container must reach before the dependent container can start. Valid conditions
     * include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * START - The dependency container must have started.
     * </p>
     * </li>
     * <li>
     * <p>
     * COMPLETE - The dependency container has run to completion (exits). Use this condition with nonessential
     * containers, such as those that run a script and then exit. The dependency container can't be an essential
     * container.
     * </p>
     * </li>
     * <li>
     * <p>
     * SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency container
     * can't be an essential container.
     * </p>
     * </li>
     * <li>
     * <p>
     * HEALTHY - The dependency container has passed its Docker health check. Use this condition with dependency
     * containers that have health checks configured. This condition is confirmed at container group startup only.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #condition} will
     * return {@link ContainerDependencyCondition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #conditionAsString}.
     * </p>
     * 
     * @return The condition that the dependency container must reach before the dependent container can start. Valid
     *         conditions include: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         START - The dependency container must have started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         COMPLETE - The dependency container has run to completion (exits). Use this condition with nonessential
     *         containers, such as those that run a script and then exit. The dependency container can't be an essential
     *         container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency
     *         container can't be an essential container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         HEALTHY - The dependency container has passed its Docker health check. Use this condition with dependency
     *         containers that have health checks configured. This condition is confirmed at container group startup
     *         only.
     *         </p>
     *         </li>
     * @see ContainerDependencyCondition
     */
    public final ContainerDependencyCondition condition() {
        return ContainerDependencyCondition.fromValue(condition);
    }

    /**
     * <p>
     * The condition that the dependency container must reach before the dependent container can start. Valid conditions
     * include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * START - The dependency container must have started.
     * </p>
     * </li>
     * <li>
     * <p>
     * COMPLETE - The dependency container has run to completion (exits). Use this condition with nonessential
     * containers, such as those that run a script and then exit. The dependency container can't be an essential
     * container.
     * </p>
     * </li>
     * <li>
     * <p>
     * SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency container
     * can't be an essential container.
     * </p>
     * </li>
     * <li>
     * <p>
     * HEALTHY - The dependency container has passed its Docker health check. Use this condition with dependency
     * containers that have health checks configured. This condition is confirmed at container group startup only.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #condition} will
     * return {@link ContainerDependencyCondition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #conditionAsString}.
     * </p>
     * 
     * @return The condition that the dependency container must reach before the dependent container can start. Valid
     *         conditions include: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         START - The dependency container must have started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         COMPLETE - The dependency container has run to completion (exits). Use this condition with nonessential
     *         containers, such as those that run a script and then exit. The dependency container can't be an essential
     *         container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency
     *         container can't be an essential container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         HEALTHY - The dependency container has passed its Docker health check. Use this condition with dependency
     *         containers that have health checks configured. This condition is confirmed at container group startup
     *         only.
     *         </p>
     *         </li>
     * @see ContainerDependencyCondition
     */
    public final String conditionAsString() {
        return condition;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(conditionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDependency)) {
            return false;
        }
        ContainerDependency other = (ContainerDependency) obj;
        return Objects.equals(containerName(), other.containerName())
                && Objects.equals(conditionAsString(), other.conditionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDependency").add("ContainerName", containerName())
                .add("Condition", conditionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "Condition":
            return Optional.ofNullable(clazz.cast(conditionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ContainerName", CONTAINER_NAME_FIELD);
        map.put("Condition", CONDITION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ContainerDependency, T> g) {
        return obj -> g.apply((ContainerDependency) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDependency> {
        /**
         * <p>
         * A descriptive label for the container definition that this container depends on.
         * </p>
         * 
         * @param containerName
         *        A descriptive label for the container definition that this container depends on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * The condition that the dependency container must reach before the dependent container can start. Valid
         * conditions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * START - The dependency container must have started.
         * </p>
         * </li>
         * <li>
         * <p>
         * COMPLETE - The dependency container has run to completion (exits). Use this condition with nonessential
         * containers, such as those that run a script and then exit. The dependency container can't be an essential
         * container.
         * </p>
         * </li>
         * <li>
         * <p>
         * SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency
         * container can't be an essential container.
         * </p>
         * </li>
         * <li>
         * <p>
         * HEALTHY - The dependency container has passed its Docker health check. Use this condition with dependency
         * containers that have health checks configured. This condition is confirmed at container group startup only.
         * </p>
         * </li>
         * </ul>
         * 
         * @param condition
         *        The condition that the dependency container must reach before the dependent container can start. Valid
         *        conditions include: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        START - The dependency container must have started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        COMPLETE - The dependency container has run to completion (exits). Use this condition with
         *        nonessential containers, such as those that run a script and then exit. The dependency container can't
         *        be an essential container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency
         *        container can't be an essential container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        HEALTHY - The dependency container has passed its Docker health check. Use this condition with
         *        dependency containers that have health checks configured. This condition is confirmed at container
         *        group startup only.
         *        </p>
         *        </li>
         * @see ContainerDependencyCondition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerDependencyCondition
         */
        Builder condition(String condition);

        /**
         * <p>
         * The condition that the dependency container must reach before the dependent container can start. Valid
         * conditions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * START - The dependency container must have started.
         * </p>
         * </li>
         * <li>
         * <p>
         * COMPLETE - The dependency container has run to completion (exits). Use this condition with nonessential
         * containers, such as those that run a script and then exit. The dependency container can't be an essential
         * container.
         * </p>
         * </li>
         * <li>
         * <p>
         * SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency
         * container can't be an essential container.
         * </p>
         * </li>
         * <li>
         * <p>
         * HEALTHY - The dependency container has passed its Docker health check. Use this condition with dependency
         * containers that have health checks configured. This condition is confirmed at container group startup only.
         * </p>
         * </li>
         * </ul>
         * 
         * @param condition
         *        The condition that the dependency container must reach before the dependent container can start. Valid
         *        conditions include: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        START - The dependency container must have started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        COMPLETE - The dependency container has run to completion (exits). Use this condition with
         *        nonessential containers, such as those that run a script and then exit. The dependency container can't
         *        be an essential container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SUCCESS - The dependency container has run to completion and exited with a zero status. The dependency
         *        container can't be an essential container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        HEALTHY - The dependency container has passed its Docker health check. Use this condition with
         *        dependency containers that have health checks configured. This condition is confirmed at container
         *        group startup only.
         *        </p>
         *        </li>
         * @see ContainerDependencyCondition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerDependencyCondition
         */
        Builder condition(ContainerDependencyCondition condition);
    }

    static final class BuilderImpl implements Builder {
        private String containerName;

        private String condition;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDependency model) {
            containerName(model.containerName);
            condition(model.condition);
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final String getCondition() {
            return condition;
        }

        public final void setCondition(String condition) {
            this.condition = condition;
        }

        @Override
        public final Builder condition(String condition) {
            this.condition = condition;
            return this;
        }

        @Override
        public final Builder condition(ContainerDependencyCondition condition) {
            this.condition(condition == null ? null : condition.toString());
            return this;
        }

        @Override
        public ContainerDependency build() {
            return new ContainerDependency(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
