/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Log entry describing an event that involves Amazon GameLift resources (such as a fleet). In addition to tracking
 * activity, event codes and messages can provide additional information for troubleshooting and debugging problems.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> EVENT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::eventCodeAsString)).setter(setter(Builder::eventCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventCode").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Instant> EVENT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Event::eventTime)).setter(setter(Builder::eventTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventTime").build()).build();

    private static final SdkField<String> PRE_SIGNED_LOG_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::preSignedLogUrl)).setter(setter(Builder::preSignedLogUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreSignedLogUrl").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_ID_FIELD,
            RESOURCE_ID_FIELD, EVENT_CODE_FIELD, MESSAGE_FIELD, EVENT_TIME_FIELD, PRE_SIGNED_LOG_URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventId;

    private final String resourceId;

    private final String eventCode;

    private final String message;

    private final Instant eventTime;

    private final String preSignedLogUrl;

    private Event(BuilderImpl builder) {
        this.eventId = builder.eventId;
        this.resourceId = builder.resourceId;
        this.eventCode = builder.eventCode;
        this.message = builder.message;
        this.eventTime = builder.eventTime;
        this.preSignedLogUrl = builder.preSignedLogUrl;
    }

    /**
     * <p>
     * A unique identifier for a fleet event.
     * </p>
     * 
     * @return A unique identifier for a fleet event.
     */
    public String eventId() {
        return eventId;
    }

    /**
     * <p>
     * A unique identifier for an event resource, such as a fleet ID.
     * </p>
     * 
     * @return A unique identifier for an event resource, such as a fleet ID.
     */
    public String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The type of event being logged.
     * </p>
     * <p>
     * <b>Fleet creation events (ordered by fleet creation activity):</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event messaging
     * includes the fleet ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The compressed
     * build has started downloading to a fleet instance for installation.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance, and the build
     * files are now being extracted from the uploaded build and saved to an instance. Failure at this stage prevents a
     * fleet from moving to <code>ACTIVE</code> status. Logs for this stage display a list of the files that are
     * extracted and saved on the instance. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the Amazon
     * GameLift is now running the build's install script (if one is included). Failure in this stage prevents a fleet
     * from moving to <code>ACTIVE</code> status. Logs for this stage list the installation steps and whether or not the
     * install completed successfully. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift is now
     * verifying that the game server launch paths, which are specified in the fleet's runtime configuration, exist. If
     * any listed launch path exists, Amazon GameLift tries to launch a game server process and waits for the process to
     * report ready. Failures in this stage prevent a fleet from moving to <code>ACTIVE</code> status. Logs for this
     * stage list the launch paths in the runtime configuration and indicate whether each is found. Access the logs by
     * using the URL in <i>PreSignedLogUrl</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to <code>VALIDATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the executable
     * specified in a launch path does not exist on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because the
     * executable specified in a launch path failed to run on the fleet instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet activation
     * process. This event code indicates that the game build was successfully downloaded to a fleet instance, built,
     * and validated, but was not able to start a server process. Learn more at <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
     * > Debug Fleet Creation Issues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>. The fleet
     * is now ready to host game sessions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>VPC peering events:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an Amazon
     * GameLift fleet and a VPC in your AWS account.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status information
     * (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for peering failure is that
     * the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this, change the CIDR block for the VPC
     * in your AWS account. For more information on VPC peering failures, see <a
     * href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html"
     * >https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Spot instance events:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Other fleet events:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances, minimum/maximum
     * scaling limits). Event messaging includes the new capacity settings.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session protection
     * policy setting. Event messaging includes both the old and new policy setting.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_DELETED -- A request to delete a fleet was initiated.
     * </p>
     * </li>
     * <li>
     * <p>
     * GENERIC_EVENT -- An unspecified event has occurred.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventCode} will
     * return {@link EventCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventCodeAsString}.
     * </p>
     * 
     * @return The type of event being logged. </p>
     *         <p>
     *         <b>Fleet creation events (ordered by fleet creation activity):</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event
     *         messaging includes the fleet ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The
     *         compressed build has started downloading to a fleet instance for installation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance, and
     *         the build files are now being extracted from the uploaded build and saved to an instance. Failure at this
     *         stage prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage display a list of
     *         the files that are extracted and saved on the instance. Access the logs by using the URL in
     *         <i>PreSignedLogUrl</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the
     *         Amazon GameLift is now running the build's install script (if one is included). Failure in this stage
     *         prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage list the installation
     *         steps and whether or not the install completed successfully. Access the logs by using the URL in
     *         <i>PreSignedLogUrl</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift is
     *         now verifying that the game server launch paths, which are specified in the fleet's runtime
     *         configuration, exist. If any listed launch path exists, Amazon GameLift tries to launch a game server
     *         process and waits for the process to report ready. Failures in this stage prevent a fleet from moving to
     *         <code>ACTIVE</code> status. Logs for this stage list the launch paths in the runtime configuration and
     *         indicate whether each is found. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to <code>VALIDATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the
     *         executable specified in a launch path does not exist on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because the
     *         executable specified in a launch path failed to run on the fleet instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet
     *         activation process. This event code indicates that the game build was successfully downloaded to a fleet
     *         instance, built, and validated, but was not able to start a server process. Learn more at <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
     *         > Debug Fleet Creation Issues</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>. The
     *         fleet is now ready to host game sessions.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>VPC peering events:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an
     *         Amazon GameLift fleet and a VPC in your AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status
     *         information (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for
     *         peering failure is that the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this,
     *         change the CIDR block for the VPC in your AWS account. For more information on VPC peering failures, see
     *         <a href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html">
     *         https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Spot instance events:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Other fleet events:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances,
     *         minimum/maximum scaling limits). Event messaging includes the new capacity settings.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session
     *         protection policy setting. Event messaging includes both the old and new policy setting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_DELETED -- A request to delete a fleet was initiated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         GENERIC_EVENT -- An unspecified event has occurred.
     *         </p>
     *         </li>
     * @see EventCode
     */
    public EventCode eventCode() {
        return EventCode.fromValue(eventCode);
    }

    /**
     * <p>
     * The type of event being logged.
     * </p>
     * <p>
     * <b>Fleet creation events (ordered by fleet creation activity):</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event messaging
     * includes the fleet ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The compressed
     * build has started downloading to a fleet instance for installation.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance, and the build
     * files are now being extracted from the uploaded build and saved to an instance. Failure at this stage prevents a
     * fleet from moving to <code>ACTIVE</code> status. Logs for this stage display a list of the files that are
     * extracted and saved on the instance. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the Amazon
     * GameLift is now running the build's install script (if one is included). Failure in this stage prevents a fleet
     * from moving to <code>ACTIVE</code> status. Logs for this stage list the installation steps and whether or not the
     * install completed successfully. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift is now
     * verifying that the game server launch paths, which are specified in the fleet's runtime configuration, exist. If
     * any listed launch path exists, Amazon GameLift tries to launch a game server process and waits for the process to
     * report ready. Failures in this stage prevent a fleet from moving to <code>ACTIVE</code> status. Logs for this
     * stage list the launch paths in the runtime configuration and indicate whether each is found. Access the logs by
     * using the URL in <i>PreSignedLogUrl</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to <code>VALIDATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the executable
     * specified in a launch path does not exist on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because the
     * executable specified in a launch path failed to run on the fleet instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet activation
     * process. This event code indicates that the game build was successfully downloaded to a fleet instance, built,
     * and validated, but was not able to start a server process. Learn more at <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
     * > Debug Fleet Creation Issues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>. The fleet
     * is now ready to host game sessions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>VPC peering events:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an Amazon
     * GameLift fleet and a VPC in your AWS account.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status information
     * (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for peering failure is that
     * the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this, change the CIDR block for the VPC
     * in your AWS account. For more information on VPC peering failures, see <a
     * href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html"
     * >https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Spot instance events:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Other fleet events:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances, minimum/maximum
     * scaling limits). Event messaging includes the new capacity settings.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session protection
     * policy setting. Event messaging includes both the old and new policy setting.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_DELETED -- A request to delete a fleet was initiated.
     * </p>
     * </li>
     * <li>
     * <p>
     * GENERIC_EVENT -- An unspecified event has occurred.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventCode} will
     * return {@link EventCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventCodeAsString}.
     * </p>
     * 
     * @return The type of event being logged. </p>
     *         <p>
     *         <b>Fleet creation events (ordered by fleet creation activity):</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event
     *         messaging includes the fleet ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The
     *         compressed build has started downloading to a fleet instance for installation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance, and
     *         the build files are now being extracted from the uploaded build and saved to an instance. Failure at this
     *         stage prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage display a list of
     *         the files that are extracted and saved on the instance. Access the logs by using the URL in
     *         <i>PreSignedLogUrl</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the
     *         Amazon GameLift is now running the build's install script (if one is included). Failure in this stage
     *         prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage list the installation
     *         steps and whether or not the install completed successfully. Access the logs by using the URL in
     *         <i>PreSignedLogUrl</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift is
     *         now verifying that the game server launch paths, which are specified in the fleet's runtime
     *         configuration, exist. If any listed launch path exists, Amazon GameLift tries to launch a game server
     *         process and waits for the process to report ready. Failures in this stage prevent a fleet from moving to
     *         <code>ACTIVE</code> status. Logs for this stage list the launch paths in the runtime configuration and
     *         indicate whether each is found. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to <code>VALIDATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the
     *         executable specified in a launch path does not exist on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because the
     *         executable specified in a launch path failed to run on the fleet instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet
     *         activation process. This event code indicates that the game build was successfully downloaded to a fleet
     *         instance, built, and validated, but was not able to start a server process. Learn more at <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
     *         > Debug Fleet Creation Issues</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>. The
     *         fleet is now ready to host game sessions.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>VPC peering events:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an
     *         Amazon GameLift fleet and a VPC in your AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status
     *         information (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for
     *         peering failure is that the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this,
     *         change the CIDR block for the VPC in your AWS account. For more information on VPC peering failures, see
     *         <a href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html">
     *         https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Spot instance events:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Other fleet events:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances,
     *         minimum/maximum scaling limits). Event messaging includes the new capacity settings.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session
     *         protection policy setting. Event messaging includes both the old and new policy setting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_DELETED -- A request to delete a fleet was initiated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         GENERIC_EVENT -- An unspecified event has occurred.
     *         </p>
     *         </li>
     * @see EventCode
     */
    public String eventCodeAsString() {
        return eventCode;
    }

    /**
     * <p>
     * Additional information related to the event.
     * </p>
     * 
     * @return Additional information related to the event.
     */
    public String message() {
        return message;
    }

    /**
     * <p>
     * Time stamp indicating when this event occurred. Format is a number expressed in Unix time as milliseconds (for
     * example "1469498468.057").
     * </p>
     * 
     * @return Time stamp indicating when this event occurred. Format is a number expressed in Unix time as milliseconds
     *         (for example "1469498468.057").
     */
    public Instant eventTime() {
        return eventTime;
    }

    /**
     * <p>
     * Location of stored logs with additional detail that is related to the event. This is useful for debugging issues.
     * The URL is valid for 15 minutes. You can also access fleet creation logs through the Amazon GameLift console.
     * </p>
     * 
     * @return Location of stored logs with additional detail that is related to the event. This is useful for debugging
     *         issues. The URL is valid for 15 minutes. You can also access fleet creation logs through the Amazon
     *         GameLift console.
     */
    public String preSignedLogUrl() {
        return preSignedLogUrl;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(eventCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(eventTime());
        hashCode = 31 * hashCode + Objects.hashCode(preSignedLogUrl());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(eventId(), other.eventId()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(eventCodeAsString(), other.eventCodeAsString()) && Objects.equals(message(), other.message())
                && Objects.equals(eventTime(), other.eventTime()) && Objects.equals(preSignedLogUrl(), other.preSignedLogUrl());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Event").add("EventId", eventId()).add("ResourceId", resourceId())
                .add("EventCode", eventCodeAsString()).add("Message", message()).add("EventTime", eventTime())
                .add("PreSignedLogUrl", preSignedLogUrl()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "EventCode":
            return Optional.ofNullable(clazz.cast(eventCodeAsString()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "EventTime":
            return Optional.ofNullable(clazz.cast(eventTime()));
        case "PreSignedLogUrl":
            return Optional.ofNullable(clazz.cast(preSignedLogUrl()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * A unique identifier for a fleet event.
         * </p>
         * 
         * @param eventId
         *        A unique identifier for a fleet event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * A unique identifier for an event resource, such as a fleet ID.
         * </p>
         * 
         * @param resourceId
         *        A unique identifier for an event resource, such as a fleet ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The type of event being logged.
         * </p>
         * <p>
         * <b>Fleet creation events (ordered by fleet creation activity):</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event messaging
         * includes the fleet ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The
         * compressed build has started downloading to a fleet instance for installation.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance, and the
         * build files are now being extracted from the uploaded build and saved to an instance. Failure at this stage
         * prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage display a list of the files
         * that are extracted and saved on the instance. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the Amazon
         * GameLift is now running the build's install script (if one is included). Failure in this stage prevents a
         * fleet from moving to <code>ACTIVE</code> status. Logs for this stage list the installation steps and whether
         * or not the install completed successfully. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift is now
         * verifying that the game server launch paths, which are specified in the fleet's runtime configuration, exist.
         * If any listed launch path exists, Amazon GameLift tries to launch a game server process and waits for the
         * process to report ready. Failures in this stage prevent a fleet from moving to <code>ACTIVE</code> status.
         * Logs for this stage list the launch paths in the runtime configuration and indicate whether each is found.
         * Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to <code>VALIDATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the
         * executable specified in a launch path does not exist on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because the
         * executable specified in a launch path failed to run on the fleet instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet activation
         * process. This event code indicates that the game build was successfully downloaded to a fleet instance,
         * built, and validated, but was not able to start a server process. Learn more at <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
         * > Debug Fleet Creation Issues</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>. The
         * fleet is now ready to host game sessions.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>VPC peering events:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an Amazon
         * GameLift fleet and a VPC in your AWS account.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status
         * information (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for peering
         * failure is that the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this, change the CIDR
         * block for the VPC in your AWS account. For more information on VPC peering failures, see <a
         * href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html"
         * >https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Spot instance events:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Other fleet events:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances, minimum/maximum
         * scaling limits). Event messaging includes the new capacity settings.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session protection
         * policy setting. Event messaging includes both the old and new policy setting.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_DELETED -- A request to delete a fleet was initiated.
         * </p>
         * </li>
         * <li>
         * <p>
         * GENERIC_EVENT -- An unspecified event has occurred.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventCode
         *        The type of event being logged. </p>
         *        <p>
         *        <b>Fleet creation events (ordered by fleet creation activity):</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event
         *        messaging includes the fleet ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The
         *        compressed build has started downloading to a fleet instance for installation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance,
         *        and the build files are now being extracted from the uploaded build and saved to an instance. Failure
         *        at this stage prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage display
         *        a list of the files that are extracted and saved on the instance. Access the logs by using the URL in
         *        <i>PreSignedLogUrl</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the
         *        Amazon GameLift is now running the build's install script (if one is included). Failure in this stage
         *        prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage list the installation
         *        steps and whether or not the install completed successfully. Access the logs by using the URL in
         *        <i>PreSignedLogUrl</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift
         *        is now verifying that the game server launch paths, which are specified in the fleet's runtime
         *        configuration, exist. If any listed launch path exists, Amazon GameLift tries to launch a game server
         *        process and waits for the process to report ready. Failures in this stage prevent a fleet from moving
         *        to <code>ACTIVE</code> status. Logs for this stage list the launch paths in the runtime configuration
         *        and indicate whether each is found. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to
         *        <code>VALIDATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the
         *        executable specified in a launch path does not exist on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because
         *        the executable specified in a launch path failed to run on the fleet instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet
         *        activation process. This event code indicates that the game build was successfully downloaded to a
         *        fleet instance, built, and validated, but was not able to start a server process. Learn more at <a
         *        href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
         *        > Debug Fleet Creation Issues</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>.
         *        The fleet is now ready to host game sessions.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>VPC peering events:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an
         *        Amazon GameLift fleet and a VPC in your AWS account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status
         *        information (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for
         *        peering failure is that the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this,
         *        change the CIDR block for the VPC in your AWS account. For more information on VPC peering failures,
         *        see <a
         *        href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html"
         *        >https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Spot instance events:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Other fleet events:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances,
         *        minimum/maximum scaling limits). Event messaging includes the new capacity settings.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session
         *        protection policy setting. Event messaging includes both the old and new policy setting.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_DELETED -- A request to delete a fleet was initiated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        GENERIC_EVENT -- An unspecified event has occurred.
         *        </p>
         *        </li>
         * @see EventCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventCode
         */
        Builder eventCode(String eventCode);

        /**
         * <p>
         * The type of event being logged.
         * </p>
         * <p>
         * <b>Fleet creation events (ordered by fleet creation activity):</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event messaging
         * includes the fleet ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The
         * compressed build has started downloading to a fleet instance for installation.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance, and the
         * build files are now being extracted from the uploaded build and saved to an instance. Failure at this stage
         * prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage display a list of the files
         * that are extracted and saved on the instance. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the Amazon
         * GameLift is now running the build's install script (if one is included). Failure in this stage prevents a
         * fleet from moving to <code>ACTIVE</code> status. Logs for this stage list the installation steps and whether
         * or not the install completed successfully. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift is now
         * verifying that the game server launch paths, which are specified in the fleet's runtime configuration, exist.
         * If any listed launch path exists, Amazon GameLift tries to launch a game server process and waits for the
         * process to report ready. Failures in this stage prevent a fleet from moving to <code>ACTIVE</code> status.
         * Logs for this stage list the launch paths in the runtime configuration and indicate whether each is found.
         * Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to <code>VALIDATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the
         * executable specified in a launch path does not exist on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because the
         * executable specified in a launch path failed to run on the fleet instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet activation
         * process. This event code indicates that the game build was successfully downloaded to a fleet instance,
         * built, and validated, but was not able to start a server process. Learn more at <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
         * > Debug Fleet Creation Issues</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>. The
         * fleet is now ready to host game sessions.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>VPC peering events:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an Amazon
         * GameLift fleet and a VPC in your AWS account.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status
         * information (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for peering
         * failure is that the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this, change the CIDR
         * block for the VPC in your AWS account. For more information on VPC peering failures, see <a
         * href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html"
         * >https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Spot instance events:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Other fleet events:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances, minimum/maximum
         * scaling limits). Event messaging includes the new capacity settings.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session protection
         * policy setting. Event messaging includes both the old and new policy setting.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_DELETED -- A request to delete a fleet was initiated.
         * </p>
         * </li>
         * <li>
         * <p>
         * GENERIC_EVENT -- An unspecified event has occurred.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventCode
         *        The type of event being logged. </p>
         *        <p>
         *        <b>Fleet creation events (ordered by fleet creation activity):</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        FLEET_CREATED -- A fleet resource was successfully created with a status of <code>NEW</code>. Event
         *        messaging includes the fleet ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_DOWNLOADING -- Fleet status changed from <code>NEW</code> to <code>DOWNLOADING</code>. The
         *        compressed build has started downloading to a fleet instance for installation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_BINARY_DOWNLOAD_FAILED -- The build failed to download to the fleet instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_CREATION_EXTRACTING_BUILD – The game server build was successfully downloaded to an instance,
         *        and the build files are now being extracted from the uploaded build and saved to an instance. Failure
         *        at this stage prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage display
         *        a list of the files that are extracted and saved on the instance. Access the logs by using the URL in
         *        <i>PreSignedLogUrl</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_CREATION_RUNNING_INSTALLER – The game server build files were successfully extracted, and the
         *        Amazon GameLift is now running the build's install script (if one is included). Failure in this stage
         *        prevents a fleet from moving to <code>ACTIVE</code> status. Logs for this stage list the installation
         *        steps and whether or not the install completed successfully. Access the logs by using the URL in
         *        <i>PreSignedLogUrl</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_CREATION_VALIDATING_RUNTIME_CONFIG -- The build process was successful, and the Amazon GameLift
         *        is now verifying that the game server launch paths, which are specified in the fleet's runtime
         *        configuration, exist. If any listed launch path exists, Amazon GameLift tries to launch a game server
         *        process and waits for the process to report ready. Failures in this stage prevent a fleet from moving
         *        to <code>ACTIVE</code> status. Logs for this stage list the launch paths in the runtime configuration
         *        and indicate whether each is found. Access the logs by using the URL in <i>PreSignedLogUrl</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_VALIDATING -- Fleet status changed from <code>DOWNLOADING</code> to
         *        <code>VALIDATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VALIDATION_LAUNCH_PATH_NOT_FOUND -- Validation of the runtime configuration failed because the
         *        executable specified in a launch path does not exist on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_BUILDING -- Fleet status changed from <code>VALIDATING</code> to <code>BUILDING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VALIDATION_EXECUTABLE_RUNTIME_FAILURE -- Validation of the runtime configuration failed because
         *        the executable specified in a launch path failed to run on the fleet instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_ACTIVATING -- Fleet status changed from <code>BUILDING</code> to <code>ACTIVATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_ACTIVATION_FAILED - The fleet failed to successfully complete one of the steps in the fleet
         *        activation process. This event code indicates that the game build was successfully downloaded to a
         *        fleet instance, built, and validated, but was not able to start a server process. Learn more at <a
         *        href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
         *        > Debug Fleet Creation Issues</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_STATE_ACTIVE -- The fleet's status changed from <code>ACTIVATING</code> to <code>ACTIVE</code>.
         *        The fleet is now ready to host game sessions.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>VPC peering events:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        FLEET_VPC_PEERING_SUCCEEDED -- A VPC peering connection has been established between the VPC for an
         *        Amazon GameLift fleet and a VPC in your AWS account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VPC_PEERING_FAILED -- A requested VPC peering connection has failed. Event details and status
         *        information (see <a>DescribeVpcPeeringConnections</a>) provide additional detail. A common reason for
         *        peering failure is that the two VPCs have overlapping CIDR blocks of IPv4 addresses. To resolve this,
         *        change the CIDR block for the VPC in your AWS account. For more information on VPC peering failures,
         *        see <a
         *        href="https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html"
         *        >https://docs.aws.amazon.com/AmazonVPC/latest/PeeringGuide/invalid-peering-configurations.html</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_VPC_PEERING_DELETED -- A VPC peering connection has been successfully deleted.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Spot instance events:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        INSTANCE_INTERRUPTED -- A spot instance was interrupted by EC2 with a two-minute notification.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Other fleet events:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        FLEET_SCALING_EVENT -- A change was made to the fleet's capacity settings (desired instances,
         *        minimum/maximum scaling limits). Event messaging includes the new capacity settings.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_NEW_GAME_SESSION_PROTECTION_POLICY_UPDATED -- A change was made to the fleet's game session
         *        protection policy setting. Event messaging includes both the old and new policy setting.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_DELETED -- A request to delete a fleet was initiated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        GENERIC_EVENT -- An unspecified event has occurred.
         *        </p>
         *        </li>
         * @see EventCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventCode
         */
        Builder eventCode(EventCode eventCode);

        /**
         * <p>
         * Additional information related to the event.
         * </p>
         * 
         * @param message
         *        Additional information related to the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * Time stamp indicating when this event occurred. Format is a number expressed in Unix time as milliseconds
         * (for example "1469498468.057").
         * </p>
         * 
         * @param eventTime
         *        Time stamp indicating when this event occurred. Format is a number expressed in Unix time as
         *        milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTime(Instant eventTime);

        /**
         * <p>
         * Location of stored logs with additional detail that is related to the event. This is useful for debugging
         * issues. The URL is valid for 15 minutes. You can also access fleet creation logs through the Amazon GameLift
         * console.
         * </p>
         * 
         * @param preSignedLogUrl
         *        Location of stored logs with additional detail that is related to the event. This is useful for
         *        debugging issues. The URL is valid for 15 minutes. You can also access fleet creation logs through the
         *        Amazon GameLift console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preSignedLogUrl(String preSignedLogUrl);
    }

    static final class BuilderImpl implements Builder {
        private String eventId;

        private String resourceId;

        private String eventCode;

        private String message;

        private Instant eventTime;

        private String preSignedLogUrl;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            eventId(model.eventId);
            resourceId(model.resourceId);
            eventCode(model.eventCode);
            message(model.message);
            eventTime(model.eventTime);
            preSignedLogUrl(model.preSignedLogUrl);
        }

        public final String getEventId() {
            return eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        public final String getResourceId() {
            return resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        public final String getEventCode() {
            return eventCode;
        }

        @Override
        public final Builder eventCode(String eventCode) {
            this.eventCode = eventCode;
            return this;
        }

        @Override
        public final Builder eventCode(EventCode eventCode) {
            this.eventCode(eventCode == null ? null : eventCode.toString());
            return this;
        }

        public final void setEventCode(String eventCode) {
            this.eventCode = eventCode;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Instant getEventTime() {
            return eventTime;
        }

        @Override
        public final Builder eventTime(Instant eventTime) {
            this.eventTime = eventTime;
            return this;
        }

        public final void setEventTime(Instant eventTime) {
            this.eventTime = eventTime;
        }

        public final String getPreSignedLogUrl() {
            return preSignedLogUrl;
        }

        @Override
        public final Builder preSignedLogUrl(String preSignedLogUrl) {
            this.preSignedLogUrl = preSignedLogUrl;
            return this;
        }

        public final void setPreSignedLogUrl(String preSignedLogUrl) {
            this.preSignedLogUrl = preSignedLogUrl;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
