/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Guidelines for use with FlexMatch to match players into games. All matchmaking requests must specify a matchmaking
 * configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MatchmakingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<MatchmakingConfiguration.Builder, MatchmakingConfiguration> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::configurationArn)).setter(setter(Builder::configurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<String>> GAME_SESSION_QUEUE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(MatchmakingConfiguration::gameSessionQueueArns))
            .setter(setter(Builder::gameSessionQueueArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionQueueArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> REQUEST_TIMEOUT_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(MatchmakingConfiguration::requestTimeoutSeconds)).setter(setter(Builder::requestTimeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestTimeoutSeconds").build())
            .build();

    private static final SdkField<Integer> ACCEPTANCE_TIMEOUT_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(MatchmakingConfiguration::acceptanceTimeoutSeconds)).setter(setter(Builder::acceptanceTimeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceptanceTimeoutSeconds").build())
            .build();

    private static final SdkField<Boolean> ACCEPTANCE_REQUIRED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(MatchmakingConfiguration::acceptanceRequired)).setter(setter(Builder::acceptanceRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceptanceRequired").build())
            .build();

    private static final SdkField<String> RULE_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::ruleSetName)).setter(setter(Builder::ruleSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetName").build()).build();

    private static final SdkField<String> RULE_SET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::ruleSetArn)).setter(setter(Builder::ruleSetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetArn").build()).build();

    private static final SdkField<String> NOTIFICATION_TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::notificationTarget)).setter(setter(Builder::notificationTarget))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationTarget").build())
            .build();

    private static final SdkField<Integer> ADDITIONAL_PLAYER_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(MatchmakingConfiguration::additionalPlayerCount)).setter(setter(Builder::additionalPlayerCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalPlayerCount").build())
            .build();

    private static final SdkField<String> CUSTOM_EVENT_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::customEventData)).setter(setter(Builder::customEventData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomEventData").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(MatchmakingConfiguration::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<List<GameProperty>> GAME_PROPERTIES_FIELD = SdkField
            .<List<GameProperty>> builder(MarshallingType.LIST)
            .getter(getter(MatchmakingConfiguration::gameProperties))
            .setter(setter(Builder::gameProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GAME_SESSION_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::gameSessionData)).setter(setter(Builder::gameSessionData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionData").build()).build();

    private static final SdkField<String> BACKFILL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MatchmakingConfiguration::backfillModeAsString)).setter(setter(Builder::backfillMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackfillMode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONFIGURATION_ARN_FIELD, DESCRIPTION_FIELD, GAME_SESSION_QUEUE_ARNS_FIELD, REQUEST_TIMEOUT_SECONDS_FIELD,
            ACCEPTANCE_TIMEOUT_SECONDS_FIELD, ACCEPTANCE_REQUIRED_FIELD, RULE_SET_NAME_FIELD, RULE_SET_ARN_FIELD,
            NOTIFICATION_TARGET_FIELD, ADDITIONAL_PLAYER_COUNT_FIELD, CUSTOM_EVENT_DATA_FIELD, CREATION_TIME_FIELD,
            GAME_PROPERTIES_FIELD, GAME_SESSION_DATA_FIELD, BACKFILL_MODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String configurationArn;

    private final String description;

    private final List<String> gameSessionQueueArns;

    private final Integer requestTimeoutSeconds;

    private final Integer acceptanceTimeoutSeconds;

    private final Boolean acceptanceRequired;

    private final String ruleSetName;

    private final String ruleSetArn;

    private final String notificationTarget;

    private final Integer additionalPlayerCount;

    private final String customEventData;

    private final Instant creationTime;

    private final List<GameProperty> gameProperties;

    private final String gameSessionData;

    private final String backfillMode;

    private MatchmakingConfiguration(BuilderImpl builder) {
        this.name = builder.name;
        this.configurationArn = builder.configurationArn;
        this.description = builder.description;
        this.gameSessionQueueArns = builder.gameSessionQueueArns;
        this.requestTimeoutSeconds = builder.requestTimeoutSeconds;
        this.acceptanceTimeoutSeconds = builder.acceptanceTimeoutSeconds;
        this.acceptanceRequired = builder.acceptanceRequired;
        this.ruleSetName = builder.ruleSetName;
        this.ruleSetArn = builder.ruleSetArn;
        this.notificationTarget = builder.notificationTarget;
        this.additionalPlayerCount = builder.additionalPlayerCount;
        this.customEventData = builder.customEventData;
        this.creationTime = builder.creationTime;
        this.gameProperties = builder.gameProperties;
        this.gameSessionData = builder.gameSessionData;
        this.backfillMode = builder.backfillMode;
    }

    /**
     * <p>
     * A unique identifier for a matchmaking configuration. This name is used to identify the configuration associated
     * with a matchmaking request or ticket.
     * </p>
     * 
     * @return A unique identifier for a matchmaking configuration. This name is used to identify the configuration
     *         associated with a matchmaking request or ticket.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Amazon Resource Name (<a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned to a
     * GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique across all Regions. In a
     * GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
     * </p>
     * 
     * @return Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
     *         assigned to a GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique
     *         across all Regions. In a GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
     */
    public String configurationArn() {
        return configurationArn;
    }

    /**
     * <p>
     * A descriptive label that is associated with matchmaking configuration.
     * </p>
     * 
     * @return A descriptive label that is associated with matchmaking configuration.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the GameSessionQueueArns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasGameSessionQueueArns() {
        return gameSessionQueueArns != null && !(gameSessionQueueArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon Resource Name (<a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned to a
     * GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions. GameLift
     * uses the listed queues when placing game sessions for matches that are created with this matchmaking
     * configuration. Queues can be located in any Region.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGameSessionQueueArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
     *         assigned to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across all
     *         Regions. GameLift uses the listed queues when placing game sessions for matches that are created with
     *         this matchmaking configuration. Queues can be located in any Region.
     */
    public List<String> gameSessionQueueArns() {
        return gameSessionQueueArns;
    }

    /**
     * <p>
     * The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out. Requests
     * that fail due to timing out can be resubmitted as needed.
     * </p>
     * 
     * @return The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out.
     *         Requests that fail due to timing out can be resubmitted as needed.
     */
    public Integer requestTimeoutSeconds() {
        return requestTimeoutSeconds;
    }

    /**
     * <p>
     * The length of time (in seconds) to wait for players to accept a proposed match. If any player rejects the match
     * or fails to accept before the timeout, the ticket continues to look for an acceptable match.
     * </p>
     * 
     * @return The length of time (in seconds) to wait for players to accept a proposed match. If any player rejects the
     *         match or fails to accept before the timeout, the ticket continues to look for an acceptable match.
     */
    public Integer acceptanceTimeoutSeconds() {
        return acceptanceTimeoutSeconds;
    }

    /**
     * <p>
     * A flag that indicates whether a match that was created with this configuration must be accepted by the matched
     * players. To require acceptance, set to TRUE.
     * </p>
     * 
     * @return A flag that indicates whether a match that was created with this configuration must be accepted by the
     *         matched players. To require acceptance, set to TRUE.
     */
    public Boolean acceptanceRequired() {
        return acceptanceRequired;
    }

    /**
     * <p>
     * A unique identifier for a matchmaking rule set to use with this configuration. A matchmaking configuration can
     * only use rule sets that are defined in the same Region.
     * </p>
     * 
     * @return A unique identifier for a matchmaking rule set to use with this configuration. A matchmaking
     *         configuration can only use rule sets that are defined in the same Region.
     */
    public String ruleSetName() {
        return ruleSetName;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated with the
     * GameLift matchmaking rule set resource that this configuration uses.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated
     *         with the GameLift matchmaking rule set resource that this configuration uses.
     */
    public String ruleSetArn() {
        return ruleSetArn;
    }

    /**
     * <p>
     * An SNS topic ARN that is set up to receive matchmaking notifications.
     * </p>
     * 
     * @return An SNS topic ARN that is set up to receive matchmaking notifications.
     */
    public String notificationTarget() {
        return notificationTarget;
    }

    /**
     * <p>
     * The number of player slots in a match to keep open for future players. For example, assume that the
     * configuration's rule set specifies a match for a single 12-person team. If the additional player count is set to
     * 2, only 10 players are initially selected for the match.
     * </p>
     * 
     * @return The number of player slots in a match to keep open for future players. For example, assume that the
     *         configuration's rule set specifies a match for a single 12-person team. If the additional player count is
     *         set to 2, only 10 players are initially selected for the match.
     */
    public Integer additionalPlayerCount() {
        return additionalPlayerCount;
    }

    /**
     * <p>
     * Information to attach to all events related to the matchmaking configuration.
     * </p>
     * 
     * @return Information to attach to all events related to the matchmaking configuration.
     */
    public String customEventData() {
        return customEventData;
    }

    /**
     * <p>
     * The time stamp indicating when this data object was created. The format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return The time stamp indicating when this data object was created. The format is a number expressed in Unix
     *         time as milliseconds (for example "1469498468.057").
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * Returns true if the GameProperties property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGameProperties() {
        return gameProperties != null && !(gameProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of custom properties for a game session, formatted as key-value pairs. These properties are passed to a
     * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is created for a
     * successful match.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGameProperties()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A set of custom properties for a game session, formatted as key-value pairs. These properties are passed
     *         to a game server process in the <a>GameSession</a> object with a request to start a new game session (see
     *         <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is
     *         created for a successful match.
     */
    public List<GameProperty> gameProperties() {
        return gameProperties;
    }

    /**
     * <p>
     * A set of custom game session properties, formatted as a single string value. This data is passed to a game server
     * process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is created for a
     * successful match.
     * </p>
     * 
     * @return A set of custom game session properties, formatted as a single string value. This data is passed to a
     *         game server process in the <a>GameSession</a> object with a request to start a new game session (see <a
     *         href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is
     *         created for a successful match.
     */
    public String gameSessionData() {
        return gameSessionData;
    }

    /**
     * <p>
     * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that the
     * game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that GameLift
     * creates <a>StartMatchBackfill</a> requests whenever a game session has one or more open slots. Learn more about
     * manual and automatic backfill in <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill Existing Games
     * with FlexMatch</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #backfillMode} will
     * return {@link BackfillMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #backfillModeAsString}.
     * </p>
     * 
     * @return The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates
     *         that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates
     *         that GameLift creates <a>StartMatchBackfill</a> requests whenever a game session has one or more open
     *         slots. Learn more about manual and automatic backfill in <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill Existing
     *         Games with FlexMatch</a>.
     * @see BackfillMode
     */
    public BackfillMode backfillMode() {
        return BackfillMode.fromValue(backfillMode);
    }

    /**
     * <p>
     * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that the
     * game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that GameLift
     * creates <a>StartMatchBackfill</a> requests whenever a game session has one or more open slots. Learn more about
     * manual and automatic backfill in <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill Existing Games
     * with FlexMatch</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #backfillMode} will
     * return {@link BackfillMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #backfillModeAsString}.
     * </p>
     * 
     * @return The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates
     *         that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates
     *         that GameLift creates <a>StartMatchBackfill</a> requests whenever a game session has one or more open
     *         slots. Learn more about manual and automatic backfill in <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill Existing
     *         Games with FlexMatch</a>.
     * @see BackfillMode
     */
    public String backfillModeAsString() {
        return backfillMode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(configurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionQueueArns());
        hashCode = 31 * hashCode + Objects.hashCode(requestTimeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(acceptanceTimeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(acceptanceRequired());
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetName());
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetArn());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTarget());
        hashCode = 31 * hashCode + Objects.hashCode(additionalPlayerCount());
        hashCode = 31 * hashCode + Objects.hashCode(customEventData());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(gameProperties());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionData());
        hashCode = 31 * hashCode + Objects.hashCode(backfillModeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MatchmakingConfiguration)) {
            return false;
        }
        MatchmakingConfiguration other = (MatchmakingConfiguration) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(configurationArn(), other.configurationArn())
                && Objects.equals(description(), other.description())
                && Objects.equals(gameSessionQueueArns(), other.gameSessionQueueArns())
                && Objects.equals(requestTimeoutSeconds(), other.requestTimeoutSeconds())
                && Objects.equals(acceptanceTimeoutSeconds(), other.acceptanceTimeoutSeconds())
                && Objects.equals(acceptanceRequired(), other.acceptanceRequired())
                && Objects.equals(ruleSetName(), other.ruleSetName()) && Objects.equals(ruleSetArn(), other.ruleSetArn())
                && Objects.equals(notificationTarget(), other.notificationTarget())
                && Objects.equals(additionalPlayerCount(), other.additionalPlayerCount())
                && Objects.equals(customEventData(), other.customEventData())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(gameProperties(), other.gameProperties())
                && Objects.equals(gameSessionData(), other.gameSessionData())
                && Objects.equals(backfillModeAsString(), other.backfillModeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MatchmakingConfiguration").add("Name", name()).add("ConfigurationArn", configurationArn())
                .add("Description", description()).add("GameSessionQueueArns", gameSessionQueueArns())
                .add("RequestTimeoutSeconds", requestTimeoutSeconds())
                .add("AcceptanceTimeoutSeconds", acceptanceTimeoutSeconds()).add("AcceptanceRequired", acceptanceRequired())
                .add("RuleSetName", ruleSetName()).add("RuleSetArn", ruleSetArn())
                .add("NotificationTarget", notificationTarget()).add("AdditionalPlayerCount", additionalPlayerCount())
                .add("CustomEventData", customEventData()).add("CreationTime", creationTime())
                .add("GameProperties", gameProperties()).add("GameSessionData", gameSessionData())
                .add("BackfillMode", backfillModeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ConfigurationArn":
            return Optional.ofNullable(clazz.cast(configurationArn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "GameSessionQueueArns":
            return Optional.ofNullable(clazz.cast(gameSessionQueueArns()));
        case "RequestTimeoutSeconds":
            return Optional.ofNullable(clazz.cast(requestTimeoutSeconds()));
        case "AcceptanceTimeoutSeconds":
            return Optional.ofNullable(clazz.cast(acceptanceTimeoutSeconds()));
        case "AcceptanceRequired":
            return Optional.ofNullable(clazz.cast(acceptanceRequired()));
        case "RuleSetName":
            return Optional.ofNullable(clazz.cast(ruleSetName()));
        case "RuleSetArn":
            return Optional.ofNullable(clazz.cast(ruleSetArn()));
        case "NotificationTarget":
            return Optional.ofNullable(clazz.cast(notificationTarget()));
        case "AdditionalPlayerCount":
            return Optional.ofNullable(clazz.cast(additionalPlayerCount()));
        case "CustomEventData":
            return Optional.ofNullable(clazz.cast(customEventData()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "GameProperties":
            return Optional.ofNullable(clazz.cast(gameProperties()));
        case "GameSessionData":
            return Optional.ofNullable(clazz.cast(gameSessionData()));
        case "BackfillMode":
            return Optional.ofNullable(clazz.cast(backfillModeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MatchmakingConfiguration, T> g) {
        return obj -> g.apply((MatchmakingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MatchmakingConfiguration> {
        /**
         * <p>
         * A unique identifier for a matchmaking configuration. This name is used to identify the configuration
         * associated with a matchmaking request or ticket.
         * </p>
         * 
         * @param name
         *        A unique identifier for a matchmaking configuration. This name is used to identify the configuration
         *        associated with a matchmaking request or ticket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned
         * to a GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique across all
         * Regions. In a GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
         * </p>
         * 
         * @param configurationArn
         *        Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
         *        assigned to a GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique
         *        across all Regions. In a GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationArn(String configurationArn);

        /**
         * <p>
         * A descriptive label that is associated with matchmaking configuration.
         * </p>
         * 
         * @param description
         *        A descriptive label that is associated with matchmaking configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned
         * to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions.
         * GameLift uses the listed queues when placing game sessions for matches that are created with this matchmaking
         * configuration. Queues can be located in any Region.
         * </p>
         * 
         * @param gameSessionQueueArns
         *        Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
         *        assigned to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across
         *        all Regions. GameLift uses the listed queues when placing game sessions for matches that are created
         *        with this matchmaking configuration. Queues can be located in any Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueArns(Collection<String> gameSessionQueueArns);

        /**
         * <p>
         * Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned
         * to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions.
         * GameLift uses the listed queues when placing game sessions for matches that are created with this matchmaking
         * configuration. Queues can be located in any Region.
         * </p>
         * 
         * @param gameSessionQueueArns
         *        Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
         *        assigned to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across
         *        all Regions. GameLift uses the listed queues when placing game sessions for matches that are created
         *        with this matchmaking configuration. Queues can be located in any Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueArns(String... gameSessionQueueArns);

        /**
         * <p>
         * The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out. Requests
         * that fail due to timing out can be resubmitted as needed.
         * </p>
         * 
         * @param requestTimeoutSeconds
         *        The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out.
         *        Requests that fail due to timing out can be resubmitted as needed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestTimeoutSeconds(Integer requestTimeoutSeconds);

        /**
         * <p>
         * The length of time (in seconds) to wait for players to accept a proposed match. If any player rejects the
         * match or fails to accept before the timeout, the ticket continues to look for an acceptable match.
         * </p>
         * 
         * @param acceptanceTimeoutSeconds
         *        The length of time (in seconds) to wait for players to accept a proposed match. If any player rejects
         *        the match or fails to accept before the timeout, the ticket continues to look for an acceptable match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceptanceTimeoutSeconds(Integer acceptanceTimeoutSeconds);

        /**
         * <p>
         * A flag that indicates whether a match that was created with this configuration must be accepted by the
         * matched players. To require acceptance, set to TRUE.
         * </p>
         * 
         * @param acceptanceRequired
         *        A flag that indicates whether a match that was created with this configuration must be accepted by the
         *        matched players. To require acceptance, set to TRUE.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceptanceRequired(Boolean acceptanceRequired);

        /**
         * <p>
         * A unique identifier for a matchmaking rule set to use with this configuration. A matchmaking configuration
         * can only use rule sets that are defined in the same Region.
         * </p>
         * 
         * @param ruleSetName
         *        A unique identifier for a matchmaking rule set to use with this configuration. A matchmaking
         *        configuration can only use rule sets that are defined in the same Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetName(String ruleSetName);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated with
         * the GameLift matchmaking rule set resource that this configuration uses.
         * </p>
         * 
         * @param ruleSetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated
         *        with the GameLift matchmaking rule set resource that this configuration uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetArn(String ruleSetArn);

        /**
         * <p>
         * An SNS topic ARN that is set up to receive matchmaking notifications.
         * </p>
         * 
         * @param notificationTarget
         *        An SNS topic ARN that is set up to receive matchmaking notifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTarget(String notificationTarget);

        /**
         * <p>
         * The number of player slots in a match to keep open for future players. For example, assume that the
         * configuration's rule set specifies a match for a single 12-person team. If the additional player count is set
         * to 2, only 10 players are initially selected for the match.
         * </p>
         * 
         * @param additionalPlayerCount
         *        The number of player slots in a match to keep open for future players. For example, assume that the
         *        configuration's rule set specifies a match for a single 12-person team. If the additional player count
         *        is set to 2, only 10 players are initially selected for the match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalPlayerCount(Integer additionalPlayerCount);

        /**
         * <p>
         * Information to attach to all events related to the matchmaking configuration.
         * </p>
         * 
         * @param customEventData
         *        Information to attach to all events related to the matchmaking configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customEventData(String customEventData);

        /**
         * <p>
         * The time stamp indicating when this data object was created. The format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param creationTime
         *        The time stamp indicating when this data object was created. The format is a number expressed in Unix
         *        time as milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key-value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is created
         * for a successful match.
         * </p>
         * 
         * @param gameProperties
         *        A set of custom properties for a game session, formatted as key-value pairs. These properties are
         *        passed to a game server process in the <a>GameSession</a> object with a request to start a new game
         *        session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is
         *        created for a successful match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(Collection<GameProperty> gameProperties);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key-value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is created
         * for a successful match.
         * </p>
         * 
         * @param gameProperties
         *        A set of custom properties for a game session, formatted as key-value pairs. These properties are
         *        passed to a game server process in the <a>GameSession</a> object with a request to start a new game
         *        session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is
         *        created for a successful match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(GameProperty... gameProperties);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key-value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is created
         * for a successful match.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GameProperty>.Builder} avoiding the need to
         * create one manually via {@link List<GameProperty>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GameProperty>.Builder#build()} is called immediately and its
         * result is passed to {@link #gameProperties(List<GameProperty>)}.
         * 
         * @param gameProperties
         *        a consumer that will call methods on {@link List<GameProperty>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameProperties(List<GameProperty>)
         */
        Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties);

        /**
         * <p>
         * A set of custom game session properties, formatted as a single string value. This data is passed to a game
         * server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is created
         * for a successful match.
         * </p>
         * 
         * @param gameSessionData
         *        A set of custom game session properties, formatted as a single string value. This data is passed to a
         *        game server process in the <a>GameSession</a> object with a request to start a new game session (see
         *        <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). This information is added to the new <a>GameSession</a> object that is
         *        created for a successful match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionData(String gameSessionData);

        /**
         * <p>
         * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that
         * the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that
         * GameLift creates <a>StartMatchBackfill</a> requests whenever a game session has one or more open slots. Learn
         * more about manual and automatic backfill in <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill Existing Games
         * with FlexMatch</a>.
         * </p>
         * 
         * @param backfillMode
         *        The method used to backfill game sessions created with this matchmaking configuration. MANUAL
         *        indicates that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC
         *        indicates that GameLift creates <a>StartMatchBackfill</a> requests whenever a game session has one or
         *        more open slots. Learn more about manual and automatic backfill in <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill
         *        Existing Games with FlexMatch</a>.
         * @see BackfillMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackfillMode
         */
        Builder backfillMode(String backfillMode);

        /**
         * <p>
         * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that
         * the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that
         * GameLift creates <a>StartMatchBackfill</a> requests whenever a game session has one or more open slots. Learn
         * more about manual and automatic backfill in <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill Existing Games
         * with FlexMatch</a>.
         * </p>
         * 
         * @param backfillMode
         *        The method used to backfill game sessions created with this matchmaking configuration. MANUAL
         *        indicates that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC
         *        indicates that GameLift creates <a>StartMatchBackfill</a> requests whenever a game session has one or
         *        more open slots. Learn more about manual and automatic backfill in <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html">Backfill
         *        Existing Games with FlexMatch</a>.
         * @see BackfillMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackfillMode
         */
        Builder backfillMode(BackfillMode backfillMode);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String configurationArn;

        private String description;

        private List<String> gameSessionQueueArns = DefaultSdkAutoConstructList.getInstance();

        private Integer requestTimeoutSeconds;

        private Integer acceptanceTimeoutSeconds;

        private Boolean acceptanceRequired;

        private String ruleSetName;

        private String ruleSetArn;

        private String notificationTarget;

        private Integer additionalPlayerCount;

        private String customEventData;

        private Instant creationTime;

        private List<GameProperty> gameProperties = DefaultSdkAutoConstructList.getInstance();

        private String gameSessionData;

        private String backfillMode;

        private BuilderImpl() {
        }

        private BuilderImpl(MatchmakingConfiguration model) {
            name(model.name);
            configurationArn(model.configurationArn);
            description(model.description);
            gameSessionQueueArns(model.gameSessionQueueArns);
            requestTimeoutSeconds(model.requestTimeoutSeconds);
            acceptanceTimeoutSeconds(model.acceptanceTimeoutSeconds);
            acceptanceRequired(model.acceptanceRequired);
            ruleSetName(model.ruleSetName);
            ruleSetArn(model.ruleSetArn);
            notificationTarget(model.notificationTarget);
            additionalPlayerCount(model.additionalPlayerCount);
            customEventData(model.customEventData);
            creationTime(model.creationTime);
            gameProperties(model.gameProperties);
            gameSessionData(model.gameSessionData);
            backfillMode(model.backfillMode);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getConfigurationArn() {
            return configurationArn;
        }

        @Override
        public final Builder configurationArn(String configurationArn) {
            this.configurationArn = configurationArn;
            return this;
        }

        public final void setConfigurationArn(String configurationArn) {
            this.configurationArn = configurationArn;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getGameSessionQueueArns() {
            return gameSessionQueueArns;
        }

        @Override
        public final Builder gameSessionQueueArns(Collection<String> gameSessionQueueArns) {
            this.gameSessionQueueArns = QueueArnsListCopier.copy(gameSessionQueueArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameSessionQueueArns(String... gameSessionQueueArns) {
            gameSessionQueueArns(Arrays.asList(gameSessionQueueArns));
            return this;
        }

        public final void setGameSessionQueueArns(Collection<String> gameSessionQueueArns) {
            this.gameSessionQueueArns = QueueArnsListCopier.copy(gameSessionQueueArns);
        }

        public final Integer getRequestTimeoutSeconds() {
            return requestTimeoutSeconds;
        }

        @Override
        public final Builder requestTimeoutSeconds(Integer requestTimeoutSeconds) {
            this.requestTimeoutSeconds = requestTimeoutSeconds;
            return this;
        }

        public final void setRequestTimeoutSeconds(Integer requestTimeoutSeconds) {
            this.requestTimeoutSeconds = requestTimeoutSeconds;
        }

        public final Integer getAcceptanceTimeoutSeconds() {
            return acceptanceTimeoutSeconds;
        }

        @Override
        public final Builder acceptanceTimeoutSeconds(Integer acceptanceTimeoutSeconds) {
            this.acceptanceTimeoutSeconds = acceptanceTimeoutSeconds;
            return this;
        }

        public final void setAcceptanceTimeoutSeconds(Integer acceptanceTimeoutSeconds) {
            this.acceptanceTimeoutSeconds = acceptanceTimeoutSeconds;
        }

        public final Boolean getAcceptanceRequired() {
            return acceptanceRequired;
        }

        @Override
        public final Builder acceptanceRequired(Boolean acceptanceRequired) {
            this.acceptanceRequired = acceptanceRequired;
            return this;
        }

        public final void setAcceptanceRequired(Boolean acceptanceRequired) {
            this.acceptanceRequired = acceptanceRequired;
        }

        public final String getRuleSetName() {
            return ruleSetName;
        }

        @Override
        public final Builder ruleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
            return this;
        }

        public final void setRuleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
        }

        public final String getRuleSetArn() {
            return ruleSetArn;
        }

        @Override
        public final Builder ruleSetArn(String ruleSetArn) {
            this.ruleSetArn = ruleSetArn;
            return this;
        }

        public final void setRuleSetArn(String ruleSetArn) {
            this.ruleSetArn = ruleSetArn;
        }

        public final String getNotificationTarget() {
            return notificationTarget;
        }

        @Override
        public final Builder notificationTarget(String notificationTarget) {
            this.notificationTarget = notificationTarget;
            return this;
        }

        public final void setNotificationTarget(String notificationTarget) {
            this.notificationTarget = notificationTarget;
        }

        public final Integer getAdditionalPlayerCount() {
            return additionalPlayerCount;
        }

        @Override
        public final Builder additionalPlayerCount(Integer additionalPlayerCount) {
            this.additionalPlayerCount = additionalPlayerCount;
            return this;
        }

        public final void setAdditionalPlayerCount(Integer additionalPlayerCount) {
            this.additionalPlayerCount = additionalPlayerCount;
        }

        public final String getCustomEventData() {
            return customEventData;
        }

        @Override
        public final Builder customEventData(String customEventData) {
            this.customEventData = customEventData;
            return this;
        }

        public final void setCustomEventData(String customEventData) {
            this.customEventData = customEventData;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Collection<GameProperty.Builder> getGameProperties() {
            return gameProperties != null ? gameProperties.stream().map(GameProperty::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder gameProperties(Collection<GameProperty> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copy(gameProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(GameProperty... gameProperties) {
            gameProperties(Arrays.asList(gameProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties) {
            gameProperties(Stream.of(gameProperties).map(c -> GameProperty.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setGameProperties(Collection<GameProperty.BuilderImpl> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copyFromBuilder(gameProperties);
        }

        public final String getGameSessionData() {
            return gameSessionData;
        }

        @Override
        public final Builder gameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
            return this;
        }

        public final void setGameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
        }

        public final String getBackfillModeAsString() {
            return backfillMode;
        }

        @Override
        public final Builder backfillMode(String backfillMode) {
            this.backfillMode = backfillMode;
            return this;
        }

        @Override
        public final Builder backfillMode(BackfillMode backfillMode) {
            this.backfillMode(backfillMode == null ? null : backfillMode.toString());
            return this;
        }

        public final void setBackfillMode(String backfillMode) {
            this.backfillMode = backfillMode;
        }

        @Override
        public MatchmakingConfiguration build() {
            return new MatchmakingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
