/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Properties that describe an instance of a virtual computing resource that hosts one or more game servers. A fleet may
 * contain zero or more instances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Instance implements SdkPojo, Serializable, ToCopyableBuilder<Instance.Builder, Instance> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::operatingSystemAsString)).setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Instance::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            INSTANCE_ID_FIELD, IP_ADDRESS_FIELD, DNS_NAME_FIELD, OPERATING_SYSTEM_FIELD, TYPE_FIELD, STATUS_FIELD,
            CREATION_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String instanceId;

    private final String ipAddress;

    private final String dnsName;

    private final String operatingSystem;

    private final String type;

    private final String status;

    private final Instant creationTime;

    private Instance(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.instanceId = builder.instanceId;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.operatingSystem = builder.operatingSystem;
        this.type = builder.type;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
    }

    /**
     * <p>
     * A unique identifier for a fleet that the instance is in.
     * </p>
     * 
     * @return A unique identifier for a fleet that the instance is in.
     */
    public String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * A unique identifier for an instance.
     * </p>
     * 
     * @return A unique identifier for an instance.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * IP address that is assigned to the instance.
     * </p>
     * 
     * @return IP address that is assigned to the instance.
     */
    public String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * DNS identifier assigned to the instance that is running the game session. Values have the following format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     * >Amazon EC2 Instance IP Addressing</a>.)
     * </p>
     * </li>
     * </ul>
     * <p>
     * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the IP
     * address.
     * </p>
     * 
     * @return DNS identifier assigned to the instance that is running the game session. Values have the following
     *         format:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     *         >Amazon EC2 Instance IP Addressing</a>.)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not
     *         the IP address.
     */
    public String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * Operating system that is running on this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that is running on this instance.
     * @see OperatingSystem
     */
    public OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * Operating system that is running on this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that is running on this instance.
     * @see OperatingSystem
     */
    public String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * EC2 instance type that defines the computing resources of this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return EC2 instance type that defines the computing resources of this instance.
     * @see EC2InstanceType
     */
    public EC2InstanceType type() {
        return EC2InstanceType.fromValue(type);
    }

    /**
     * <p>
     * EC2 instance type that defines the computing resources of this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return EC2 instance type that defines the computing resources of this instance.
     * @see EC2InstanceType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Current status of the instance. Possible statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined in
     * the fleet's run-time configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
     * launched and reported back to Amazon GameLift that it is ready to host a game session. The instance is now
     * considered ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity during
     * a scaling down event or to recycle resources in the event of a problem.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link InstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the instance. Possible statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
     *         defined in the fleet's run-time configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
     *         successfully launched and reported back to Amazon GameLift that it is ready to host a game session. The
     *         instance is now considered ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
     *         during a scaling down event or to recycle resources in the event of a problem.
     *         </p>
     *         </li>
     * @see InstanceStatus
     */
    public InstanceStatus status() {
        return InstanceStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the instance. Possible statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined in
     * the fleet's run-time configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
     * launched and reported back to Amazon GameLift that it is ready to host a game session. The instance is now
     * considered ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity during
     * a scaling down event or to recycle resources in the event of a problem.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link InstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the instance. Possible statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
     *         defined in the fleet's run-time configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
     *         successfully launched and reported back to Amazon GameLift that it is ready to host a game session. The
     *         instance is now considered ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
     *         during a scaling down event or to recycle resources in the event of a problem.
     *         </p>
     *         </li>
     * @see InstanceStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example "1469498468.057").
     */
    public Instant creationTime() {
        return creationTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Instance)) {
            return false;
        }
        Instance other = (Instance) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(ipAddress(), other.ipAddress()) && Objects.equals(dnsName(), other.dnsName())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationTime(), other.creationTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Instance").add("FleetId", fleetId()).add("InstanceId", instanceId())
                .add("IpAddress", ipAddress()).add("DnsName", dnsName()).add("OperatingSystem", operatingSystemAsString())
                .add("Type", typeAsString()).add("Status", statusAsString()).add("CreationTime", creationTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Instance, T> g) {
        return obj -> g.apply((Instance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Instance> {
        /**
         * <p>
         * A unique identifier for a fleet that the instance is in.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for a fleet that the instance is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * A unique identifier for an instance.
         * </p>
         * 
         * @param instanceId
         *        A unique identifier for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * IP address that is assigned to the instance.
         * </p>
         * 
         * @param ipAddress
         *        IP address that is assigned to the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * DNS identifier assigned to the instance that is running the game session. Values have the following format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         * >Amazon EC2 Instance IP Addressing</a>.)
         * </p>
         * </li>
         * </ul>
         * <p>
         * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the
         * IP address.
         * </p>
         * 
         * @param dnsName
         *        DNS identifier assigned to the instance that is running the game session. Values have the following
         *        format:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TLS-enabled fleets:
         *        <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         *        >Amazon EC2 Instance IP Addressing</a>.)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name,
         *        not the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * Operating system that is running on this instance.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that is running on this instance.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * Operating system that is running on this instance.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that is running on this instance.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * EC2 instance type that defines the computing resources of this instance.
         * </p>
         * 
         * @param type
         *        EC2 instance type that defines the computing resources of this instance.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder type(String type);

        /**
         * <p>
         * EC2 instance type that defines the computing resources of this instance.
         * </p>
         * 
         * @param type
         *        EC2 instance type that defines the computing resources of this instance.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder type(EC2InstanceType type);

        /**
         * <p>
         * Current status of the instance. Possible statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined
         * in the fleet's run-time configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
         * launched and reported back to Amazon GameLift that it is ready to host a game session. The instance is now
         * considered ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
         * during a scaling down event or to recycle resources in the event of a problem.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the instance. Possible statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
         *        defined in the fleet's run-time configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
         *        successfully launched and reported back to Amazon GameLift that it is ready to host a game session.
         *        The instance is now considered ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce
         *        capacity during a scaling down event or to recycle resources in the event of a problem.
         *        </p>
         *        </li>
         * @see InstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the instance. Possible statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined
         * in the fleet's run-time configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
         * launched and reported back to Amazon GameLift that it is ready to host a game session. The instance is now
         * considered ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
         * during a scaling down event or to recycle resources in the event of a problem.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the instance. Possible statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
         *        defined in the fleet's run-time configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
         *        successfully launched and reported back to Amazon GameLift that it is ready to host a game session.
         *        The instance is now considered ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce
         *        capacity during a scaling down event or to recycle resources in the event of a problem.
         *        </p>
         *        </li>
         * @see InstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceStatus
         */
        Builder status(InstanceStatus status);

        /**
         * <p>
         * Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param creationTime
         *        Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         *        milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String instanceId;

        private String ipAddress;

        private String dnsName;

        private String operatingSystem;

        private String type;

        private String status;

        private Instant creationTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Instance model) {
            fleetId(model.fleetId);
            instanceId(model.instanceId);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            operatingSystem(model.operatingSystem);
            type(model.type);
            status(model.status);
            creationTime(model.creationTime);
        }

        public final String getFleetId() {
            return fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final String getOperatingSystemAsString() {
            return operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(EC2InstanceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(InstanceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public Instance build() {
            return new Instance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
