/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for Lustre root squash used to restrict root-level access from clients that try to access your FSx
 * for Lustre file system as root. Use the <code>RootSquash</code> parameter to enable root squash. To learn more about
 * Lustre root squash, see <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/root-squash.html">Lustre root
 * squash</a>.
 * </p>
 * <p>
 * You can also use the <code>NoSquashNids</code> parameter to provide an array of clients who are not affected by the
 * root squash setting. These clients will access the file system as root, with unrestricted privileges.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LustreRootSquashConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LustreRootSquashConfiguration.Builder, LustreRootSquashConfiguration> {
    private static final SdkField<String> ROOT_SQUASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RootSquash").getter(getter(LustreRootSquashConfiguration::rootSquash))
            .setter(setter(Builder::rootSquash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootSquash").build()).build();

    private static final SdkField<List<String>> NO_SQUASH_NIDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NoSquashNids")
            .getter(getter(LustreRootSquashConfiguration::noSquashNids))
            .setter(setter(Builder::noSquashNids))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NoSquashNids").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROOT_SQUASH_FIELD,
            NO_SQUASH_NIDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String rootSquash;

    private final List<String> noSquashNids;

    private LustreRootSquashConfiguration(BuilderImpl builder) {
        this.rootSquash = builder.rootSquash;
        this.noSquashNids = builder.noSquashNids;
    }

    /**
     * <p>
     * You enable root squash by setting a user ID (UID) and group ID (GID) for the file system in the format
     * <code>UID:GID</code> (for example, <code>365534:65534</code>). The UID and GID values can range from
     * <code>0</code> to <code>4294967294</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A non-zero value for UID and GID enables root squash. The UID and GID values can be different, but each must be a
     * non-zero value.
     * </p>
     * </li>
     * <li>
     * <p>
     * A value of <code>0</code> (zero) for UID and GID indicates root, and therefore disables root squash.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When root squash is enabled, the user ID and group ID of a root user accessing the file system are re-mapped to
     * the UID and GID you provide.
     * </p>
     * 
     * @return You enable root squash by setting a user ID (UID) and group ID (GID) for the file system in the format
     *         <code>UID:GID</code> (for example, <code>365534:65534</code>). The UID and GID values can range from
     *         <code>0</code> to <code>4294967294</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A non-zero value for UID and GID enables root squash. The UID and GID values can be different, but each
     *         must be a non-zero value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A value of <code>0</code> (zero) for UID and GID indicates root, and therefore disables root squash.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When root squash is enabled, the user ID and group ID of a root user accessing the file system are
     *         re-mapped to the UID and GID you provide.
     */
    public final String rootSquash() {
        return rootSquash;
    }

    /**
     * For responses, this returns true if the service returned a value for the NoSquashNids property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNoSquashNids() {
        return noSquashNids != null && !(noSquashNids instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * When root squash is enabled, you can optionally specify an array of NIDs of clients for which root squash does
     * not apply. A client NID is a Lustre Network Identifier used to uniquely identify a client. You can specify the
     * NID as either a single address or a range of addresses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A single address is described in standard Lustre NID format by specifying the client’s IP address followed by the
     * Lustre network ID (for example, <code>10.0.1.6@tcp</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * An address range is described using a dash to separate the range (for example,
     * <code>10.0.[2-10].[1-255]@tcp</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNoSquashNids} method.
     * </p>
     * 
     * @return When root squash is enabled, you can optionally specify an array of NIDs of clients for which root squash
     *         does not apply. A client NID is a Lustre Network Identifier used to uniquely identify a client. You can
     *         specify the NID as either a single address or a range of addresses:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A single address is described in standard Lustre NID format by specifying the client’s IP address
     *         followed by the Lustre network ID (for example, <code>10.0.1.6@tcp</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An address range is described using a dash to separate the range (for example,
     *         <code>10.0.[2-10].[1-255]@tcp</code>).
     *         </p>
     *         </li>
     */
    public final List<String> noSquashNids() {
        return noSquashNids;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(rootSquash());
        hashCode = 31 * hashCode + Objects.hashCode(hasNoSquashNids() ? noSquashNids() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LustreRootSquashConfiguration)) {
            return false;
        }
        LustreRootSquashConfiguration other = (LustreRootSquashConfiguration) obj;
        return Objects.equals(rootSquash(), other.rootSquash()) && hasNoSquashNids() == other.hasNoSquashNids()
                && Objects.equals(noSquashNids(), other.noSquashNids());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LustreRootSquashConfiguration").add("RootSquash", rootSquash())
                .add("NoSquashNids", hasNoSquashNids() ? noSquashNids() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RootSquash":
            return Optional.ofNullable(clazz.cast(rootSquash()));
        case "NoSquashNids":
            return Optional.ofNullable(clazz.cast(noSquashNids()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RootSquash", ROOT_SQUASH_FIELD);
        map.put("NoSquashNids", NO_SQUASH_NIDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LustreRootSquashConfiguration, T> g) {
        return obj -> g.apply((LustreRootSquashConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LustreRootSquashConfiguration> {
        /**
         * <p>
         * You enable root squash by setting a user ID (UID) and group ID (GID) for the file system in the format
         * <code>UID:GID</code> (for example, <code>365534:65534</code>). The UID and GID values can range from
         * <code>0</code> to <code>4294967294</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A non-zero value for UID and GID enables root squash. The UID and GID values can be different, but each must
         * be a non-zero value.
         * </p>
         * </li>
         * <li>
         * <p>
         * A value of <code>0</code> (zero) for UID and GID indicates root, and therefore disables root squash.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When root squash is enabled, the user ID and group ID of a root user accessing the file system are re-mapped
         * to the UID and GID you provide.
         * </p>
         * 
         * @param rootSquash
         *        You enable root squash by setting a user ID (UID) and group ID (GID) for the file system in the format
         *        <code>UID:GID</code> (for example, <code>365534:65534</code>). The UID and GID values can range from
         *        <code>0</code> to <code>4294967294</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A non-zero value for UID and GID enables root squash. The UID and GID values can be different, but
         *        each must be a non-zero value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A value of <code>0</code> (zero) for UID and GID indicates root, and therefore disables root squash.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When root squash is enabled, the user ID and group ID of a root user accessing the file system are
         *        re-mapped to the UID and GID you provide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootSquash(String rootSquash);

        /**
         * <p>
         * When root squash is enabled, you can optionally specify an array of NIDs of clients for which root squash
         * does not apply. A client NID is a Lustre Network Identifier used to uniquely identify a client. You can
         * specify the NID as either a single address or a range of addresses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A single address is described in standard Lustre NID format by specifying the client’s IP address followed by
         * the Lustre network ID (for example, <code>10.0.1.6@tcp</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * An address range is described using a dash to separate the range (for example,
         * <code>10.0.[2-10].[1-255]@tcp</code>).
         * </p>
         * </li>
         * </ul>
         * 
         * @param noSquashNids
         *        When root squash is enabled, you can optionally specify an array of NIDs of clients for which root
         *        squash does not apply. A client NID is a Lustre Network Identifier used to uniquely identify a client.
         *        You can specify the NID as either a single address or a range of addresses:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A single address is described in standard Lustre NID format by specifying the client’s IP address
         *        followed by the Lustre network ID (for example, <code>10.0.1.6@tcp</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An address range is described using a dash to separate the range (for example,
         *        <code>10.0.[2-10].[1-255]@tcp</code>).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noSquashNids(Collection<String> noSquashNids);

        /**
         * <p>
         * When root squash is enabled, you can optionally specify an array of NIDs of clients for which root squash
         * does not apply. A client NID is a Lustre Network Identifier used to uniquely identify a client. You can
         * specify the NID as either a single address or a range of addresses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A single address is described in standard Lustre NID format by specifying the client’s IP address followed by
         * the Lustre network ID (for example, <code>10.0.1.6@tcp</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * An address range is described using a dash to separate the range (for example,
         * <code>10.0.[2-10].[1-255]@tcp</code>).
         * </p>
         * </li>
         * </ul>
         * 
         * @param noSquashNids
         *        When root squash is enabled, you can optionally specify an array of NIDs of clients for which root
         *        squash does not apply. A client NID is a Lustre Network Identifier used to uniquely identify a client.
         *        You can specify the NID as either a single address or a range of addresses:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A single address is described in standard Lustre NID format by specifying the client’s IP address
         *        followed by the Lustre network ID (for example, <code>10.0.1.6@tcp</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An address range is described using a dash to separate the range (for example,
         *        <code>10.0.[2-10].[1-255]@tcp</code>).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noSquashNids(String... noSquashNids);
    }

    static final class BuilderImpl implements Builder {
        private String rootSquash;

        private List<String> noSquashNids = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LustreRootSquashConfiguration model) {
            rootSquash(model.rootSquash);
            noSquashNids(model.noSquashNids);
        }

        public final String getRootSquash() {
            return rootSquash;
        }

        public final void setRootSquash(String rootSquash) {
            this.rootSquash = rootSquash;
        }

        @Override
        public final Builder rootSquash(String rootSquash) {
            this.rootSquash = rootSquash;
            return this;
        }

        public final Collection<String> getNoSquashNids() {
            if (noSquashNids instanceof SdkAutoConstructList) {
                return null;
            }
            return noSquashNids;
        }

        public final void setNoSquashNids(Collection<String> noSquashNids) {
            this.noSquashNids = LustreNoSquashNidsCopier.copy(noSquashNids);
        }

        @Override
        public final Builder noSquashNids(Collection<String> noSquashNids) {
            this.noSquashNids = LustreNoSquashNidsCopier.copy(noSquashNids);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder noSquashNids(String... noSquashNids) {
            noSquashNids(Arrays.asList(noSquashNids));
            return this;
        }

        @Override
        public LustreRootSquashConfiguration build() {
            return new LustreRootSquashConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
