/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The path to the file(s) in an Amazon Simple Storage Service (Amazon S3) bucket, and an AWS Identity and Access
 * Management (IAM) role that Amazon Forecast can assume to access the file(s). Optionally, includes an AWS Key
 * Management Service (KMS) key. This object is part of the <a>DataSource</a> object that is submitted in the
 * <a>CreateDatasetImportJob</a> request, and part of the <a>DataDestination</a> object that is submitted in the
 * <a>CreateForecastExportJob</a> request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Config implements SdkPojo, Serializable, ToCopyableBuilder<S3Config.Builder, S3Config> {
    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Config::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Config::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Config::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KMSKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATH_FIELD, ROLE_ARN_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String path;

    private final String roleArn;

    private final String kmsKeyArn;

    private S3Config(BuilderImpl builder) {
        this.path = builder.path;
        this.roleArn = builder.roleArn;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The path to an Amazon Simple Storage Service (Amazon S3) bucket or file(s) in an Amazon S3 bucket.
     * </p>
     * 
     * @return The path to an Amazon Simple Storage Service (Amazon S3) bucket or file(s) in an Amazon S3 bucket.
     */
    public String path() {
        return path;
    }

    /**
     * <p>
     * The ARN of the AWS Identity and Access Management (IAM) role that Amazon Forecast can assume to access the Amazon
     * S3 bucket or files. If you provide a value for the <code>KMSKeyArn</code> key, the role must allow access to the
     * key.
     * </p>
     * <p>
     * Passing a role across AWS accounts is not allowed. If you pass a role that isn't in your account, you get an
     * <code>InvalidInputException</code> error.
     * </p>
     * 
     * @return The ARN of the AWS Identity and Access Management (IAM) role that Amazon Forecast can assume to access
     *         the Amazon S3 bucket or files. If you provide a value for the <code>KMSKeyArn</code> key, the role must
     *         allow access to the key.</p>
     *         <p>
     *         Passing a role across AWS accounts is not allowed. If you pass a role that isn't in your account, you get
     *         an <code>InvalidInputException</code> error.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an AWS Key Management Service (KMS) key.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an AWS Key Management Service (KMS) key.
     */
    public String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Config)) {
            return false;
        }
        S3Config other = (S3Config) obj;
        return Objects.equals(path(), other.path()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3Config").add("Path", path()).add("RoleArn", roleArn()).add("KMSKeyArn", kmsKeyArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "KMSKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Config, T> g) {
        return obj -> g.apply((S3Config) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Config> {
        /**
         * <p>
         * The path to an Amazon Simple Storage Service (Amazon S3) bucket or file(s) in an Amazon S3 bucket.
         * </p>
         * 
         * @param path
         *        The path to an Amazon Simple Storage Service (Amazon S3) bucket or file(s) in an Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * The ARN of the AWS Identity and Access Management (IAM) role that Amazon Forecast can assume to access the
         * Amazon S3 bucket or files. If you provide a value for the <code>KMSKeyArn</code> key, the role must allow
         * access to the key.
         * </p>
         * <p>
         * Passing a role across AWS accounts is not allowed. If you pass a role that isn't in your account, you get an
         * <code>InvalidInputException</code> error.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the AWS Identity and Access Management (IAM) role that Amazon Forecast can assume to access
         *        the Amazon S3 bucket or files. If you provide a value for the <code>KMSKeyArn</code> key, the role
         *        must allow access to the key.</p>
         *        <p>
         *        Passing a role across AWS accounts is not allowed. If you pass a role that isn't in your account, you
         *        get an <code>InvalidInputException</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an AWS Key Management Service (KMS) key.
         * </p>
         * 
         * @param kmsKeyArn
         *        The Amazon Resource Name (ARN) of an AWS Key Management Service (KMS) key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String path;

        private String roleArn;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Config model) {
            path(model.path);
            roleArn(model.roleArn);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public S3Config build() {
            return new S3Config(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
