/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the compliance status for the account. An account is considered noncompliant if it includes resources that
 * are not protected by the specified policy or that don't comply with the policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EvaluationResult implements SdkPojo, Serializable,
        ToCopyableBuilder<EvaluationResult.Builder, EvaluationResult> {
    private static final SdkField<String> COMPLIANCE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EvaluationResult::complianceStatusAsString)).setter(setter(Builder::complianceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceStatus").build()).build();

    private static final SdkField<Long> VIOLATOR_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(EvaluationResult::violatorCount)).setter(setter(Builder::violatorCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViolatorCount").build()).build();

    private static final SdkField<Boolean> EVALUATION_LIMIT_EXCEEDED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(EvaluationResult::evaluationLimitExceeded)).setter(setter(Builder::evaluationLimitExceeded))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationLimitExceeded").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPLIANCE_STATUS_FIELD,
            VIOLATOR_COUNT_FIELD, EVALUATION_LIMIT_EXCEEDED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String complianceStatus;

    private final Long violatorCount;

    private final Boolean evaluationLimitExceeded;

    private EvaluationResult(BuilderImpl builder) {
        this.complianceStatus = builder.complianceStatus;
        this.violatorCount = builder.violatorCount;
        this.evaluationLimitExceeded = builder.evaluationLimitExceeded;
    }

    /**
     * <p>
     * Describes an AWS account's compliance with the AWS Firewall Manager policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceStatus}
     * will return {@link PolicyComplianceStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #complianceStatusAsString}.
     * </p>
     * 
     * @return Describes an AWS account's compliance with the AWS Firewall Manager policy.
     * @see PolicyComplianceStatusType
     */
    public PolicyComplianceStatusType complianceStatus() {
        return PolicyComplianceStatusType.fromValue(complianceStatus);
    }

    /**
     * <p>
     * Describes an AWS account's compliance with the AWS Firewall Manager policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceStatus}
     * will return {@link PolicyComplianceStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #complianceStatusAsString}.
     * </p>
     * 
     * @return Describes an AWS account's compliance with the AWS Firewall Manager policy.
     * @see PolicyComplianceStatusType
     */
    public String complianceStatusAsString() {
        return complianceStatus;
    }

    /**
     * <p>
     * The number of resources that are noncompliant with the specified policy. For AWS WAF and Shield Advanced
     * policies, a resource is considered noncompliant if it is not associated with the policy. For security group
     * policies, a resource is considered noncompliant if it doesn't comply with the rules of the policy and remediation
     * is disabled or not possible.
     * </p>
     * 
     * @return The number of resources that are noncompliant with the specified policy. For AWS WAF and Shield Advanced
     *         policies, a resource is considered noncompliant if it is not associated with the policy. For security
     *         group policies, a resource is considered noncompliant if it doesn't comply with the rules of the policy
     *         and remediation is disabled or not possible.
     */
    public Long violatorCount() {
        return violatorCount;
    }

    /**
     * <p>
     * Indicates that over 100 resources are noncompliant with the AWS Firewall Manager policy.
     * </p>
     * 
     * @return Indicates that over 100 resources are noncompliant with the AWS Firewall Manager policy.
     */
    public Boolean evaluationLimitExceeded() {
        return evaluationLimitExceeded;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(complianceStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(violatorCount());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationLimitExceeded());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EvaluationResult)) {
            return false;
        }
        EvaluationResult other = (EvaluationResult) obj;
        return Objects.equals(complianceStatusAsString(), other.complianceStatusAsString())
                && Objects.equals(violatorCount(), other.violatorCount())
                && Objects.equals(evaluationLimitExceeded(), other.evaluationLimitExceeded());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EvaluationResult").add("ComplianceStatus", complianceStatusAsString())
                .add("ViolatorCount", violatorCount()).add("EvaluationLimitExceeded", evaluationLimitExceeded()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComplianceStatus":
            return Optional.ofNullable(clazz.cast(complianceStatusAsString()));
        case "ViolatorCount":
            return Optional.ofNullable(clazz.cast(violatorCount()));
        case "EvaluationLimitExceeded":
            return Optional.ofNullable(clazz.cast(evaluationLimitExceeded()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EvaluationResult, T> g) {
        return obj -> g.apply((EvaluationResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EvaluationResult> {
        /**
         * <p>
         * Describes an AWS account's compliance with the AWS Firewall Manager policy.
         * </p>
         * 
         * @param complianceStatus
         *        Describes an AWS account's compliance with the AWS Firewall Manager policy.
         * @see PolicyComplianceStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyComplianceStatusType
         */
        Builder complianceStatus(String complianceStatus);

        /**
         * <p>
         * Describes an AWS account's compliance with the AWS Firewall Manager policy.
         * </p>
         * 
         * @param complianceStatus
         *        Describes an AWS account's compliance with the AWS Firewall Manager policy.
         * @see PolicyComplianceStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyComplianceStatusType
         */
        Builder complianceStatus(PolicyComplianceStatusType complianceStatus);

        /**
         * <p>
         * The number of resources that are noncompliant with the specified policy. For AWS WAF and Shield Advanced
         * policies, a resource is considered noncompliant if it is not associated with the policy. For security group
         * policies, a resource is considered noncompliant if it doesn't comply with the rules of the policy and
         * remediation is disabled or not possible.
         * </p>
         * 
         * @param violatorCount
         *        The number of resources that are noncompliant with the specified policy. For AWS WAF and Shield
         *        Advanced policies, a resource is considered noncompliant if it is not associated with the policy. For
         *        security group policies, a resource is considered noncompliant if it doesn't comply with the rules of
         *        the policy and remediation is disabled or not possible.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder violatorCount(Long violatorCount);

        /**
         * <p>
         * Indicates that over 100 resources are noncompliant with the AWS Firewall Manager policy.
         * </p>
         * 
         * @param evaluationLimitExceeded
         *        Indicates that over 100 resources are noncompliant with the AWS Firewall Manager policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationLimitExceeded(Boolean evaluationLimitExceeded);
    }

    static final class BuilderImpl implements Builder {
        private String complianceStatus;

        private Long violatorCount;

        private Boolean evaluationLimitExceeded;

        private BuilderImpl() {
        }

        private BuilderImpl(EvaluationResult model) {
            complianceStatus(model.complianceStatus);
            violatorCount(model.violatorCount);
            evaluationLimitExceeded(model.evaluationLimitExceeded);
        }

        public final String getComplianceStatus() {
            return complianceStatus;
        }

        @Override
        public final Builder complianceStatus(String complianceStatus) {
            this.complianceStatus = complianceStatus;
            return this;
        }

        @Override
        public final Builder complianceStatus(PolicyComplianceStatusType complianceStatus) {
            this.complianceStatus(complianceStatus == null ? null : complianceStatus.toString());
            return this;
        }

        public final void setComplianceStatus(String complianceStatus) {
            this.complianceStatus = complianceStatus;
        }

        public final Long getViolatorCount() {
            return violatorCount;
        }

        @Override
        public final Builder violatorCount(Long violatorCount) {
            this.violatorCount = violatorCount;
            return this;
        }

        public final void setViolatorCount(Long violatorCount) {
            this.violatorCount = violatorCount;
        }

        public final Boolean getEvaluationLimitExceeded() {
            return evaluationLimitExceeded;
        }

        @Override
        public final Builder evaluationLimitExceeded(Boolean evaluationLimitExceeded) {
            this.evaluationLimitExceeded = evaluationLimitExceeded;
            return this;
        }

        public final void setEvaluationLimitExceeded(Boolean evaluationLimitExceeded) {
            this.evaluationLimitExceeded = evaluationLimitExceeded;
        }

        @Override
        public EvaluationResult build() {
            return new EvaluationResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
