/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evs;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.evs.model.AssociateEipToVlanRequest;
import software.amazon.awssdk.services.evs.model.AssociateEipToVlanResponse;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentHostRequest;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentHostResponse;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentHostRequest;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentHostResponse;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.evs.model.DisassociateEipFromVlanRequest;
import software.amazon.awssdk.services.evs.model.DisassociateEipFromVlanResponse;
import software.amazon.awssdk.services.evs.model.EvsException;
import software.amazon.awssdk.services.evs.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.evs.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest;
import software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse;
import software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest;
import software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse;
import software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.evs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.evs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.evs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.evs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.evs.model.TagPolicyException;
import software.amazon.awssdk.services.evs.model.TagResourceRequest;
import software.amazon.awssdk.services.evs.model.TagResourceResponse;
import software.amazon.awssdk.services.evs.model.ThrottlingException;
import software.amazon.awssdk.services.evs.model.TooManyTagsException;
import software.amazon.awssdk.services.evs.model.UntagResourceRequest;
import software.amazon.awssdk.services.evs.model.UntagResourceResponse;
import software.amazon.awssdk.services.evs.model.ValidationException;
import software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable;
import software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable;
import software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable;

/**
 * Service client for accessing EVS. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Amazon Elastic VMware Service (Amazon EVS) is a service that you can use to deploy a VMware Cloud Foundation (VCF)
 * software environment directly on EC2 bare metal instances within an Amazon Virtual Private Cloud (VPC).
 * </p>
 * <p>
 * Workloads running on Amazon EVS are fully compatible with workloads running on any standard VMware vSphere
 * environment. This means that you can migrate any VMware-based workload to Amazon EVS without workload modification.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface EvsClient extends AwsClient {
    String SERVICE_NAME = "evs";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "evs";

    /**
     * <p>
     * Associates an Elastic IP address with a public HCX VLAN. This operation is only allowed for public HCX VLANs at
     * this time.
     * </p>
     *
     * @param associateEipToVlanRequest
     * @return Result of the AssociateEipToVlan operation returned by the service.
     * @throws ThrottlingException
     *         The operation couldn't be performed because the service is throttling requests. This exception is thrown
     *         when there are too many requests accepted concurrently from the service endpoint.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.AssociateEipToVlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/AssociateEipToVlan" target="_top">AWS API
     *      Documentation</a>
     */
    default AssociateEipToVlanResponse associateEipToVlan(AssociateEipToVlanRequest associateEipToVlanRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates an Elastic IP address with a public HCX VLAN. This operation is only allowed for public HCX VLANs at
     * this time.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AssociateEipToVlanRequest.Builder} avoiding the
     * need to create one manually via {@link AssociateEipToVlanRequest#builder()}
     * </p>
     *
     * @param associateEipToVlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.AssociateEipToVlanRequest.Builder} to create a request.
     * @return Result of the AssociateEipToVlan operation returned by the service.
     * @throws ThrottlingException
     *         The operation couldn't be performed because the service is throttling requests. This exception is thrown
     *         when there are too many requests accepted concurrently from the service endpoint.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.AssociateEipToVlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/AssociateEipToVlan" target="_top">AWS API
     *      Documentation</a>
     */
    default AssociateEipToVlanResponse associateEipToVlan(Consumer<AssociateEipToVlanRequest.Builder> associateEipToVlanRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            EvsException {
        return associateEipToVlan(AssociateEipToVlanRequest.builder().applyMutation(associateEipToVlanRequest).build());
    }

    /**
     * <p>
     * Creates an Amazon EVS environment that runs VCF software, such as SDDC Manager, NSX Manager, and vCenter Server.
     * </p>
     * <p>
     * During environment creation, Amazon EVS performs validations on DNS settings, provisions VLAN subnets and hosts,
     * and deploys the supplied version of VCF.
     * </p>
     * <p>
     * It can take several hours to create an environment. After the deployment completes, you can configure VCF in the
     * vSphere user interface according to your needs.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironment</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note>
     *
     * @param createEnvironmentRequest
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws ValidationException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Amazon EVS environment that runs VCF software, such as SDDC Manager, NSX Manager, and vCenter Server.
     * </p>
     * <p>
     * During environment creation, Amazon EVS performs validations on DNS settings, provisions VLAN subnets and hosts,
     * and deploys the supplied version of VCF.
     * </p>
     * <p>
     * It can take several hours to create an environment. After the deployment completes, you can configure VCF in the
     * vSphere user interface according to your needs.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironment</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link CreateEnvironmentRequest#builder()}
     * </p>
     *
     * @param createEnvironmentRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.CreateEnvironmentRequest.Builder} to create a request.
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEnvironmentResponse createEnvironment(Consumer<CreateEnvironmentRequest.Builder> createEnvironmentRequest)
            throws ValidationException, AwsServiceException, SdkClientException, EvsException {
        return createEnvironment(CreateEnvironmentRequest.builder().applyMutation(createEnvironmentRequest).build());
    }

    /**
     * <p>
     * Creates an ESXi host and adds it to an Amazon EVS environment. Amazon EVS supports 4-16 hosts per environment.
     * </p>
     * <p>
     * This action can only be used after the Amazon EVS environment is deployed.
     * </p>
     * <p>
     * You can use the <code>dedicatedHostId</code> parameter to specify an Amazon EC2 Dedicated Host for ESXi host
     * creation.
     * </p>
     * <p>
     * You can use the <code>placementGroupId</code> parameter to specify a cluster or partition placement group to
     * launch EC2 instances into.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironmentHost</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note>
     *
     * @param createEnvironmentHostRequest
     * @return Result of the CreateEnvironmentHost operation returned by the service.
     * @throws ThrottlingException
     *         The operation couldn't be performed because the service is throttling requests. This exception is thrown
     *         when there are too many requests accepted concurrently from the service endpoint.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.CreateEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEnvironmentHostResponse createEnvironmentHost(CreateEnvironmentHostRequest createEnvironmentHostRequest)
            throws ThrottlingException, ValidationException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an ESXi host and adds it to an Amazon EVS environment. Amazon EVS supports 4-16 hosts per environment.
     * </p>
     * <p>
     * This action can only be used after the Amazon EVS environment is deployed.
     * </p>
     * <p>
     * You can use the <code>dedicatedHostId</code> parameter to specify an Amazon EC2 Dedicated Host for ESXi host
     * creation.
     * </p>
     * <p>
     * You can use the <code>placementGroupId</code> parameter to specify a cluster or partition placement group to
     * launch EC2 instances into.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironmentHost</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEnvironmentHostRequest.Builder} avoiding the
     * need to create one manually via {@link CreateEnvironmentHostRequest#builder()}
     * </p>
     *
     * @param createEnvironmentHostRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.CreateEnvironmentHostRequest.Builder} to create a
     *        request.
     * @return Result of the CreateEnvironmentHost operation returned by the service.
     * @throws ThrottlingException
     *         The operation couldn't be performed because the service is throttling requests. This exception is thrown
     *         when there are too many requests accepted concurrently from the service endpoint.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.CreateEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEnvironmentHostResponse createEnvironmentHost(
            Consumer<CreateEnvironmentHostRequest.Builder> createEnvironmentHostRequest) throws ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, EvsException {
        return createEnvironmentHost(CreateEnvironmentHostRequest.builder().applyMutation(createEnvironmentHostRequest).build());
    }

    /**
     * <p>
     * Deletes an Amazon EVS environment.
     * </p>
     * <p>
     * Amazon EVS environments will only be enabled for deletion once the hosts are deleted. You can delete hosts using
     * the <code>DeleteEnvironmentHost</code> action.
     * </p>
     * <p>
     * Environment deletion also deletes the associated Amazon EVS VLAN subnets and Amazon Web Services Secrets Manager
     * secrets that Amazon EVS created. Amazon Web Services resources that you create are not deleted. These resources
     * may continue to incur costs.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an Amazon EVS environment.
     * </p>
     * <p>
     * Amazon EVS environments will only be enabled for deletion once the hosts are deleted. You can delete hosts using
     * the <code>DeleteEnvironmentHost</code> action.
     * </p>
     * <p>
     * Environment deletion also deletes the associated Amazon EVS VLAN subnets and Amazon Web Services Secrets Manager
     * secrets that Amazon EVS created. Amazon Web Services resources that you create are not deleted. These resources
     * may continue to incur costs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteEnvironmentRequest#builder()}
     * </p>
     *
     * @param deleteEnvironmentRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.DeleteEnvironmentRequest.Builder} to create a request.
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEnvironmentResponse deleteEnvironment(Consumer<DeleteEnvironmentRequest.Builder> deleteEnvironmentRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return deleteEnvironment(DeleteEnvironmentRequest.builder().applyMutation(deleteEnvironmentRequest).build());
    }

    /**
     * <p>
     * Deletes a host from an Amazon EVS environment.
     * </p>
     * <note>
     * <p>
     * Before deleting a host, you must unassign and decommission the host from within the SDDC Manager user interface.
     * Not doing so could impact the availability of your virtual machines or result in data loss.
     * </p>
     * </note>
     *
     * @param deleteEnvironmentHostRequest
     * @return Result of the DeleteEnvironmentHost operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.DeleteEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEnvironmentHostResponse deleteEnvironmentHost(DeleteEnvironmentHostRequest deleteEnvironmentHostRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a host from an Amazon EVS environment.
     * </p>
     * <note>
     * <p>
     * Before deleting a host, you must unassign and decommission the host from within the SDDC Manager user interface.
     * Not doing so could impact the availability of your virtual machines or result in data loss.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEnvironmentHostRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteEnvironmentHostRequest#builder()}
     * </p>
     *
     * @param deleteEnvironmentHostRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.DeleteEnvironmentHostRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteEnvironmentHost operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.DeleteEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEnvironmentHostResponse deleteEnvironmentHost(
            Consumer<DeleteEnvironmentHostRequest.Builder> deleteEnvironmentHostRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return deleteEnvironmentHost(DeleteEnvironmentHostRequest.builder().applyMutation(deleteEnvironmentHostRequest).build());
    }

    /**
     * <p>
     * Disassociates an Elastic IP address from a public HCX VLAN. This operation is only allowed for public HCX VLANs
     * at this time.
     * </p>
     *
     * @param disassociateEipFromVlanRequest
     * @return Result of the DisassociateEipFromVlan operation returned by the service.
     * @throws ThrottlingException
     *         The operation couldn't be performed because the service is throttling requests. This exception is thrown
     *         when there are too many requests accepted concurrently from the service endpoint.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.DisassociateEipFromVlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DisassociateEipFromVlan" target="_top">AWS
     *      API Documentation</a>
     */
    default DisassociateEipFromVlanResponse disassociateEipFromVlan(DisassociateEipFromVlanRequest disassociateEipFromVlanRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disassociates an Elastic IP address from a public HCX VLAN. This operation is only allowed for public HCX VLANs
     * at this time.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisassociateEipFromVlanRequest.Builder} avoiding
     * the need to create one manually via {@link DisassociateEipFromVlanRequest#builder()}
     * </p>
     *
     * @param disassociateEipFromVlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.DisassociateEipFromVlanRequest.Builder} to create a
     *        request.
     * @return Result of the DisassociateEipFromVlan operation returned by the service.
     * @throws ThrottlingException
     *         The operation couldn't be performed because the service is throttling requests. This exception is thrown
     *         when there are too many requests accepted concurrently from the service endpoint.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.DisassociateEipFromVlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DisassociateEipFromVlan" target="_top">AWS
     *      API Documentation</a>
     */
    default DisassociateEipFromVlanResponse disassociateEipFromVlan(
            Consumer<DisassociateEipFromVlanRequest.Builder> disassociateEipFromVlanRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return disassociateEipFromVlan(DisassociateEipFromVlanRequest.builder().applyMutation(disassociateEipFromVlanRequest)
                .build());
    }

    /**
     * <p>
     * Returns a description of the specified environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a description of the specified environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetEnvironmentRequest.Builder} avoiding the need to
     * create one manually via {@link GetEnvironmentRequest#builder()}
     * </p>
     *
     * @param getEnvironmentRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.GetEnvironmentRequest.Builder} to create a request.
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default GetEnvironmentResponse getEnvironment(Consumer<GetEnvironmentRequest.Builder> getEnvironmentRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return getEnvironment(GetEnvironmentRequest.builder().applyMutation(getEnvironmentRequest).build());
    }

    /**
     * <p>
     * List the hosts within an environment.
     * </p>
     *
     * @param listEnvironmentHostsRequest
     * @return Result of the ListEnvironmentHosts operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentHostsResponse listEnvironmentHosts(ListEnvironmentHostsRequest listEnvironmentHostsRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List the hosts within an environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentHostsRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentHostsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentHostsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest.Builder} to create a request.
     * @return Result of the ListEnvironmentHosts operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentHostsResponse listEnvironmentHosts(
            Consumer<ListEnvironmentHostsRequest.Builder> listEnvironmentHostsRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return listEnvironmentHosts(ListEnvironmentHostsRequest.builder().applyMutation(listEnvironmentHostsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable responses = client.listEnvironmentHostsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable responses = client
     *             .listEnvironmentHostsPaginator(request);
     *     for (software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable responses = client.listEnvironmentHostsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnvironmentHostsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentHostsIterable listEnvironmentHostsPaginator(ListEnvironmentHostsRequest listEnvironmentHostsRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return new ListEnvironmentHostsIterable(this, listEnvironmentHostsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable responses = client.listEnvironmentHostsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable responses = client
     *             .listEnvironmentHostsPaginator(request);
     *     for (software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsIterable responses = client.listEnvironmentHostsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentHostsRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentHostsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentHostsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentHostsIterable listEnvironmentHostsPaginator(
            Consumer<ListEnvironmentHostsRequest.Builder> listEnvironmentHostsRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return listEnvironmentHostsPaginator(ListEnvironmentHostsRequest.builder().applyMutation(listEnvironmentHostsRequest)
                .build());
    }

    /**
     * <p>
     * Lists environment VLANs that are associated with the specified environment.
     * </p>
     *
     * @param listEnvironmentVlansRequest
     * @return Result of the ListEnvironmentVlans operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentVlansResponse listEnvironmentVlans(ListEnvironmentVlansRequest listEnvironmentVlansRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists environment VLANs that are associated with the specified environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentVlansRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentVlansRequest#builder()}
     * </p>
     *
     * @param listEnvironmentVlansRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest.Builder} to create a request.
     * @return Result of the ListEnvironmentVlans operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentVlansResponse listEnvironmentVlans(
            Consumer<ListEnvironmentVlansRequest.Builder> listEnvironmentVlansRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return listEnvironmentVlans(ListEnvironmentVlansRequest.builder().applyMutation(listEnvironmentVlansRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable responses = client.listEnvironmentVlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable responses = client
     *             .listEnvironmentVlansPaginator(request);
     *     for (software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable responses = client.listEnvironmentVlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnvironmentVlansRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentVlansIterable listEnvironmentVlansPaginator(ListEnvironmentVlansRequest listEnvironmentVlansRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return new ListEnvironmentVlansIterable(this, listEnvironmentVlansRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable responses = client.listEnvironmentVlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable responses = client
     *             .listEnvironmentVlansPaginator(request);
     *     for (software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansIterable responses = client.listEnvironmentVlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentVlansRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentVlansRequest#builder()}
     * </p>
     *
     * @param listEnvironmentVlansRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentVlansIterable listEnvironmentVlansPaginator(
            Consumer<ListEnvironmentVlansRequest.Builder> listEnvironmentVlansRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        return listEnvironmentVlansPaginator(ListEnvironmentVlansRequest.builder().applyMutation(listEnvironmentVlansRequest)
                .build());
    }

    /**
     * <p>
     * Lists the Amazon EVS environments in your Amazon Web Services account in the specified Amazon Web Services
     * Region.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ValidationException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Amazon EVS environments in your Amazon Web Services account in the specified Amazon Web Services
     * Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentsRequest.Builder} avoiding the need
     * to create one manually via {@link ListEnvironmentsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest.Builder} to create a request.
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsResponse listEnvironments(Consumer<ListEnvironmentsRequest.Builder> listEnvironmentsRequest)
            throws ValidationException, AwsServiceException, SdkClientException, EvsException {
        return listEnvironments(ListEnvironmentsRequest.builder().applyMutation(listEnvironmentsRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsIterable listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ValidationException, AwsServiceException, SdkClientException, EvsException {
        return new ListEnvironmentsIterable(this, listEnvironmentsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentsRequest.Builder} avoiding the need
     * to create one manually via {@link ListEnvironmentsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints. You will see this exception if invalid inputs are
     *         provided for any of the Amazon EVS environment operations, or if a list operation is performed on an
     *         environment resource that is still initializing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsIterable listEnvironmentsPaginator(Consumer<ListEnvironmentsRequest.Builder> listEnvironmentsRequest)
            throws ValidationException, AwsServiceException, SdkClientException, EvsException {
        return listEnvironmentsPaginator(ListEnvironmentsRequest.builder().applyMutation(listEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Lists the tags for an Amazon EVS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags for an Amazon EVS resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, EvsException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Associates the specified tags to an Amazon EVS resource with the specified <code>resourceArn</code>. If existing
     * tags on a resource are not specified in the request parameters, they aren't changed. When a resource is deleted,
     * the tags associated with that resource are also deleted. Tags that you create for Amazon EVS resources don't
     * propagate to any other resources associated with the environment. For example, if you tag an environment with
     * this operation, that tag doesn't automatically propagate to the VLAN subnets and hosts associated with the
     * environment.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The number of one or more Amazon EVS resources exceeds the maximum allowed. For a list of Amazon EVS
     *         quotas, see <a href="https://docs.aws.amazon.com/evs/latest/userguide/service-quotas-evs.html">Amazon EVS
     *         endpoints and quotas</a> in the <i>Amazon EVS User Guide</i>. Delete some resources or request an
     *         increase in your service quota. To request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">Amazon Web Services Service
     *         Quotas</a> in the <i>Amazon Web Services General Reference Guide</i>.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws TooManyTagsException
     *         <p>
     *         <code>TooManyTagsException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ServiceQuotaExceededException.html">
     *         <code>ServiceQuotaExceededException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         A service resource associated with the request has more than 200 tags.
     * @throws TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, TooManyTagsException, TagPolicyException, AwsServiceException, SdkClientException,
            EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates the specified tags to an Amazon EVS resource with the specified <code>resourceArn</code>. If existing
     * tags on a resource are not specified in the request parameters, they aren't changed. When a resource is deleted,
     * the tags associated with that resource are also deleted. Tags that you create for Amazon EVS resources don't
     * propagate to any other resources associated with the environment. For example, if you tag an environment with
     * this operation, that tag doesn't automatically propagate to the VLAN subnets and hosts associated with the
     * environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The number of one or more Amazon EVS resources exceeds the maximum allowed. For a list of Amazon EVS
     *         quotas, see <a href="https://docs.aws.amazon.com/evs/latest/userguide/service-quotas-evs.html">Amazon EVS
     *         endpoints and quotas</a> in the <i>Amazon EVS User Guide</i>. Delete some resources or request an
     *         increase in your service quota. To request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">Amazon Web Services Service
     *         Quotas</a> in the <i>Amazon Web Services General Reference Guide</i>.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws TooManyTagsException
     *         <p>
     *         <code>TooManyTagsException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ServiceQuotaExceededException.html">
     *         <code>ServiceQuotaExceededException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         A service resource associated with the request has more than 200 tags.
     * @throws TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, TooManyTagsException, TagPolicyException,
            AwsServiceException, SdkClientException, EvsException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Deletes specified tags from an Amazon EVS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            TagPolicyException, AwsServiceException, SdkClientException, EvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes specified tags from an Amazon EVS resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A service resource associated with the request could not be found. The resource might not be specified
     *         correctly, or it may have a <code>state</code> of <code>DELETED</code>.
     * @throws TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, TagPolicyException, AwsServiceException, SdkClientException, EvsException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * Create a {@link EvsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EvsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EvsClient}.
     */
    static EvsClientBuilder builder() {
        return new DefaultEvsClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default EvsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
