/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration specification for each instance type in an instance fleet.
 * </p>
 * <note>
 * <p>
 * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceTypeSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceTypeSpecification.Builder, InstanceTypeSpecification> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTypeSpecification::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> WEIGHTED_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceTypeSpecification::weightedCapacity)).setter(setter(Builder::weightedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeightedCapacity").build()).build();

    private static final SdkField<String> BID_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTypeSpecification::bidPrice)).setter(setter(Builder::bidPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BidPrice").build()).build();

    private static final SdkField<Double> BID_PRICE_AS_PERCENTAGE_OF_ON_DEMAND_PRICE_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(InstanceTypeSpecification::bidPriceAsPercentageOfOnDemandPrice))
            .setter(setter(Builder::bidPriceAsPercentageOfOnDemandPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("BidPriceAsPercentageOfOnDemandPrice").build()).build();

    private static final SdkField<List<Configuration>> CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .getter(getter(InstanceTypeSpecification::configurations))
            .setter(setter(Builder::configurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EbsBlockDevice>> EBS_BLOCK_DEVICES_FIELD = SdkField
            .<List<EbsBlockDevice>> builder(MarshallingType.LIST)
            .getter(getter(InstanceTypeSpecification::ebsBlockDevices))
            .setter(setter(Builder::ebsBlockDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsBlockDevices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EbsBlockDevice> builder(MarshallingType.SDK_POJO)
                                            .constructor(EbsBlockDevice::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> EBS_OPTIMIZED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(InstanceTypeSpecification::ebsOptimized)).setter(setter(Builder::ebsOptimized))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsOptimized").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            WEIGHTED_CAPACITY_FIELD, BID_PRICE_FIELD, BID_PRICE_AS_PERCENTAGE_OF_ON_DEMAND_PRICE_FIELD, CONFIGURATIONS_FIELD,
            EBS_BLOCK_DEVICES_FIELD, EBS_OPTIMIZED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Integer weightedCapacity;

    private final String bidPrice;

    private final Double bidPriceAsPercentageOfOnDemandPrice;

    private final List<Configuration> configurations;

    private final List<EbsBlockDevice> ebsBlockDevices;

    private final Boolean ebsOptimized;

    private InstanceTypeSpecification(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.weightedCapacity = builder.weightedCapacity;
        this.bidPrice = builder.bidPrice;
        this.bidPriceAsPercentageOfOnDemandPrice = builder.bidPriceAsPercentageOfOnDemandPrice;
        this.configurations = builder.configurations;
        this.ebsBlockDevices = builder.ebsBlockDevices;
        this.ebsOptimized = builder.ebsOptimized;
    }

    /**
     * <p>
     * The EC2 instance type, for example <code>m3.xlarge</code>.
     * </p>
     * 
     * @return The EC2 instance type, for example <code>m3.xlarge</code>.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The number of units that a provisioned instance of this type provides toward fulfilling the target capacities
     * defined in <a>InstanceFleetConfig</a>. Capacity values represent performance characteristics such as vCPUs,
     * memory, or I/O. If not specified, the default value is 1.
     * </p>
     * 
     * @return The number of units that a provisioned instance of this type provides toward fulfilling the target
     *         capacities defined in <a>InstanceFleetConfig</a>. Capacity values represent performance characteristics
     *         such as vCPUs, memory, or I/O. If not specified, the default value is 1.
     */
    public Integer weightedCapacity() {
        return weightedCapacity;
    }

    /**
     * <p>
     * The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD.
     * </p>
     * 
     * @return The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD.
     */
    public String bidPrice() {
        return bidPrice;
    }

    /**
     * <p>
     * The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
     * <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%).
     * </p>
     * 
     * @return The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
     *         <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%).
     */
    public Double bidPriceAsPercentageOfOnDemandPrice() {
        return bidPriceAsPercentageOfOnDemandPrice;
    }

    /**
     * Returns true if the Configurations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasConfigurations() {
        return configurations != null && !(configurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A configuration classification that applies when provisioning cluster instances, which can include configurations
     * for applications and software bundled with Amazon EMR.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfigurations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A configuration classification that applies when provisioning cluster instances, which can include
     *         configurations for applications and software bundled with Amazon EMR.
     */
    public List<Configuration> configurations() {
        return configurations;
    }

    /**
     * Returns true if the EbsBlockDevices property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEbsBlockDevices() {
        return ebsBlockDevices != null && !(ebsBlockDevices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
     * <code>InstanceType</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEbsBlockDevices()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
     *         <code>InstanceType</code>.
     */
    public List<EbsBlockDevice> ebsBlockDevices() {
        return ebsBlockDevices;
    }

    /**
     * <p>
     * Evaluates to <code>TRUE</code> when the specified <code>InstanceType</code> is EBS-optimized.
     * </p>
     * 
     * @return Evaluates to <code>TRUE</code> when the specified <code>InstanceType</code> is EBS-optimized.
     */
    public Boolean ebsOptimized() {
        return ebsOptimized;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(weightedCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(bidPrice());
        hashCode = 31 * hashCode + Objects.hashCode(bidPriceAsPercentageOfOnDemandPrice());
        hashCode = 31 * hashCode + Objects.hashCode(configurations());
        hashCode = 31 * hashCode + Objects.hashCode(ebsBlockDevices());
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptimized());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceTypeSpecification)) {
            return false;
        }
        InstanceTypeSpecification other = (InstanceTypeSpecification) obj;
        return Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(weightedCapacity(), other.weightedCapacity()) && Objects.equals(bidPrice(), other.bidPrice())
                && Objects.equals(bidPriceAsPercentageOfOnDemandPrice(), other.bidPriceAsPercentageOfOnDemandPrice())
                && Objects.equals(configurations(), other.configurations())
                && Objects.equals(ebsBlockDevices(), other.ebsBlockDevices())
                && Objects.equals(ebsOptimized(), other.ebsOptimized());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceTypeSpecification").add("InstanceType", instanceType())
                .add("WeightedCapacity", weightedCapacity()).add("BidPrice", bidPrice())
                .add("BidPriceAsPercentageOfOnDemandPrice", bidPriceAsPercentageOfOnDemandPrice())
                .add("Configurations", configurations()).add("EbsBlockDevices", ebsBlockDevices())
                .add("EbsOptimized", ebsOptimized()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "WeightedCapacity":
            return Optional.ofNullable(clazz.cast(weightedCapacity()));
        case "BidPrice":
            return Optional.ofNullable(clazz.cast(bidPrice()));
        case "BidPriceAsPercentageOfOnDemandPrice":
            return Optional.ofNullable(clazz.cast(bidPriceAsPercentageOfOnDemandPrice()));
        case "Configurations":
            return Optional.ofNullable(clazz.cast(configurations()));
        case "EbsBlockDevices":
            return Optional.ofNullable(clazz.cast(ebsBlockDevices()));
        case "EbsOptimized":
            return Optional.ofNullable(clazz.cast(ebsOptimized()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceTypeSpecification, T> g) {
        return obj -> g.apply((InstanceTypeSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceTypeSpecification> {
        /**
         * <p>
         * The EC2 instance type, for example <code>m3.xlarge</code>.
         * </p>
         * 
         * @param instanceType
         *        The EC2 instance type, for example <code>m3.xlarge</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The number of units that a provisioned instance of this type provides toward fulfilling the target capacities
         * defined in <a>InstanceFleetConfig</a>. Capacity values represent performance characteristics such as vCPUs,
         * memory, or I/O. If not specified, the default value is 1.
         * </p>
         * 
         * @param weightedCapacity
         *        The number of units that a provisioned instance of this type provides toward fulfilling the target
         *        capacities defined in <a>InstanceFleetConfig</a>. Capacity values represent performance
         *        characteristics such as vCPUs, memory, or I/O. If not specified, the default value is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weightedCapacity(Integer weightedCapacity);

        /**
         * <p>
         * The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD.
         * </p>
         * 
         * @param bidPrice
         *        The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in
         *        USD.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPrice(String bidPrice);

        /**
         * <p>
         * The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
         * <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%).
         * </p>
         * 
         * @param bidPriceAsPercentageOfOnDemandPrice
         *        The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
         *        <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPriceAsPercentageOfOnDemandPrice(Double bidPriceAsPercentageOfOnDemandPrice);

        /**
         * <p>
         * A configuration classification that applies when provisioning cluster instances, which can include
         * configurations for applications and software bundled with Amazon EMR.
         * </p>
         * 
         * @param configurations
         *        A configuration classification that applies when provisioning cluster instances, which can include
         *        configurations for applications and software bundled with Amazon EMR.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Collection<Configuration> configurations);

        /**
         * <p>
         * A configuration classification that applies when provisioning cluster instances, which can include
         * configurations for applications and software bundled with Amazon EMR.
         * </p>
         * 
         * @param configurations
         *        A configuration classification that applies when provisioning cluster instances, which can include
         *        configurations for applications and software bundled with Amazon EMR.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Configuration... configurations);

        /**
         * <p>
         * A configuration classification that applies when provisioning cluster instances, which can include
         * configurations for applications and software bundled with Amazon EMR.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Configuration>.Builder} avoiding the need
         * to create one manually via {@link List<Configuration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Configuration>.Builder#build()} is called immediately and
         * its result is passed to {@link #configurations(List<Configuration>)}.
         * 
         * @param configurations
         *        a consumer that will call methods on {@link List<Configuration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurations(List<Configuration>)
         */
        Builder configurations(Consumer<Configuration.Builder>... configurations);

        /**
         * <p>
         * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         * <code>InstanceType</code>.
         * </p>
         * 
         * @param ebsBlockDevices
         *        The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         *        <code>InstanceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsBlockDevices(Collection<EbsBlockDevice> ebsBlockDevices);

        /**
         * <p>
         * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         * <code>InstanceType</code>.
         * </p>
         * 
         * @param ebsBlockDevices
         *        The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         *        <code>InstanceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsBlockDevices(EbsBlockDevice... ebsBlockDevices);

        /**
         * <p>
         * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         * <code>InstanceType</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EbsBlockDevice>.Builder} avoiding the need
         * to create one manually via {@link List<EbsBlockDevice>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EbsBlockDevice>.Builder#build()} is called immediately and
         * its result is passed to {@link #ebsBlockDevices(List<EbsBlockDevice>)}.
         * 
         * @param ebsBlockDevices
         *        a consumer that will call methods on {@link List<EbsBlockDevice>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsBlockDevices(List<EbsBlockDevice>)
         */
        Builder ebsBlockDevices(Consumer<EbsBlockDevice.Builder>... ebsBlockDevices);

        /**
         * <p>
         * Evaluates to <code>TRUE</code> when the specified <code>InstanceType</code> is EBS-optimized.
         * </p>
         * 
         * @param ebsOptimized
         *        Evaluates to <code>TRUE</code> when the specified <code>InstanceType</code> is EBS-optimized.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptimized(Boolean ebsOptimized);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Integer weightedCapacity;

        private String bidPrice;

        private Double bidPriceAsPercentageOfOnDemandPrice;

        private List<Configuration> configurations = DefaultSdkAutoConstructList.getInstance();

        private List<EbsBlockDevice> ebsBlockDevices = DefaultSdkAutoConstructList.getInstance();

        private Boolean ebsOptimized;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceTypeSpecification model) {
            instanceType(model.instanceType);
            weightedCapacity(model.weightedCapacity);
            bidPrice(model.bidPrice);
            bidPriceAsPercentageOfOnDemandPrice(model.bidPriceAsPercentageOfOnDemandPrice);
            configurations(model.configurations);
            ebsBlockDevices(model.ebsBlockDevices);
            ebsOptimized(model.ebsOptimized);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Integer getWeightedCapacity() {
            return weightedCapacity;
        }

        @Override
        public final Builder weightedCapacity(Integer weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
            return this;
        }

        public final void setWeightedCapacity(Integer weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
        }

        public final String getBidPrice() {
            return bidPrice;
        }

        @Override
        public final Builder bidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
            return this;
        }

        public final void setBidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
        }

        public final Double getBidPriceAsPercentageOfOnDemandPrice() {
            return bidPriceAsPercentageOfOnDemandPrice;
        }

        @Override
        public final Builder bidPriceAsPercentageOfOnDemandPrice(Double bidPriceAsPercentageOfOnDemandPrice) {
            this.bidPriceAsPercentageOfOnDemandPrice = bidPriceAsPercentageOfOnDemandPrice;
            return this;
        }

        public final void setBidPriceAsPercentageOfOnDemandPrice(Double bidPriceAsPercentageOfOnDemandPrice) {
            this.bidPriceAsPercentageOfOnDemandPrice = bidPriceAsPercentageOfOnDemandPrice;
        }

        public final Collection<Configuration.Builder> getConfigurations() {
            return configurations != null ? configurations.stream().map(Configuration::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder configurations(Collection<Configuration> configurations) {
            this.configurations = ConfigurationListCopier.copy(configurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Configuration... configurations) {
            configurations(Arrays.asList(configurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Consumer<Configuration.Builder>... configurations) {
            configurations(Stream.of(configurations).map(c -> Configuration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setConfigurations(Collection<Configuration.BuilderImpl> configurations) {
            this.configurations = ConfigurationListCopier.copyFromBuilder(configurations);
        }

        public final Collection<EbsBlockDevice.Builder> getEbsBlockDevices() {
            return ebsBlockDevices != null ? ebsBlockDevices.stream().map(EbsBlockDevice::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder ebsBlockDevices(Collection<EbsBlockDevice> ebsBlockDevices) {
            this.ebsBlockDevices = EbsBlockDeviceListCopier.copy(ebsBlockDevices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsBlockDevices(EbsBlockDevice... ebsBlockDevices) {
            ebsBlockDevices(Arrays.asList(ebsBlockDevices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsBlockDevices(Consumer<EbsBlockDevice.Builder>... ebsBlockDevices) {
            ebsBlockDevices(Stream.of(ebsBlockDevices).map(c -> EbsBlockDevice.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEbsBlockDevices(Collection<EbsBlockDevice.BuilderImpl> ebsBlockDevices) {
            this.ebsBlockDevices = EbsBlockDeviceListCopier.copyFromBuilder(ebsBlockDevices);
        }

        public final Boolean getEbsOptimized() {
            return ebsOptimized;
        }

        @Override
        public final Builder ebsOptimized(Boolean ebsOptimized) {
            this.ebsOptimized = ebsOptimized;
            return this;
        }

        public final void setEbsOptimized(Boolean ebsOptimized) {
            this.ebsOptimized = ebsOptimized;
        }

        @Override
        public InstanceTypeSpecification build() {
            return new InstanceTypeSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
