/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an EC2 instance provisioned as part of cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Instance implements SdkPojo, Serializable, ToCopyableBuilder<Instance.Builder, Instance> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> EC2_INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::ec2InstanceId)).setter(setter(Builder::ec2InstanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2InstanceId").build()).build();

    private static final SdkField<String> PUBLIC_DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::publicDnsName)).setter(setter(Builder::publicDnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicDnsName").build()).build();

    private static final SdkField<String> PUBLIC_IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::publicIpAddress)).setter(setter(Builder::publicIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicIpAddress").build()).build();

    private static final SdkField<String> PRIVATE_DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::privateDnsName)).setter(setter(Builder::privateDnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateDnsName").build()).build();

    private static final SdkField<String> PRIVATE_IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::privateIpAddress)).setter(setter(Builder::privateIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddress").build()).build();

    private static final SdkField<InstanceStatus> STATUS_FIELD = SdkField.<InstanceStatus> builder(MarshallingType.SDK_POJO)
            .getter(getter(Instance::status)).setter(setter(Builder::status)).constructor(InstanceStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> INSTANCE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::instanceGroupId)).setter(setter(Builder::instanceGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroupId").build()).build();

    private static final SdkField<String> INSTANCE_FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::instanceFleetId)).setter(setter(Builder::instanceFleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFleetId").build()).build();

    private static final SdkField<String> MARKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::marketAsString)).setter(setter(Builder::market))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Market").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Instance::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<List<EbsVolume>> EBS_VOLUMES_FIELD = SdkField
            .<List<EbsVolume>> builder(MarshallingType.LIST)
            .getter(getter(Instance::ebsVolumes))
            .setter(setter(Builder::ebsVolumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsVolumes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EbsVolume> builder(MarshallingType.SDK_POJO)
                                            .constructor(EbsVolume::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            EC2_INSTANCE_ID_FIELD, PUBLIC_DNS_NAME_FIELD, PUBLIC_IP_ADDRESS_FIELD, PRIVATE_DNS_NAME_FIELD,
            PRIVATE_IP_ADDRESS_FIELD, STATUS_FIELD, INSTANCE_GROUP_ID_FIELD, INSTANCE_FLEET_ID_FIELD, MARKET_FIELD,
            INSTANCE_TYPE_FIELD, EBS_VOLUMES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String ec2InstanceId;

    private final String publicDnsName;

    private final String publicIpAddress;

    private final String privateDnsName;

    private final String privateIpAddress;

    private final InstanceStatus status;

    private final String instanceGroupId;

    private final String instanceFleetId;

    private final String market;

    private final String instanceType;

    private final List<EbsVolume> ebsVolumes;

    private Instance(BuilderImpl builder) {
        this.id = builder.id;
        this.ec2InstanceId = builder.ec2InstanceId;
        this.publicDnsName = builder.publicDnsName;
        this.publicIpAddress = builder.publicIpAddress;
        this.privateDnsName = builder.privateDnsName;
        this.privateIpAddress = builder.privateIpAddress;
        this.status = builder.status;
        this.instanceGroupId = builder.instanceGroupId;
        this.instanceFleetId = builder.instanceFleetId;
        this.market = builder.market;
        this.instanceType = builder.instanceType;
        this.ebsVolumes = builder.ebsVolumes;
    }

    /**
     * <p>
     * The unique identifier for the instance in Amazon EMR.
     * </p>
     * 
     * @return The unique identifier for the instance in Amazon EMR.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The unique identifier of the instance in Amazon EC2.
     * </p>
     * 
     * @return The unique identifier of the instance in Amazon EC2.
     */
    public String ec2InstanceId() {
        return ec2InstanceId;
    }

    /**
     * <p>
     * The public DNS name of the instance.
     * </p>
     * 
     * @return The public DNS name of the instance.
     */
    public String publicDnsName() {
        return publicDnsName;
    }

    /**
     * <p>
     * The public IP address of the instance.
     * </p>
     * 
     * @return The public IP address of the instance.
     */
    public String publicIpAddress() {
        return publicIpAddress;
    }

    /**
     * <p>
     * The private DNS name of the instance.
     * </p>
     * 
     * @return The private DNS name of the instance.
     */
    public String privateDnsName() {
        return privateDnsName;
    }

    /**
     * <p>
     * The private IP address of the instance.
     * </p>
     * 
     * @return The private IP address of the instance.
     */
    public String privateIpAddress() {
        return privateIpAddress;
    }

    /**
     * <p>
     * The current status of the instance.
     * </p>
     * 
     * @return The current status of the instance.
     */
    public InstanceStatus status() {
        return status;
    }

    /**
     * <p>
     * The identifier of the instance group to which this instance belongs.
     * </p>
     * 
     * @return The identifier of the instance group to which this instance belongs.
     */
    public String instanceGroupId() {
        return instanceGroupId;
    }

    /**
     * <p>
     * The unique identifier of the instance fleet to which an EC2 instance belongs.
     * </p>
     * 
     * @return The unique identifier of the instance fleet to which an EC2 instance belongs.
     */
    public String instanceFleetId() {
        return instanceFleetId;
    }

    /**
     * <p>
     * The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
     * @see MarketType
     */
    public MarketType market() {
        return MarketType.fromValue(market);
    }

    /**
     * <p>
     * The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
     * @see MarketType
     */
    public String marketAsString() {
        return market;
    }

    /**
     * <p>
     * The EC2 instance type, for example <code>m3.xlarge</code>.
     * </p>
     * 
     * @return The EC2 instance type, for example <code>m3.xlarge</code>.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * Returns true if the EbsVolumes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEbsVolumes() {
        return ebsVolumes != null && !(ebsVolumes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of EBS volumes that are attached to this instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEbsVolumes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of EBS volumes that are attached to this instance.
     */
    public List<EbsVolume> ebsVolumes() {
        return ebsVolumes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(ec2InstanceId());
        hashCode = 31 * hashCode + Objects.hashCode(publicDnsName());
        hashCode = 31 * hashCode + Objects.hashCode(publicIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(privateDnsName());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(instanceGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceFleetId());
        hashCode = 31 * hashCode + Objects.hashCode(marketAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(ebsVolumes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Instance)) {
            return false;
        }
        Instance other = (Instance) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(ec2InstanceId(), other.ec2InstanceId())
                && Objects.equals(publicDnsName(), other.publicDnsName())
                && Objects.equals(publicIpAddress(), other.publicIpAddress())
                && Objects.equals(privateDnsName(), other.privateDnsName())
                && Objects.equals(privateIpAddress(), other.privateIpAddress()) && Objects.equals(status(), other.status())
                && Objects.equals(instanceGroupId(), other.instanceGroupId())
                && Objects.equals(instanceFleetId(), other.instanceFleetId())
                && Objects.equals(marketAsString(), other.marketAsString())
                && Objects.equals(instanceType(), other.instanceType()) && Objects.equals(ebsVolumes(), other.ebsVolumes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Instance").add("Id", id()).add("Ec2InstanceId", ec2InstanceId())
                .add("PublicDnsName", publicDnsName()).add("PublicIpAddress", publicIpAddress())
                .add("PrivateDnsName", privateDnsName()).add("PrivateIpAddress", privateIpAddress()).add("Status", status())
                .add("InstanceGroupId", instanceGroupId()).add("InstanceFleetId", instanceFleetId())
                .add("Market", marketAsString()).add("InstanceType", instanceType()).add("EbsVolumes", ebsVolumes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Ec2InstanceId":
            return Optional.ofNullable(clazz.cast(ec2InstanceId()));
        case "PublicDnsName":
            return Optional.ofNullable(clazz.cast(publicDnsName()));
        case "PublicIpAddress":
            return Optional.ofNullable(clazz.cast(publicIpAddress()));
        case "PrivateDnsName":
            return Optional.ofNullable(clazz.cast(privateDnsName()));
        case "PrivateIpAddress":
            return Optional.ofNullable(clazz.cast(privateIpAddress()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "InstanceGroupId":
            return Optional.ofNullable(clazz.cast(instanceGroupId()));
        case "InstanceFleetId":
            return Optional.ofNullable(clazz.cast(instanceFleetId()));
        case "Market":
            return Optional.ofNullable(clazz.cast(marketAsString()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "EbsVolumes":
            return Optional.ofNullable(clazz.cast(ebsVolumes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Instance, T> g) {
        return obj -> g.apply((Instance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Instance> {
        /**
         * <p>
         * The unique identifier for the instance in Amazon EMR.
         * </p>
         * 
         * @param id
         *        The unique identifier for the instance in Amazon EMR.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The unique identifier of the instance in Amazon EC2.
         * </p>
         * 
         * @param ec2InstanceId
         *        The unique identifier of the instance in Amazon EC2.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceId(String ec2InstanceId);

        /**
         * <p>
         * The public DNS name of the instance.
         * </p>
         * 
         * @param publicDnsName
         *        The public DNS name of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicDnsName(String publicDnsName);

        /**
         * <p>
         * The public IP address of the instance.
         * </p>
         * 
         * @param publicIpAddress
         *        The public IP address of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicIpAddress(String publicIpAddress);

        /**
         * <p>
         * The private DNS name of the instance.
         * </p>
         * 
         * @param privateDnsName
         *        The private DNS name of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateDnsName(String privateDnsName);

        /**
         * <p>
         * The private IP address of the instance.
         * </p>
         * 
         * @param privateIpAddress
         *        The private IP address of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);

        /**
         * <p>
         * The current status of the instance.
         * </p>
         * 
         * @param status
         *        The current status of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(InstanceStatus status);

        /**
         * <p>
         * The current status of the instance.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceStatus.Builder} avoiding the need to
         * create one manually via {@link InstanceStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #status(InstanceStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link InstanceStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(InstanceStatus)
         */
        default Builder status(Consumer<InstanceStatus.Builder> status) {
            return status(InstanceStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The identifier of the instance group to which this instance belongs.
         * </p>
         * 
         * @param instanceGroupId
         *        The identifier of the instance group to which this instance belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroupId(String instanceGroupId);

        /**
         * <p>
         * The unique identifier of the instance fleet to which an EC2 instance belongs.
         * </p>
         * 
         * @param instanceFleetId
         *        The unique identifier of the instance fleet to which an EC2 instance belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFleetId(String instanceFleetId);

        /**
         * <p>
         * The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
         * </p>
         * 
         * @param market
         *        The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(String market);

        /**
         * <p>
         * The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
         * </p>
         * 
         * @param market
         *        The instance purchasing option. Valid values are <code>ON_DEMAND</code> or <code>SPOT</code>.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(MarketType market);

        /**
         * <p>
         * The EC2 instance type, for example <code>m3.xlarge</code>.
         * </p>
         * 
         * @param instanceType
         *        The EC2 instance type, for example <code>m3.xlarge</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The list of EBS volumes that are attached to this instance.
         * </p>
         * 
         * @param ebsVolumes
         *        The list of EBS volumes that are attached to this instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsVolumes(Collection<EbsVolume> ebsVolumes);

        /**
         * <p>
         * The list of EBS volumes that are attached to this instance.
         * </p>
         * 
         * @param ebsVolumes
         *        The list of EBS volumes that are attached to this instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsVolumes(EbsVolume... ebsVolumes);

        /**
         * <p>
         * The list of EBS volumes that are attached to this instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EbsVolume>.Builder} avoiding the need to
         * create one manually via {@link List<EbsVolume>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EbsVolume>.Builder#build()} is called immediately and its
         * result is passed to {@link #ebsVolumes(List<EbsVolume>)}.
         * 
         * @param ebsVolumes
         *        a consumer that will call methods on {@link List<EbsVolume>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsVolumes(List<EbsVolume>)
         */
        Builder ebsVolumes(Consumer<EbsVolume.Builder>... ebsVolumes);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String ec2InstanceId;

        private String publicDnsName;

        private String publicIpAddress;

        private String privateDnsName;

        private String privateIpAddress;

        private InstanceStatus status;

        private String instanceGroupId;

        private String instanceFleetId;

        private String market;

        private String instanceType;

        private List<EbsVolume> ebsVolumes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Instance model) {
            id(model.id);
            ec2InstanceId(model.ec2InstanceId);
            publicDnsName(model.publicDnsName);
            publicIpAddress(model.publicIpAddress);
            privateDnsName(model.privateDnsName);
            privateIpAddress(model.privateIpAddress);
            status(model.status);
            instanceGroupId(model.instanceGroupId);
            instanceFleetId(model.instanceFleetId);
            market(model.market);
            instanceType(model.instanceType);
            ebsVolumes(model.ebsVolumes);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getEc2InstanceId() {
            return ec2InstanceId;
        }

        @Override
        public final Builder ec2InstanceId(String ec2InstanceId) {
            this.ec2InstanceId = ec2InstanceId;
            return this;
        }

        public final void setEc2InstanceId(String ec2InstanceId) {
            this.ec2InstanceId = ec2InstanceId;
        }

        public final String getPublicDnsName() {
            return publicDnsName;
        }

        @Override
        public final Builder publicDnsName(String publicDnsName) {
            this.publicDnsName = publicDnsName;
            return this;
        }

        public final void setPublicDnsName(String publicDnsName) {
            this.publicDnsName = publicDnsName;
        }

        public final String getPublicIpAddress() {
            return publicIpAddress;
        }

        @Override
        public final Builder publicIpAddress(String publicIpAddress) {
            this.publicIpAddress = publicIpAddress;
            return this;
        }

        public final void setPublicIpAddress(String publicIpAddress) {
            this.publicIpAddress = publicIpAddress;
        }

        public final String getPrivateDnsName() {
            return privateDnsName;
        }

        @Override
        public final Builder privateDnsName(String privateDnsName) {
            this.privateDnsName = privateDnsName;
            return this;
        }

        public final void setPrivateDnsName(String privateDnsName) {
            this.privateDnsName = privateDnsName;
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        public final InstanceStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(InstanceStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(InstanceStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final String getInstanceGroupId() {
            return instanceGroupId;
        }

        @Override
        public final Builder instanceGroupId(String instanceGroupId) {
            this.instanceGroupId = instanceGroupId;
            return this;
        }

        public final void setInstanceGroupId(String instanceGroupId) {
            this.instanceGroupId = instanceGroupId;
        }

        public final String getInstanceFleetId() {
            return instanceFleetId;
        }

        @Override
        public final Builder instanceFleetId(String instanceFleetId) {
            this.instanceFleetId = instanceFleetId;
            return this;
        }

        public final void setInstanceFleetId(String instanceFleetId) {
            this.instanceFleetId = instanceFleetId;
        }

        public final String getMarket() {
            return market;
        }

        @Override
        public final Builder market(String market) {
            this.market = market;
            return this;
        }

        @Override
        public final Builder market(MarketType market) {
            this.market(market == null ? null : market.toString());
            return this;
        }

        public final void setMarket(String market) {
            this.market = market;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Collection<EbsVolume.Builder> getEbsVolumes() {
            return ebsVolumes != null ? ebsVolumes.stream().map(EbsVolume::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ebsVolumes(Collection<EbsVolume> ebsVolumes) {
            this.ebsVolumes = EbsVolumeListCopier.copy(ebsVolumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsVolumes(EbsVolume... ebsVolumes) {
            ebsVolumes(Arrays.asList(ebsVolumes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsVolumes(Consumer<EbsVolume.Builder>... ebsVolumes) {
            ebsVolumes(Stream.of(ebsVolumes).map(c -> EbsVolume.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setEbsVolumes(Collection<EbsVolume.BuilderImpl> ebsVolumes) {
            this.ebsVolumes = EbsVolumeListCopier.copyFromBuilder(ebsVolumes);
        }

        @Override
        public Instance build() {
            return new Instance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
