/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.emr.model.AddInstanceFleetRequest;
import software.amazon.awssdk.services.emr.model.AddInstanceFleetResponse;
import software.amazon.awssdk.services.emr.model.AddInstanceGroupsRequest;
import software.amazon.awssdk.services.emr.model.AddInstanceGroupsResponse;
import software.amazon.awssdk.services.emr.model.AddJobFlowStepsRequest;
import software.amazon.awssdk.services.emr.model.AddJobFlowStepsResponse;
import software.amazon.awssdk.services.emr.model.AddTagsRequest;
import software.amazon.awssdk.services.emr.model.AddTagsResponse;
import software.amazon.awssdk.services.emr.model.CancelStepsRequest;
import software.amazon.awssdk.services.emr.model.CancelStepsResponse;
import software.amazon.awssdk.services.emr.model.CreateSecurityConfigurationRequest;
import software.amazon.awssdk.services.emr.model.CreateSecurityConfigurationResponse;
import software.amazon.awssdk.services.emr.model.DeleteSecurityConfigurationRequest;
import software.amazon.awssdk.services.emr.model.DeleteSecurityConfigurationResponse;
import software.amazon.awssdk.services.emr.model.DescribeClusterRequest;
import software.amazon.awssdk.services.emr.model.DescribeClusterResponse;
import software.amazon.awssdk.services.emr.model.DescribeSecurityConfigurationRequest;
import software.amazon.awssdk.services.emr.model.DescribeSecurityConfigurationResponse;
import software.amazon.awssdk.services.emr.model.DescribeStepRequest;
import software.amazon.awssdk.services.emr.model.DescribeStepResponse;
import software.amazon.awssdk.services.emr.model.EmrException;
import software.amazon.awssdk.services.emr.model.EmrRequest;
import software.amazon.awssdk.services.emr.model.GetBlockPublicAccessConfigurationRequest;
import software.amazon.awssdk.services.emr.model.GetBlockPublicAccessConfigurationResponse;
import software.amazon.awssdk.services.emr.model.GetManagedScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.GetManagedScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.InternalServerErrorException;
import software.amazon.awssdk.services.emr.model.InternalServerException;
import software.amazon.awssdk.services.emr.model.InvalidRequestException;
import software.amazon.awssdk.services.emr.model.ListBootstrapActionsRequest;
import software.amazon.awssdk.services.emr.model.ListBootstrapActionsResponse;
import software.amazon.awssdk.services.emr.model.ListClustersRequest;
import software.amazon.awssdk.services.emr.model.ListClustersResponse;
import software.amazon.awssdk.services.emr.model.ListInstanceFleetsRequest;
import software.amazon.awssdk.services.emr.model.ListInstanceFleetsResponse;
import software.amazon.awssdk.services.emr.model.ListInstanceGroupsRequest;
import software.amazon.awssdk.services.emr.model.ListInstanceGroupsResponse;
import software.amazon.awssdk.services.emr.model.ListInstancesRequest;
import software.amazon.awssdk.services.emr.model.ListInstancesResponse;
import software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsRequest;
import software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsResponse;
import software.amazon.awssdk.services.emr.model.ListStepsRequest;
import software.amazon.awssdk.services.emr.model.ListStepsResponse;
import software.amazon.awssdk.services.emr.model.ModifyClusterRequest;
import software.amazon.awssdk.services.emr.model.ModifyClusterResponse;
import software.amazon.awssdk.services.emr.model.ModifyInstanceFleetRequest;
import software.amazon.awssdk.services.emr.model.ModifyInstanceFleetResponse;
import software.amazon.awssdk.services.emr.model.ModifyInstanceGroupsRequest;
import software.amazon.awssdk.services.emr.model.ModifyInstanceGroupsResponse;
import software.amazon.awssdk.services.emr.model.PutAutoScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.PutAutoScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.PutBlockPublicAccessConfigurationRequest;
import software.amazon.awssdk.services.emr.model.PutBlockPublicAccessConfigurationResponse;
import software.amazon.awssdk.services.emr.model.PutManagedScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.PutManagedScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveAutoScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.RemoveAutoScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveManagedScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.RemoveManagedScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveTagsRequest;
import software.amazon.awssdk.services.emr.model.RemoveTagsResponse;
import software.amazon.awssdk.services.emr.model.RunJobFlowRequest;
import software.amazon.awssdk.services.emr.model.RunJobFlowResponse;
import software.amazon.awssdk.services.emr.model.SetTerminationProtectionRequest;
import software.amazon.awssdk.services.emr.model.SetTerminationProtectionResponse;
import software.amazon.awssdk.services.emr.model.SetVisibleToAllUsersRequest;
import software.amazon.awssdk.services.emr.model.SetVisibleToAllUsersResponse;
import software.amazon.awssdk.services.emr.model.TerminateJobFlowsRequest;
import software.amazon.awssdk.services.emr.model.TerminateJobFlowsResponse;
import software.amazon.awssdk.services.emr.paginators.ListBootstrapActionsIterable;
import software.amazon.awssdk.services.emr.paginators.ListClustersIterable;
import software.amazon.awssdk.services.emr.paginators.ListInstanceFleetsIterable;
import software.amazon.awssdk.services.emr.paginators.ListInstanceGroupsIterable;
import software.amazon.awssdk.services.emr.paginators.ListInstancesIterable;
import software.amazon.awssdk.services.emr.paginators.ListSecurityConfigurationsIterable;
import software.amazon.awssdk.services.emr.paginators.ListStepsIterable;
import software.amazon.awssdk.services.emr.transform.AddInstanceFleetRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.AddInstanceGroupsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.AddJobFlowStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.CancelStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.CreateSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DeleteSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeStepRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetBlockPublicAccessConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetManagedScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListBootstrapActionsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListInstanceFleetsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListInstanceGroupsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListSecurityConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ModifyClusterRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ModifyInstanceFleetRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ModifyInstanceGroupsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutAutoScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutBlockPublicAccessConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutManagedScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveAutoScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveManagedScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RunJobFlowRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.SetTerminationProtectionRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.SetVisibleToAllUsersRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.TerminateJobFlowsRequestMarshaller;

/**
 * Internal implementation of {@link EmrClient}.
 *
 * @see EmrClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEmrClient implements EmrClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEmrClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds an instance fleet to a running cluster.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x.
     * </p>
     * </note>
     *
     * @param addInstanceFleetRequest
     * @return Result of the AddInstanceFleet operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddInstanceFleet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddInstanceFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddInstanceFleetResponse addInstanceFleet(AddInstanceFleetRequest addInstanceFleetRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddInstanceFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddInstanceFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddInstanceFleetRequest, AddInstanceFleetResponse>()
                .withOperationName("AddInstanceFleet").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addInstanceFleetRequest)
                .withMarshaller(new AddInstanceFleetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds one or more instance groups to a running cluster.
     * </p>
     *
     * @param addInstanceGroupsRequest
     *        Input to an AddInstanceGroups call.
     * @return Result of the AddInstanceGroups operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddInstanceGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddInstanceGroupsResponse addInstanceGroups(AddInstanceGroupsRequest addInstanceGroupsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddInstanceGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddInstanceGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddInstanceGroupsRequest, AddInstanceGroupsResponse>()
                .withOperationName("AddInstanceGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addInstanceGroupsRequest)
                .withMarshaller(new AddInstanceGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * AddJobFlowSteps adds new steps to a running cluster. A maximum of 256 steps are allowed in each job flow.
     * </p>
     * <p>
     * If your cluster is long-running (such as a Hive data warehouse) or complex, you may require more than 256 steps
     * to process your data. You can bypass the 256-step limitation in various ways, including using SSH to connect to
     * the master node and submitting queries directly to the software running on the master node, such as Hive and
     * Hadoop. For more information on how to do this, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/AddMoreThan256Steps.html">Add More than 256 Steps to
     * a Cluster</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     * <p>
     * A step specifies the location of a JAR file stored either on the master node of the cluster or in Amazon S3. Each
     * step is performed by the main function of the main class of the JAR file. The main class can be specified either
     * in the manifest of the JAR or by using the MainFunction parameter of the step.
     * </p>
     * <p>
     * Amazon EMR executes each step in the order listed. For a step to be considered complete, the main function must
     * exit with a zero exit code and all Hadoop jobs started while the step was running must have completed and run
     * successfully.
     * </p>
     * <p>
     * You can only add steps to a cluster that is in one of the following states: STARTING, BOOTSTRAPPING, RUNNING, or
     * WAITING.
     * </p>
     *
     * @param addJobFlowStepsRequest
     *        The input argument to the <a>AddJobFlowSteps</a> operation.
     * @return Result of the AddJobFlowSteps operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddJobFlowSteps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddJobFlowSteps"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddJobFlowStepsResponse addJobFlowSteps(AddJobFlowStepsRequest addJobFlowStepsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddJobFlowStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddJobFlowStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddJobFlowStepsRequest, AddJobFlowStepsResponse>()
                .withOperationName("AddJobFlowSteps").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addJobFlowStepsRequest)
                .withMarshaller(new AddJobFlowStepsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds tags to an Amazon EMR resource. Tags make it easier to associate clusters in various ways, such as grouping
     * clusters to track your Amazon EMR resource allocation costs. For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-tags.html">Tag Clusters</a>.
     * </p>
     *
     * @param addTagsRequest
     *        This input identifies a cluster and a list of tags to attach.
     * @return Result of the AddTags operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddTagsResponse addTags(AddTagsRequest addTagsRequest) throws InternalServerException, InvalidRequestException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(addTagsRequest)
                .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Cancels a pending step or steps in a running cluster. Available only in Amazon EMR versions 4.8.0 and later,
     * excluding version 5.0.0. A maximum of 256 steps are allowed in each CancelSteps request. CancelSteps is
     * idempotent but asynchronous; it does not guarantee a step will be canceled, even if the request is successfully
     * submitted. You can only cancel steps that are in a <code>PENDING</code> state.
     * </p>
     *
     * @param cancelStepsRequest
     *        The input argument to the <a>CancelSteps</a> operation.
     * @return Result of the CancelSteps operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.CancelSteps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/CancelSteps" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelStepsResponse cancelSteps(CancelStepsRequest cancelStepsRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CancelStepsRequest, CancelStepsResponse>()
                .withOperationName("CancelSteps").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(cancelStepsRequest)
                .withMarshaller(new CancelStepsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a security configuration, which is stored in the service and can be specified when a cluster is created.
     * </p>
     *
     * @param createSecurityConfigurationRequest
     * @return Result of the CreateSecurityConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.CreateSecurityConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/CreateSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSecurityConfigurationResponse createSecurityConfiguration(
            CreateSecurityConfigurationRequest createSecurityConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateSecurityConfigurationRequest, CreateSecurityConfigurationResponse>()
                        .withOperationName("CreateSecurityConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createSecurityConfigurationRequest)
                        .withMarshaller(new CreateSecurityConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a security configuration.
     * </p>
     *
     * @param deleteSecurityConfigurationRequest
     * @return Result of the DeleteSecurityConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DeleteSecurityConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DeleteSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSecurityConfigurationResponse deleteSecurityConfiguration(
            DeleteSecurityConfigurationRequest deleteSecurityConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteSecurityConfigurationRequest, DeleteSecurityConfigurationResponse>()
                        .withOperationName("DeleteSecurityConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteSecurityConfigurationRequest)
                        .withMarshaller(new DeleteSecurityConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides cluster-level details including status, hardware and software configuration, VPC settings, and so on.
     * </p>
     *
     * @param describeClusterRequest
     *        This input determines which cluster to describe.
     * @return Result of the DescribeCluster operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeClusterResponse describeCluster(DescribeClusterRequest describeClusterRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                .withOperationName("DescribeCluster").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeClusterRequest)
                .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides the details of a security configuration by returning the configuration JSON.
     * </p>
     *
     * @param describeSecurityConfigurationRequest
     * @return Result of the DescribeSecurityConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeSecurityConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSecurityConfigurationResponse describeSecurityConfiguration(
            DescribeSecurityConfigurationRequest describeSecurityConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeSecurityConfigurationRequest, DescribeSecurityConfigurationResponse>()
                        .withOperationName("DescribeSecurityConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeSecurityConfigurationRequest)
                        .withMarshaller(new DescribeSecurityConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides more detail about the cluster step.
     * </p>
     *
     * @param describeStepRequest
     *        This input determines which step to describe.
     * @return Result of the DescribeStep operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeStep
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeStep" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeStepResponse describeStep(DescribeStepRequest describeStepRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStepResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStepResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeStepRequest, DescribeStepResponse>()
                .withOperationName("DescribeStep").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeStepRequest)
                .withMarshaller(new DescribeStepRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the Amazon EMR block public access configuration for your AWS account in the current Region. For more
     * information see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/configure-block-public-access.html">Configure Block
     * Public Access for Amazon EMR</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param getBlockPublicAccessConfigurationRequest
     * @return Result of the GetBlockPublicAccessConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetBlockPublicAccessConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetBlockPublicAccessConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBlockPublicAccessConfigurationResponse getBlockPublicAccessConfiguration(
            GetBlockPublicAccessConfigurationRequest getBlockPublicAccessConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBlockPublicAccessConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBlockPublicAccessConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetBlockPublicAccessConfigurationRequest, GetBlockPublicAccessConfigurationResponse>()
                        .withOperationName("GetBlockPublicAccessConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getBlockPublicAccessConfigurationRequest)
                        .withMarshaller(new GetBlockPublicAccessConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Fetches the attached managed scaling policy for an Amazon EMR cluster.
     * </p>
     *
     * @param getManagedScalingPolicyRequest
     * @return Result of the GetManagedScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetManagedScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetManagedScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedScalingPolicyResponse getManagedScalingPolicy(GetManagedScalingPolicyRequest getManagedScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetManagedScalingPolicyRequest, GetManagedScalingPolicyResponse>()
                .withOperationName("GetManagedScalingPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getManagedScalingPolicyRequest)
                .withMarshaller(new GetManagedScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides information about the bootstrap actions associated with a cluster.
     * </p>
     *
     * @param listBootstrapActionsRequest
     *        This input determines which bootstrap actions to retrieve.
     * @return Result of the ListBootstrapActions operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListBootstrapActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListBootstrapActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBootstrapActionsResponse listBootstrapActions(ListBootstrapActionsRequest listBootstrapActionsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBootstrapActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBootstrapActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListBootstrapActionsRequest, ListBootstrapActionsResponse>()
                .withOperationName("ListBootstrapActions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listBootstrapActionsRequest)
                .withMarshaller(new ListBootstrapActionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides information about the bootstrap actions associated with a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBootstrapActions(software.amazon.awssdk.services.emr.model.ListBootstrapActionsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListBootstrapActionsIterable responses = client.listBootstrapActionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListBootstrapActionsIterable responses = client
     *             .listBootstrapActionsPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListBootstrapActionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListBootstrapActionsIterable responses = client.listBootstrapActionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBootstrapActions(software.amazon.awssdk.services.emr.model.ListBootstrapActionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listBootstrapActionsRequest
     *        This input determines which bootstrap actions to retrieve.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListBootstrapActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListBootstrapActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBootstrapActionsIterable listBootstrapActionsPaginator(ListBootstrapActionsRequest listBootstrapActionsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListBootstrapActionsIterable(this, applyPaginatorUserAgent(listBootstrapActionsRequest));
    }

    /**
     * <p>
     * Provides the status of all clusters visible to this AWS account. Allows you to filter the list of clusters based
     * on certain criteria; for example, filtering by cluster creation date and time or by status. This call returns a
     * maximum of 50 clusters per call, but returns a marker to track the paging of the cluster list across multiple
     * ListClusters calls.
     * </p>
     *
     * @param listClustersRequest
     *        This input determines how the ListClusters action filters the list of clusters that it returns.
     * @return Result of the ListClusters operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                .withOperationName("ListClusters").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listClustersRequest)
                .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides the status of all clusters visible to this AWS account. Allows you to filter the list of clusters based
     * on certain criteria; for example, filtering by cluster creation date and time or by status. This call returns a
     * maximum of 50 clusters per call, but returns a marker to track the paging of the cluster list across multiple
     * ListClusters calls.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.emr.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.emr.model.ListClustersRequest)} operation.</b>
     * </p>
     *
     * @param listClustersRequest
     *        This input determines how the ListClusters action filters the list of clusters that it returns.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListClustersIterable listClustersPaginator(ListClustersRequest listClustersRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListClustersIterable(this, applyPaginatorUserAgent(listClustersRequest));
    }

    /**
     * <p>
     * Lists all available details about the instance fleets in a cluster.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note>
     *
     * @param listInstanceFleetsRequest
     * @return Result of the ListInstanceFleets operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstanceFleets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstanceFleets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceFleetsResponse listInstanceFleets(ListInstanceFleetsRequest listInstanceFleetsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstanceFleetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInstanceFleetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInstanceFleetsRequest, ListInstanceFleetsResponse>()
                .withOperationName("ListInstanceFleets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInstanceFleetsRequest)
                .withMarshaller(new ListInstanceFleetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all available details about the instance fleets in a cluster.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listInstanceFleets(software.amazon.awssdk.services.emr.model.ListInstanceFleetsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListInstanceFleetsIterable responses = client.listInstanceFleetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListInstanceFleetsIterable responses = client
     *             .listInstanceFleetsPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListInstanceFleetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListInstanceFleetsIterable responses = client.listInstanceFleetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstanceFleets(software.amazon.awssdk.services.emr.model.ListInstanceFleetsRequest)} operation.</b>
     * </p>
     *
     * @param listInstanceFleetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstanceFleets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstanceFleets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceFleetsIterable listInstanceFleetsPaginator(ListInstanceFleetsRequest listInstanceFleetsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListInstanceFleetsIterable(this, applyPaginatorUserAgent(listInstanceFleetsRequest));
    }

    /**
     * <p>
     * Provides all available details about the instance groups in a cluster.
     * </p>
     *
     * @param listInstanceGroupsRequest
     *        This input determines which instance groups to retrieve.
     * @return Result of the ListInstanceGroups operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstanceGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceGroupsResponse listInstanceGroups(ListInstanceGroupsRequest listInstanceGroupsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstanceGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInstanceGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInstanceGroupsRequest, ListInstanceGroupsResponse>()
                .withOperationName("ListInstanceGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInstanceGroupsRequest)
                .withMarshaller(new ListInstanceGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides all available details about the instance groups in a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInstanceGroups(software.amazon.awssdk.services.emr.model.ListInstanceGroupsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListInstanceGroupsIterable responses = client.listInstanceGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListInstanceGroupsIterable responses = client
     *             .listInstanceGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListInstanceGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListInstanceGroupsIterable responses = client.listInstanceGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstanceGroups(software.amazon.awssdk.services.emr.model.ListInstanceGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listInstanceGroupsRequest
     *        This input determines which instance groups to retrieve.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstanceGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceGroupsIterable listInstanceGroupsPaginator(ListInstanceGroupsRequest listInstanceGroupsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListInstanceGroupsIterable(this, applyPaginatorUserAgent(listInstanceGroupsRequest));
    }

    /**
     * <p>
     * Provides information for all active EC2 instances and EC2 instances terminated in the last 30 days, up to a
     * maximum of 2,000. EC2 instances in any of the following states are considered active: AWAITING_FULFILLMENT,
     * PROVISIONING, BOOTSTRAPPING, RUNNING.
     * </p>
     *
     * @param listInstancesRequest
     *        This input determines which instances to list.
     * @return Result of the ListInstances operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInstancesResponse listInstances(ListInstancesRequest listInstancesRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                .withOperationName("ListInstances").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInstancesRequest)
                .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides information for all active EC2 instances and EC2 instances terminated in the last 30 days, up to a
     * maximum of 2,000. EC2 instances in any of the following states are considered active: AWAITING_FULFILLMENT,
     * PROVISIONING, BOOTSTRAPPING, RUNNING.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listInstances(software.amazon.awssdk.services.emr.model.ListInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstances(software.amazon.awssdk.services.emr.model.ListInstancesRequest)} operation.</b>
     * </p>
     *
     * @param listInstancesRequest
     *        This input determines which instances to list.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInstancesIterable listInstancesPaginator(ListInstancesRequest listInstancesRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListInstancesIterable(this, applyPaginatorUserAgent(listInstancesRequest));
    }

    /**
     * <p>
     * Lists all the security configurations visible to this account, providing their creation dates and times, and
     * their names. This call returns a maximum of 50 clusters per call, but returns a marker to track the paging of the
     * cluster list across multiple ListSecurityConfigurations calls.
     * </p>
     *
     * @param listSecurityConfigurationsRequest
     * @return Result of the ListSecurityConfigurations operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSecurityConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSecurityConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSecurityConfigurationsResponse listSecurityConfigurations(
            ListSecurityConfigurationsRequest listSecurityConfigurationsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSecurityConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSecurityConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListSecurityConfigurationsRequest, ListSecurityConfigurationsResponse>()
                        .withOperationName("ListSecurityConfigurations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listSecurityConfigurationsRequest)
                        .withMarshaller(new ListSecurityConfigurationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all the security configurations visible to this account, providing their creation dates and times, and
     * their names. This call returns a maximum of 50 clusters per call, but returns a marker to track the paging of the
     * cluster list across multiple ListSecurityConfigurations calls.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecurityConfigurations(software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListSecurityConfigurationsIterable responses = client.listSecurityConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListSecurityConfigurationsIterable responses = client
     *             .listSecurityConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListSecurityConfigurationsIterable responses = client.listSecurityConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecurityConfigurations(software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSecurityConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSecurityConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSecurityConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSecurityConfigurationsIterable listSecurityConfigurationsPaginator(
            ListSecurityConfigurationsRequest listSecurityConfigurationsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListSecurityConfigurationsIterable(this, applyPaginatorUserAgent(listSecurityConfigurationsRequest));
    }

    /**
     * <p>
     * Provides a list of steps for the cluster in reverse order unless you specify <code>stepIds</code> with the
     * request of filter by <code>StepStates</code>. You can specify a maximum of ten <code>stepIDs</code>.
     * </p>
     *
     * @param listStepsRequest
     *        This input determines which steps to list.
     * @return Result of the ListSteps operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSteps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSteps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStepsResponse listSteps(ListStepsRequest listStepsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListStepsRequest, ListStepsResponse>()
                .withOperationName("ListSteps").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listStepsRequest)
                .withMarshaller(new ListStepsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides a list of steps for the cluster in reverse order unless you specify <code>stepIds</code> with the
     * request of filter by <code>StepStates</code>. You can specify a maximum of ten <code>stepIDs</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSteps(software.amazon.awssdk.services.emr.model.ListStepsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListStepsIterable responses = client.listStepsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.emr.paginators.ListStepsIterable responses = client.listStepsPaginator(request);
     *     for (software.amazon.awssdk.services.emr.model.ListStepsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emr.paginators.ListStepsIterable responses = client.listStepsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSteps(software.amazon.awssdk.services.emr.model.ListStepsRequest)} operation.</b>
     * </p>
     *
     * @param listStepsRequest
     *        This input determines which steps to list.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSteps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSteps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStepsIterable listStepsPaginator(ListStepsRequest listStepsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        return new ListStepsIterable(this, applyPaginatorUserAgent(listStepsRequest));
    }

    /**
     * <p>
     * Modifies the number of steps that can be executed concurrently for the cluster specified using ClusterID.
     * </p>
     *
     * @param modifyClusterRequest
     * @return Result of the ModifyCluster operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ModifyCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyCluster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ModifyClusterResponse modifyCluster(ModifyClusterRequest modifyClusterRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ModifyClusterRequest, ModifyClusterResponse>()
                .withOperationName("ModifyCluster").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(modifyClusterRequest)
                .withMarshaller(new ModifyClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Modifies the target On-Demand and target Spot capacities for the instance fleet with the specified
     * InstanceFleetID within the cluster specified using ClusterID. The call either succeeds or fails atomically.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note>
     *
     * @param modifyInstanceFleetRequest
     * @return Result of the ModifyInstanceFleet operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ModifyInstanceFleet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyInstanceFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyInstanceFleetResponse modifyInstanceFleet(ModifyInstanceFleetRequest modifyInstanceFleetRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyInstanceFleetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyInstanceFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ModifyInstanceFleetRequest, ModifyInstanceFleetResponse>()
                .withOperationName("ModifyInstanceFleet").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(modifyInstanceFleetRequest)
                .withMarshaller(new ModifyInstanceFleetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * ModifyInstanceGroups modifies the number of nodes and configuration settings of an instance group. The input
     * parameters include the new target instance count for the group and the instance group ID. The call will either
     * succeed or fail atomically.
     * </p>
     *
     * @param modifyInstanceGroupsRequest
     *        Change the size of some instance groups.
     * @return Result of the ModifyInstanceGroups operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ModifyInstanceGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyInstanceGroupsResponse modifyInstanceGroups(ModifyInstanceGroupsRequest modifyInstanceGroupsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyInstanceGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyInstanceGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ModifyInstanceGroupsRequest, ModifyInstanceGroupsResponse>()
                .withOperationName("ModifyInstanceGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(modifyInstanceGroupsRequest)
                .withMarshaller(new ModifyInstanceGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or updates an automatic scaling policy for a core instance group or task instance group in an Amazon EMR
     * cluster. The automatic scaling policy defines how an instance group dynamically adds and terminates EC2 instances
     * in response to the value of a CloudWatch metric.
     * </p>
     *
     * @param putAutoScalingPolicyRequest
     * @return Result of the PutAutoScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutAutoScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutAutoScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutAutoScalingPolicyResponse putAutoScalingPolicy(PutAutoScalingPolicyRequest putAutoScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAutoScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAutoScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutAutoScalingPolicyRequest, PutAutoScalingPolicyResponse>()
                .withOperationName("PutAutoScalingPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putAutoScalingPolicyRequest)
                .withMarshaller(new PutAutoScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or updates an Amazon EMR block public access configuration for your AWS account in the current Region.
     * For more information see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/configure-block-public-access.html">Configure Block
     * Public Access for Amazon EMR</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param putBlockPublicAccessConfigurationRequest
     * @return Result of the PutBlockPublicAccessConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutBlockPublicAccessConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutBlockPublicAccessConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutBlockPublicAccessConfigurationResponse putBlockPublicAccessConfiguration(
            PutBlockPublicAccessConfigurationRequest putBlockPublicAccessConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBlockPublicAccessConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutBlockPublicAccessConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<PutBlockPublicAccessConfigurationRequest, PutBlockPublicAccessConfigurationResponse>()
                        .withOperationName("PutBlockPublicAccessConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(putBlockPublicAccessConfigurationRequest)
                        .withMarshaller(new PutBlockPublicAccessConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or updates a managed scaling policy for an Amazon EMR cluster. The managed scaling policy defines the
     * limits for resources, such as EC2 instances that can be added or terminated from a cluster. The policy only
     * applies to the core and task nodes. The master node cannot be scaled after initial configuration.
     * </p>
     *
     * @param putManagedScalingPolicyRequest
     * @return Result of the PutManagedScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutManagedScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutManagedScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutManagedScalingPolicyResponse putManagedScalingPolicy(PutManagedScalingPolicyRequest putManagedScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutManagedScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutManagedScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutManagedScalingPolicyRequest, PutManagedScalingPolicyResponse>()
                .withOperationName("PutManagedScalingPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putManagedScalingPolicyRequest)
                .withMarshaller(new PutManagedScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes an automatic scaling policy from a specified instance group within an EMR cluster.
     * </p>
     *
     * @param removeAutoScalingPolicyRequest
     * @return Result of the RemoveAutoScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveAutoScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveAutoScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveAutoScalingPolicyResponse removeAutoScalingPolicy(RemoveAutoScalingPolicyRequest removeAutoScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveAutoScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveAutoScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemoveAutoScalingPolicyRequest, RemoveAutoScalingPolicyResponse>()
                .withOperationName("RemoveAutoScalingPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removeAutoScalingPolicyRequest)
                .withMarshaller(new RemoveAutoScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes a managed scaling policy from a specified EMR cluster.
     * </p>
     *
     * @param removeManagedScalingPolicyRequest
     * @return Result of the RemoveManagedScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveManagedScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveManagedScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveManagedScalingPolicyResponse removeManagedScalingPolicy(
            RemoveManagedScalingPolicyRequest removeManagedScalingPolicyRequest) throws AwsServiceException, SdkClientException,
            EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveManagedScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveManagedScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<RemoveManagedScalingPolicyRequest, RemoveManagedScalingPolicyResponse>()
                        .withOperationName("RemoveManagedScalingPolicy").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(removeManagedScalingPolicyRequest)
                        .withMarshaller(new RemoveManagedScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes tags from an Amazon EMR resource. Tags make it easier to associate clusters in various ways, such as
     * grouping clusters to track your Amazon EMR resource allocation costs. For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-tags.html">Tag Clusters</a>.
     * </p>
     * <p>
     * The following example removes the stack tag with value Prod from a cluster:
     * </p>
     *
     * @param removeTagsRequest
     *        This input identifies a cluster and a list of tags to remove.
     * @return Result of the RemoveTags operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RemoveTagsResponse removeTags(RemoveTagsRequest removeTagsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>()
                .withOperationName("RemoveTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removeTagsRequest)
                .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * RunJobFlow creates and starts running a new cluster (job flow). The cluster runs the steps specified. After the
     * steps complete, the cluster stops and the HDFS partition is lost. To prevent loss of data, configure the last
     * step of the job flow to store results in Amazon S3. If the <a>JobFlowInstancesConfig</a>
     * <code>KeepJobFlowAliveWhenNoSteps</code> parameter is set to <code>TRUE</code>, the cluster transitions to the
     * WAITING state rather than shutting down after the steps have completed.
     * </p>
     * <p>
     * For additional protection, you can set the <a>JobFlowInstancesConfig</a> <code>TerminationProtected</code>
     * parameter to <code>TRUE</code> to lock the cluster and prevent it from being terminated by API call, user
     * intervention, or in the event of a job flow error.
     * </p>
     * <p>
     * A maximum of 256 steps are allowed in each job flow.
     * </p>
     * <p>
     * If your cluster is long-running (such as a Hive data warehouse) or complex, you may require more than 256 steps
     * to process your data. You can bypass the 256-step limitation in various ways, including using the SSH shell to
     * connect to the master node and submitting queries directly to the software running on the master node, such as
     * Hive and Hadoop. For more information on how to do this, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/AddMoreThan256Steps.html">Add More than 256 Steps to
     * a Cluster</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     * <p>
     * For long running clusters, we recommend that you periodically store your results.
     * </p>
     * <note>
     * <p>
     * The instance fleets configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
     * versions. The RunJobFlow request can contain InstanceFleets parameters or InstanceGroups parameters, but not
     * both.
     * </p>
     * </note>
     *
     * @param runJobFlowRequest
     *        Input to the <a>RunJobFlow</a> operation.
     * @return Result of the RunJobFlow operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RunJobFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RunJobFlow" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RunJobFlowResponse runJobFlow(RunJobFlowRequest runJobFlowRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RunJobFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RunJobFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RunJobFlowRequest, RunJobFlowResponse>()
                .withOperationName("RunJobFlow").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(runJobFlowRequest)
                .withMarshaller(new RunJobFlowRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * SetTerminationProtection locks a cluster (job flow) so the EC2 instances in the cluster cannot be terminated by
     * user intervention, an API call, or in the event of a job-flow error. The cluster still terminates upon successful
     * completion of the job flow. Calling <code>SetTerminationProtection</code> on a cluster is similar to calling the
     * Amazon EC2 <code>DisableAPITermination</code> API on all EC2 instances in a cluster.
     * </p>
     * <p>
     * <code>SetTerminationProtection</code> is used to prevent accidental termination of a cluster and to ensure that
     * in the event of an error, the instances persist so that you can recover any data stored in their ephemeral
     * instance storage.
     * </p>
     * <p>
     * To terminate a cluster that has been locked by setting <code>SetTerminationProtection</code> to <code>true</code>
     * , you must first unlock the job flow by a subsequent call to <code>SetTerminationProtection</code> in which you
     * set the value to <code>false</code>.
     * </p>
     * <p>
     * For more information, see<a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/UsingEMR_TerminationProtection.html">Managing
     * Cluster Termination</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param setTerminationProtectionRequest
     *        The input argument to the <a>TerminationProtection</a> operation.
     * @return Result of the SetTerminationProtection operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.SetTerminationProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SetTerminationProtection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetTerminationProtectionResponse setTerminationProtection(
            SetTerminationProtectionRequest setTerminationProtectionRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetTerminationProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetTerminationProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<SetTerminationProtectionRequest, SetTerminationProtectionResponse>()
                        .withOperationName("SetTerminationProtection").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(setTerminationProtectionRequest)
                        .withMarshaller(new SetTerminationProtectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the <a>Cluster&#36VisibleToAllUsers</a> value, which determines whether the cluster is visible to all IAM
     * users of the AWS account associated with the cluster. Only the IAM user who created the cluster or the AWS
     * account root user can call this action. The default value, <code>true</code>, indicates that all IAM users in the
     * AWS account can perform cluster actions if they have the proper IAM policy permissions. If set to
     * <code>false</code>, only the IAM user that created the cluster can perform actions. This action works on running
     * clusters. You can override the default <code>true</code> setting when you create a cluster by using the
     * <code>VisibleToAllUsers</code> parameter with <code>RunJobFlow</code>.
     * </p>
     *
     * @param setVisibleToAllUsersRequest
     *        The input to the SetVisibleToAllUsers action.
     * @return Result of the SetVisibleToAllUsers operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.SetVisibleToAllUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SetVisibleToAllUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetVisibleToAllUsersResponse setVisibleToAllUsers(SetVisibleToAllUsersRequest setVisibleToAllUsersRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetVisibleToAllUsersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetVisibleToAllUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SetVisibleToAllUsersRequest, SetVisibleToAllUsersResponse>()
                .withOperationName("SetVisibleToAllUsers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setVisibleToAllUsersRequest)
                .withMarshaller(new SetVisibleToAllUsersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * TerminateJobFlows shuts a list of clusters (job flows) down. When a job flow is shut down, any step not yet
     * completed is canceled and the EC2 instances on which the cluster is running are stopped. Any log files not
     * already saved are uploaded to Amazon S3 if a LogUri was specified when the cluster was created.
     * </p>
     * <p>
     * The maximum number of clusters allowed is 10. The call to <code>TerminateJobFlows</code> is asynchronous.
     * Depending on the configuration of the cluster, it may take up to 1-5 minutes for the cluster to completely
     * terminate and release allocated resources, such as Amazon EC2 instances.
     * </p>
     *
     * @param terminateJobFlowsRequest
     *        Input to the <a>TerminateJobFlows</a> operation.
     * @return Result of the TerminateJobFlows operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.TerminateJobFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/TerminateJobFlows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TerminateJobFlowsResponse terminateJobFlows(TerminateJobFlowsRequest terminateJobFlowsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TerminateJobFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TerminateJobFlowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TerminateJobFlowsRequest, TerminateJobFlowsResponse>()
                .withOperationName("TerminateJobFlows").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(terminateJobFlowsRequest)
                .withMarshaller(new TerminateJobFlowsRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EmrException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends EmrRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
