/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This entity represents an instance group, which is a group of instances that have common purpose. For example, CORE
 * instance group is used for HDFS.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceGroup implements SdkPojo, Serializable, ToCopyableBuilder<InstanceGroup.Builder, InstanceGroup> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceGroup::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceGroup::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> MARKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceGroup::marketAsString)).setter(setter(Builder::market))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Market").build()).build();

    private static final SdkField<String> INSTANCE_GROUP_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceGroup::instanceGroupTypeAsString)).setter(setter(Builder::instanceGroupType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroupType").build()).build();

    private static final SdkField<String> BID_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceGroup::bidPrice)).setter(setter(Builder::bidPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BidPrice").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceGroup::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> REQUESTED_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceGroup::requestedInstanceCount)).setter(setter(Builder::requestedInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedInstanceCount").build())
            .build();

    private static final SdkField<Integer> RUNNING_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceGroup::runningInstanceCount)).setter(setter(Builder::runningInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunningInstanceCount").build())
            .build();

    private static final SdkField<InstanceGroupStatus> STATUS_FIELD = SdkField
            .<InstanceGroupStatus> builder(MarshallingType.SDK_POJO).getter(getter(InstanceGroup::status))
            .setter(setter(Builder::status)).constructor(InstanceGroupStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<Configuration>> CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .getter(getter(InstanceGroup::configurations))
            .setter(setter(Builder::configurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> CONFIGURATIONS_VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(InstanceGroup::configurationsVersion)).setter(setter(Builder::configurationsVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationsVersion").build())
            .build();

    private static final SdkField<List<Configuration>> LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .getter(getter(InstanceGroup::lastSuccessfullyAppliedConfigurations))
            .setter(setter(Builder::lastSuccessfullyAppliedConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("LastSuccessfullyAppliedConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_VERSION_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .getter(getter(InstanceGroup::lastSuccessfullyAppliedConfigurationsVersion))
            .setter(setter(Builder::lastSuccessfullyAppliedConfigurationsVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("LastSuccessfullyAppliedConfigurationsVersion").build()).build();

    private static final SdkField<List<EbsBlockDevice>> EBS_BLOCK_DEVICES_FIELD = SdkField
            .<List<EbsBlockDevice>> builder(MarshallingType.LIST)
            .getter(getter(InstanceGroup::ebsBlockDevices))
            .setter(setter(Builder::ebsBlockDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsBlockDevices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EbsBlockDevice> builder(MarshallingType.SDK_POJO)
                                            .constructor(EbsBlockDevice::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> EBS_OPTIMIZED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(InstanceGroup::ebsOptimized)).setter(setter(Builder::ebsOptimized))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsOptimized").build()).build();

    private static final SdkField<ShrinkPolicy> SHRINK_POLICY_FIELD = SdkField.<ShrinkPolicy> builder(MarshallingType.SDK_POJO)
            .getter(getter(InstanceGroup::shrinkPolicy)).setter(setter(Builder::shrinkPolicy)).constructor(ShrinkPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShrinkPolicy").build()).build();

    private static final SdkField<AutoScalingPolicyDescription> AUTO_SCALING_POLICY_FIELD = SdkField
            .<AutoScalingPolicyDescription> builder(MarshallingType.SDK_POJO).getter(getter(InstanceGroup::autoScalingPolicy))
            .setter(setter(Builder::autoScalingPolicy)).constructor(AutoScalingPolicyDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingPolicy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            MARKET_FIELD, INSTANCE_GROUP_TYPE_FIELD, BID_PRICE_FIELD, INSTANCE_TYPE_FIELD, REQUESTED_INSTANCE_COUNT_FIELD,
            RUNNING_INSTANCE_COUNT_FIELD, STATUS_FIELD, CONFIGURATIONS_FIELD, CONFIGURATIONS_VERSION_FIELD,
            LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_FIELD, LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_VERSION_FIELD,
            EBS_BLOCK_DEVICES_FIELD, EBS_OPTIMIZED_FIELD, SHRINK_POLICY_FIELD, AUTO_SCALING_POLICY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String market;

    private final String instanceGroupType;

    private final String bidPrice;

    private final String instanceType;

    private final Integer requestedInstanceCount;

    private final Integer runningInstanceCount;

    private final InstanceGroupStatus status;

    private final List<Configuration> configurations;

    private final Long configurationsVersion;

    private final List<Configuration> lastSuccessfullyAppliedConfigurations;

    private final Long lastSuccessfullyAppliedConfigurationsVersion;

    private final List<EbsBlockDevice> ebsBlockDevices;

    private final Boolean ebsOptimized;

    private final ShrinkPolicy shrinkPolicy;

    private final AutoScalingPolicyDescription autoScalingPolicy;

    private InstanceGroup(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.market = builder.market;
        this.instanceGroupType = builder.instanceGroupType;
        this.bidPrice = builder.bidPrice;
        this.instanceType = builder.instanceType;
        this.requestedInstanceCount = builder.requestedInstanceCount;
        this.runningInstanceCount = builder.runningInstanceCount;
        this.status = builder.status;
        this.configurations = builder.configurations;
        this.configurationsVersion = builder.configurationsVersion;
        this.lastSuccessfullyAppliedConfigurations = builder.lastSuccessfullyAppliedConfigurations;
        this.lastSuccessfullyAppliedConfigurationsVersion = builder.lastSuccessfullyAppliedConfigurationsVersion;
        this.ebsBlockDevices = builder.ebsBlockDevices;
        this.ebsOptimized = builder.ebsOptimized;
        this.shrinkPolicy = builder.shrinkPolicy;
        this.autoScalingPolicy = builder.autoScalingPolicy;
    }

    /**
     * <p>
     * The identifier of the instance group.
     * </p>
     * 
     * @return The identifier of the instance group.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name of the instance group.
     * </p>
     * 
     * @return The name of the instance group.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * @see MarketType
     */
    public MarketType market() {
        return MarketType.fromValue(market);
    }

    /**
     * <p>
     * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * @see MarketType
     */
    public String marketAsString() {
        return market;
    }

    /**
     * <p>
     * The type of the instance group. Valid values are MASTER, CORE or TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceGroupType}
     * will return {@link InstanceGroupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceGroupTypeAsString}.
     * </p>
     * 
     * @return The type of the instance group. Valid values are MASTER, CORE or TASK.
     * @see InstanceGroupType
     */
    public InstanceGroupType instanceGroupType() {
        return InstanceGroupType.fromValue(instanceGroupType);
    }

    /**
     * <p>
     * The type of the instance group. Valid values are MASTER, CORE or TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceGroupType}
     * will return {@link InstanceGroupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceGroupTypeAsString}.
     * </p>
     * 
     * @return The type of the instance group. Valid values are MASTER, CORE or TASK.
     * @see InstanceGroupType
     */
    public String instanceGroupTypeAsString() {
        return instanceGroupType;
    }

    /**
     * <p>
     * The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD. If
     * neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
     * <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
     * </p>
     * 
     * @return The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD.
     *         If neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
     *         <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
     */
    public String bidPrice() {
        return bidPrice;
    }

    /**
     * <p>
     * The EC2 instance type for all instances in the instance group.
     * </p>
     * 
     * @return The EC2 instance type for all instances in the instance group.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The target number of instances for the instance group.
     * </p>
     * 
     * @return The target number of instances for the instance group.
     */
    public Integer requestedInstanceCount() {
        return requestedInstanceCount;
    }

    /**
     * <p>
     * The number of instances currently running in this instance group.
     * </p>
     * 
     * @return The number of instances currently running in this instance group.
     */
    public Integer runningInstanceCount() {
        return runningInstanceCount;
    }

    /**
     * <p>
     * The current status of the instance group.
     * </p>
     * 
     * @return The current status of the instance group.
     */
    public InstanceGroupStatus status() {
        return status;
    }

    /**
     * Returns true if the Configurations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasConfigurations() {
        return configurations != null && !(configurations instanceof SdkAutoConstructList);
    }

    /**
     * <note>
     * <p>
     * Amazon EMR releases 4.x or later.
     * </p>
     * </note>
     * <p>
     * The list of configurations supplied for an EMR cluster instance group. You can specify a separate configuration
     * for each instance group (master, core, and task).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfigurations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return <p>
     *         Amazon EMR releases 4.x or later.
     *         </p>
     *         </note>
     *         <p>
     *         The list of configurations supplied for an EMR cluster instance group. You can specify a separate
     *         configuration for each instance group (master, core, and task).
     */
    public List<Configuration> configurations() {
        return configurations;
    }

    /**
     * <p>
     * The version number of the requested configuration specification for this instance group.
     * </p>
     * 
     * @return The version number of the requested configuration specification for this instance group.
     */
    public Long configurationsVersion() {
        return configurationsVersion;
    }

    /**
     * Returns true if the LastSuccessfullyAppliedConfigurations property was specified by the sender (it may be empty),
     * or false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the
     * sender is the AWS service.
     */
    public boolean hasLastSuccessfullyAppliedConfigurations() {
        return lastSuccessfullyAppliedConfigurations != null
                && !(lastSuccessfullyAppliedConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of configurations that were successfully applied for an instance group last time.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLastSuccessfullyAppliedConfigurations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of configurations that were successfully applied for an instance group last time.
     */
    public List<Configuration> lastSuccessfullyAppliedConfigurations() {
        return lastSuccessfullyAppliedConfigurations;
    }

    /**
     * <p>
     * The version number of a configuration specification that was successfully applied for an instance group last
     * time.
     * </p>
     * 
     * @return The version number of a configuration specification that was successfully applied for an instance group
     *         last time.
     */
    public Long lastSuccessfullyAppliedConfigurationsVersion() {
        return lastSuccessfullyAppliedConfigurationsVersion;
    }

    /**
     * Returns true if the EbsBlockDevices property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEbsBlockDevices() {
        return ebsBlockDevices != null && !(ebsBlockDevices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The EBS block devices that are mapped to this instance group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEbsBlockDevices()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The EBS block devices that are mapped to this instance group.
     */
    public List<EbsBlockDevice> ebsBlockDevices() {
        return ebsBlockDevices;
    }

    /**
     * <p>
     * If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized configuration stack
     * and provides additional, dedicated capacity for Amazon EBS I/O.
     * </p>
     * 
     * @return If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized configuration
     *         stack and provides additional, dedicated capacity for Amazon EBS I/O.
     */
    public Boolean ebsOptimized() {
        return ebsOptimized;
    }

    /**
     * <p>
     * Policy for customizing shrink operations.
     * </p>
     * 
     * @return Policy for customizing shrink operations.
     */
    public ShrinkPolicy shrinkPolicy() {
        return shrinkPolicy;
    }

    /**
     * <p>
     * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
     * automatic scaling policy defines how an instance group dynamically adds and terminates EC2 instances in response
     * to the value of a CloudWatch metric. See PutAutoScalingPolicy.
     * </p>
     * 
     * @return An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.
     *         The automatic scaling policy defines how an instance group dynamically adds and terminates EC2 instances
     *         in response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
     */
    public AutoScalingPolicyDescription autoScalingPolicy() {
        return autoScalingPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(marketAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceGroupTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bidPrice());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(requestedInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(runningInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(configurations());
        hashCode = 31 * hashCode + Objects.hashCode(configurationsVersion());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfullyAppliedConfigurations());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfullyAppliedConfigurationsVersion());
        hashCode = 31 * hashCode + Objects.hashCode(ebsBlockDevices());
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptimized());
        hashCode = 31 * hashCode + Objects.hashCode(shrinkPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingPolicy());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceGroup)) {
            return false;
        }
        InstanceGroup other = (InstanceGroup) obj;
        return Objects.equals(id(), other.id())
                && Objects.equals(name(), other.name())
                && Objects.equals(marketAsString(), other.marketAsString())
                && Objects.equals(instanceGroupTypeAsString(), other.instanceGroupTypeAsString())
                && Objects.equals(bidPrice(), other.bidPrice())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(requestedInstanceCount(), other.requestedInstanceCount())
                && Objects.equals(runningInstanceCount(), other.runningInstanceCount())
                && Objects.equals(status(), other.status())
                && Objects.equals(configurations(), other.configurations())
                && Objects.equals(configurationsVersion(), other.configurationsVersion())
                && Objects.equals(lastSuccessfullyAppliedConfigurations(), other.lastSuccessfullyAppliedConfigurations())
                && Objects.equals(lastSuccessfullyAppliedConfigurationsVersion(),
                        other.lastSuccessfullyAppliedConfigurationsVersion())
                && Objects.equals(ebsBlockDevices(), other.ebsBlockDevices())
                && Objects.equals(ebsOptimized(), other.ebsOptimized()) && Objects.equals(shrinkPolicy(), other.shrinkPolicy())
                && Objects.equals(autoScalingPolicy(), other.autoScalingPolicy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceGroup").add("Id", id()).add("Name", name()).add("Market", marketAsString())
                .add("InstanceGroupType", instanceGroupTypeAsString()).add("BidPrice", bidPrice())
                .add("InstanceType", instanceType()).add("RequestedInstanceCount", requestedInstanceCount())
                .add("RunningInstanceCount", runningInstanceCount()).add("Status", status())
                .add("Configurations", configurations()).add("ConfigurationsVersion", configurationsVersion())
                .add("LastSuccessfullyAppliedConfigurations", lastSuccessfullyAppliedConfigurations())
                .add("LastSuccessfullyAppliedConfigurationsVersion", lastSuccessfullyAppliedConfigurationsVersion())
                .add("EbsBlockDevices", ebsBlockDevices()).add("EbsOptimized", ebsOptimized())
                .add("ShrinkPolicy", shrinkPolicy()).add("AutoScalingPolicy", autoScalingPolicy()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Market":
            return Optional.ofNullable(clazz.cast(marketAsString()));
        case "InstanceGroupType":
            return Optional.ofNullable(clazz.cast(instanceGroupTypeAsString()));
        case "BidPrice":
            return Optional.ofNullable(clazz.cast(bidPrice()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "RequestedInstanceCount":
            return Optional.ofNullable(clazz.cast(requestedInstanceCount()));
        case "RunningInstanceCount":
            return Optional.ofNullable(clazz.cast(runningInstanceCount()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Configurations":
            return Optional.ofNullable(clazz.cast(configurations()));
        case "ConfigurationsVersion":
            return Optional.ofNullable(clazz.cast(configurationsVersion()));
        case "LastSuccessfullyAppliedConfigurations":
            return Optional.ofNullable(clazz.cast(lastSuccessfullyAppliedConfigurations()));
        case "LastSuccessfullyAppliedConfigurationsVersion":
            return Optional.ofNullable(clazz.cast(lastSuccessfullyAppliedConfigurationsVersion()));
        case "EbsBlockDevices":
            return Optional.ofNullable(clazz.cast(ebsBlockDevices()));
        case "EbsOptimized":
            return Optional.ofNullable(clazz.cast(ebsOptimized()));
        case "ShrinkPolicy":
            return Optional.ofNullable(clazz.cast(shrinkPolicy()));
        case "AutoScalingPolicy":
            return Optional.ofNullable(clazz.cast(autoScalingPolicy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceGroup, T> g) {
        return obj -> g.apply((InstanceGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceGroup> {
        /**
         * <p>
         * The identifier of the instance group.
         * </p>
         * 
         * @param id
         *        The identifier of the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the instance group.
         * </p>
         * 
         * @param name
         *        The name of the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * </p>
         * 
         * @param market
         *        The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(String market);

        /**
         * <p>
         * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * </p>
         * 
         * @param market
         *        The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(MarketType market);

        /**
         * <p>
         * The type of the instance group. Valid values are MASTER, CORE or TASK.
         * </p>
         * 
         * @param instanceGroupType
         *        The type of the instance group. Valid values are MASTER, CORE or TASK.
         * @see InstanceGroupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceGroupType
         */
        Builder instanceGroupType(String instanceGroupType);

        /**
         * <p>
         * The type of the instance group. Valid values are MASTER, CORE or TASK.
         * </p>
         * 
         * @param instanceGroupType
         *        The type of the instance group. Valid values are MASTER, CORE or TASK.
         * @see InstanceGroupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceGroupType
         */
        Builder instanceGroupType(InstanceGroupType instanceGroupType);

        /**
         * <p>
         * The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD. If
         * neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
         * <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
         * </p>
         * 
         * @param bidPrice
         *        The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in
         *        USD. If neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is
         *        provided, <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPrice(String bidPrice);

        /**
         * <p>
         * The EC2 instance type for all instances in the instance group.
         * </p>
         * 
         * @param instanceType
         *        The EC2 instance type for all instances in the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The target number of instances for the instance group.
         * </p>
         * 
         * @param requestedInstanceCount
         *        The target number of instances for the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedInstanceCount(Integer requestedInstanceCount);

        /**
         * <p>
         * The number of instances currently running in this instance group.
         * </p>
         * 
         * @param runningInstanceCount
         *        The number of instances currently running in this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningInstanceCount(Integer runningInstanceCount);

        /**
         * <p>
         * The current status of the instance group.
         * </p>
         * 
         * @param status
         *        The current status of the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(InstanceGroupStatus status);

        /**
         * <p>
         * The current status of the instance group.
         * </p>
         * This is a convenience that creates an instance of the {@link InstanceGroupStatus.Builder} avoiding the need
         * to create one manually via {@link InstanceGroupStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link InstanceGroupStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #status(InstanceGroupStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link InstanceGroupStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(InstanceGroupStatus)
         */
        default Builder status(Consumer<InstanceGroupStatus.Builder> status) {
            return status(InstanceGroupStatus.builder().applyMutation(status).build());
        }

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or later.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * 
         * @param configurations
         *        <p>
         *        Amazon EMR releases 4.x or later.
         *        </p>
         *        </note>
         *        <p>
         *        The list of configurations supplied for an EMR cluster instance group. You can specify a separate
         *        configuration for each instance group (master, core, and task).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Collection<Configuration> configurations);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or later.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * 
         * @param configurations
         *        <p>
         *        Amazon EMR releases 4.x or later.
         *        </p>
         *        </note>
         *        <p>
         *        The list of configurations supplied for an EMR cluster instance group. You can specify a separate
         *        configuration for each instance group (master, core, and task).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Configuration... configurations);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or later.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * This is a convenience that creates an instance of the {@link List<Configuration>.Builder} avoiding the need
         * to create one manually via {@link List<Configuration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Configuration>.Builder#build()} is called immediately and
         * its result is passed to {@link #configurations(List<Configuration>)}.
         * 
         * @param configurations
         *        a consumer that will call methods on {@link List<Configuration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurations(List<Configuration>)
         */
        Builder configurations(Consumer<Configuration.Builder>... configurations);

        /**
         * <p>
         * The version number of the requested configuration specification for this instance group.
         * </p>
         * 
         * @param configurationsVersion
         *        The version number of the requested configuration specification for this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationsVersion(Long configurationsVersion);

        /**
         * <p>
         * A list of configurations that were successfully applied for an instance group last time.
         * </p>
         * 
         * @param lastSuccessfullyAppliedConfigurations
         *        A list of configurations that were successfully applied for an instance group last time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfullyAppliedConfigurations(Collection<Configuration> lastSuccessfullyAppliedConfigurations);

        /**
         * <p>
         * A list of configurations that were successfully applied for an instance group last time.
         * </p>
         * 
         * @param lastSuccessfullyAppliedConfigurations
         *        A list of configurations that were successfully applied for an instance group last time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfullyAppliedConfigurations(Configuration... lastSuccessfullyAppliedConfigurations);

        /**
         * <p>
         * A list of configurations that were successfully applied for an instance group last time.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Configuration>.Builder} avoiding the need
         * to create one manually via {@link List<Configuration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Configuration>.Builder#build()} is called immediately and
         * its result is passed to {@link #lastSuccessfullyAppliedConfigurations(List<Configuration>)}.
         * 
         * @param lastSuccessfullyAppliedConfigurations
         *        a consumer that will call methods on {@link List<Configuration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastSuccessfullyAppliedConfigurations(List<Configuration>)
         */
        Builder lastSuccessfullyAppliedConfigurations(Consumer<Configuration.Builder>... lastSuccessfullyAppliedConfigurations);

        /**
         * <p>
         * The version number of a configuration specification that was successfully applied for an instance group last
         * time.
         * </p>
         * 
         * @param lastSuccessfullyAppliedConfigurationsVersion
         *        The version number of a configuration specification that was successfully applied for an instance
         *        group last time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfullyAppliedConfigurationsVersion(Long lastSuccessfullyAppliedConfigurationsVersion);

        /**
         * <p>
         * The EBS block devices that are mapped to this instance group.
         * </p>
         * 
         * @param ebsBlockDevices
         *        The EBS block devices that are mapped to this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsBlockDevices(Collection<EbsBlockDevice> ebsBlockDevices);

        /**
         * <p>
         * The EBS block devices that are mapped to this instance group.
         * </p>
         * 
         * @param ebsBlockDevices
         *        The EBS block devices that are mapped to this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsBlockDevices(EbsBlockDevice... ebsBlockDevices);

        /**
         * <p>
         * The EBS block devices that are mapped to this instance group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EbsBlockDevice>.Builder} avoiding the need
         * to create one manually via {@link List<EbsBlockDevice>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EbsBlockDevice>.Builder#build()} is called immediately and
         * its result is passed to {@link #ebsBlockDevices(List<EbsBlockDevice>)}.
         * 
         * @param ebsBlockDevices
         *        a consumer that will call methods on {@link List<EbsBlockDevice>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsBlockDevices(List<EbsBlockDevice>)
         */
        Builder ebsBlockDevices(Consumer<EbsBlockDevice.Builder>... ebsBlockDevices);

        /**
         * <p>
         * If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized configuration
         * stack and provides additional, dedicated capacity for Amazon EBS I/O.
         * </p>
         * 
         * @param ebsOptimized
         *        If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized
         *        configuration stack and provides additional, dedicated capacity for Amazon EBS I/O.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptimized(Boolean ebsOptimized);

        /**
         * <p>
         * Policy for customizing shrink operations.
         * </p>
         * 
         * @param shrinkPolicy
         *        Policy for customizing shrink operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shrinkPolicy(ShrinkPolicy shrinkPolicy);

        /**
         * <p>
         * Policy for customizing shrink operations.
         * </p>
         * This is a convenience that creates an instance of the {@link ShrinkPolicy.Builder} avoiding the need to
         * create one manually via {@link ShrinkPolicy#builder()}.
         *
         * When the {@link Consumer} completes, {@link ShrinkPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #shrinkPolicy(ShrinkPolicy)}.
         * 
         * @param shrinkPolicy
         *        a consumer that will call methods on {@link ShrinkPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shrinkPolicy(ShrinkPolicy)
         */
        default Builder shrinkPolicy(Consumer<ShrinkPolicy.Builder> shrinkPolicy) {
            return shrinkPolicy(ShrinkPolicy.builder().applyMutation(shrinkPolicy).build());
        }

        /**
         * <p>
         * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
         * automatic scaling policy defines how an instance group dynamically adds and terminates EC2 instances in
         * response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
         * </p>
         * 
         * @param autoScalingPolicy
         *        An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.
         *        The automatic scaling policy defines how an instance group dynamically adds and terminates EC2
         *        instances in response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingPolicy(AutoScalingPolicyDescription autoScalingPolicy);

        /**
         * <p>
         * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
         * automatic scaling policy defines how an instance group dynamically adds and terminates EC2 instances in
         * response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
         * </p>
         * This is a convenience that creates an instance of the {@link AutoScalingPolicyDescription.Builder} avoiding
         * the need to create one manually via {@link AutoScalingPolicyDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoScalingPolicyDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #autoScalingPolicy(AutoScalingPolicyDescription)}.
         * 
         * @param autoScalingPolicy
         *        a consumer that will call methods on {@link AutoScalingPolicyDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoScalingPolicy(AutoScalingPolicyDescription)
         */
        default Builder autoScalingPolicy(Consumer<AutoScalingPolicyDescription.Builder> autoScalingPolicy) {
            return autoScalingPolicy(AutoScalingPolicyDescription.builder().applyMutation(autoScalingPolicy).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String market;

        private String instanceGroupType;

        private String bidPrice;

        private String instanceType;

        private Integer requestedInstanceCount;

        private Integer runningInstanceCount;

        private InstanceGroupStatus status;

        private List<Configuration> configurations = DefaultSdkAutoConstructList.getInstance();

        private Long configurationsVersion;

        private List<Configuration> lastSuccessfullyAppliedConfigurations = DefaultSdkAutoConstructList.getInstance();

        private Long lastSuccessfullyAppliedConfigurationsVersion;

        private List<EbsBlockDevice> ebsBlockDevices = DefaultSdkAutoConstructList.getInstance();

        private Boolean ebsOptimized;

        private ShrinkPolicy shrinkPolicy;

        private AutoScalingPolicyDescription autoScalingPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceGroup model) {
            id(model.id);
            name(model.name);
            market(model.market);
            instanceGroupType(model.instanceGroupType);
            bidPrice(model.bidPrice);
            instanceType(model.instanceType);
            requestedInstanceCount(model.requestedInstanceCount);
            runningInstanceCount(model.runningInstanceCount);
            status(model.status);
            configurations(model.configurations);
            configurationsVersion(model.configurationsVersion);
            lastSuccessfullyAppliedConfigurations(model.lastSuccessfullyAppliedConfigurations);
            lastSuccessfullyAppliedConfigurationsVersion(model.lastSuccessfullyAppliedConfigurationsVersion);
            ebsBlockDevices(model.ebsBlockDevices);
            ebsOptimized(model.ebsOptimized);
            shrinkPolicy(model.shrinkPolicy);
            autoScalingPolicy(model.autoScalingPolicy);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getMarketAsString() {
            return market;
        }

        @Override
        public final Builder market(String market) {
            this.market = market;
            return this;
        }

        @Override
        public final Builder market(MarketType market) {
            this.market(market == null ? null : market.toString());
            return this;
        }

        public final void setMarket(String market) {
            this.market = market;
        }

        public final String getInstanceGroupTypeAsString() {
            return instanceGroupType;
        }

        @Override
        public final Builder instanceGroupType(String instanceGroupType) {
            this.instanceGroupType = instanceGroupType;
            return this;
        }

        @Override
        public final Builder instanceGroupType(InstanceGroupType instanceGroupType) {
            this.instanceGroupType(instanceGroupType == null ? null : instanceGroupType.toString());
            return this;
        }

        public final void setInstanceGroupType(String instanceGroupType) {
            this.instanceGroupType = instanceGroupType;
        }

        public final String getBidPrice() {
            return bidPrice;
        }

        @Override
        public final Builder bidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
            return this;
        }

        public final void setBidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Integer getRequestedInstanceCount() {
            return requestedInstanceCount;
        }

        @Override
        public final Builder requestedInstanceCount(Integer requestedInstanceCount) {
            this.requestedInstanceCount = requestedInstanceCount;
            return this;
        }

        public final void setRequestedInstanceCount(Integer requestedInstanceCount) {
            this.requestedInstanceCount = requestedInstanceCount;
        }

        public final Integer getRunningInstanceCount() {
            return runningInstanceCount;
        }

        @Override
        public final Builder runningInstanceCount(Integer runningInstanceCount) {
            this.runningInstanceCount = runningInstanceCount;
            return this;
        }

        public final void setRunningInstanceCount(Integer runningInstanceCount) {
            this.runningInstanceCount = runningInstanceCount;
        }

        public final InstanceGroupStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(InstanceGroupStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(InstanceGroupStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final Collection<Configuration.Builder> getConfigurations() {
            return configurations != null ? configurations.stream().map(Configuration::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder configurations(Collection<Configuration> configurations) {
            this.configurations = ConfigurationListCopier.copy(configurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Configuration... configurations) {
            configurations(Arrays.asList(configurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Consumer<Configuration.Builder>... configurations) {
            configurations(Stream.of(configurations).map(c -> Configuration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setConfigurations(Collection<Configuration.BuilderImpl> configurations) {
            this.configurations = ConfigurationListCopier.copyFromBuilder(configurations);
        }

        public final Long getConfigurationsVersion() {
            return configurationsVersion;
        }

        @Override
        public final Builder configurationsVersion(Long configurationsVersion) {
            this.configurationsVersion = configurationsVersion;
            return this;
        }

        public final void setConfigurationsVersion(Long configurationsVersion) {
            this.configurationsVersion = configurationsVersion;
        }

        public final Collection<Configuration.Builder> getLastSuccessfullyAppliedConfigurations() {
            return lastSuccessfullyAppliedConfigurations != null ? lastSuccessfullyAppliedConfigurations.stream()
                    .map(Configuration::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder lastSuccessfullyAppliedConfigurations(Collection<Configuration> lastSuccessfullyAppliedConfigurations) {
            this.lastSuccessfullyAppliedConfigurations = ConfigurationListCopier.copy(lastSuccessfullyAppliedConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lastSuccessfullyAppliedConfigurations(Configuration... lastSuccessfullyAppliedConfigurations) {
            lastSuccessfullyAppliedConfigurations(Arrays.asList(lastSuccessfullyAppliedConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lastSuccessfullyAppliedConfigurations(
                Consumer<Configuration.Builder>... lastSuccessfullyAppliedConfigurations) {
            lastSuccessfullyAppliedConfigurations(Stream.of(lastSuccessfullyAppliedConfigurations)
                    .map(c -> Configuration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setLastSuccessfullyAppliedConfigurations(
                Collection<Configuration.BuilderImpl> lastSuccessfullyAppliedConfigurations) {
            this.lastSuccessfullyAppliedConfigurations = ConfigurationListCopier
                    .copyFromBuilder(lastSuccessfullyAppliedConfigurations);
        }

        public final Long getLastSuccessfullyAppliedConfigurationsVersion() {
            return lastSuccessfullyAppliedConfigurationsVersion;
        }

        @Override
        public final Builder lastSuccessfullyAppliedConfigurationsVersion(Long lastSuccessfullyAppliedConfigurationsVersion) {
            this.lastSuccessfullyAppliedConfigurationsVersion = lastSuccessfullyAppliedConfigurationsVersion;
            return this;
        }

        public final void setLastSuccessfullyAppliedConfigurationsVersion(Long lastSuccessfullyAppliedConfigurationsVersion) {
            this.lastSuccessfullyAppliedConfigurationsVersion = lastSuccessfullyAppliedConfigurationsVersion;
        }

        public final Collection<EbsBlockDevice.Builder> getEbsBlockDevices() {
            return ebsBlockDevices != null ? ebsBlockDevices.stream().map(EbsBlockDevice::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder ebsBlockDevices(Collection<EbsBlockDevice> ebsBlockDevices) {
            this.ebsBlockDevices = EbsBlockDeviceListCopier.copy(ebsBlockDevices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsBlockDevices(EbsBlockDevice... ebsBlockDevices) {
            ebsBlockDevices(Arrays.asList(ebsBlockDevices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsBlockDevices(Consumer<EbsBlockDevice.Builder>... ebsBlockDevices) {
            ebsBlockDevices(Stream.of(ebsBlockDevices).map(c -> EbsBlockDevice.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEbsBlockDevices(Collection<EbsBlockDevice.BuilderImpl> ebsBlockDevices) {
            this.ebsBlockDevices = EbsBlockDeviceListCopier.copyFromBuilder(ebsBlockDevices);
        }

        public final Boolean getEbsOptimized() {
            return ebsOptimized;
        }

        @Override
        public final Builder ebsOptimized(Boolean ebsOptimized) {
            this.ebsOptimized = ebsOptimized;
            return this;
        }

        public final void setEbsOptimized(Boolean ebsOptimized) {
            this.ebsOptimized = ebsOptimized;
        }

        public final ShrinkPolicy.Builder getShrinkPolicy() {
            return shrinkPolicy != null ? shrinkPolicy.toBuilder() : null;
        }

        @Override
        public final Builder shrinkPolicy(ShrinkPolicy shrinkPolicy) {
            this.shrinkPolicy = shrinkPolicy;
            return this;
        }

        public final void setShrinkPolicy(ShrinkPolicy.BuilderImpl shrinkPolicy) {
            this.shrinkPolicy = shrinkPolicy != null ? shrinkPolicy.build() : null;
        }

        public final AutoScalingPolicyDescription.Builder getAutoScalingPolicy() {
            return autoScalingPolicy != null ? autoScalingPolicy.toBuilder() : null;
        }

        @Override
        public final Builder autoScalingPolicy(AutoScalingPolicyDescription autoScalingPolicy) {
            this.autoScalingPolicy = autoScalingPolicy;
            return this;
        }

        public final void setAutoScalingPolicy(AutoScalingPolicyDescription.BuilderImpl autoScalingPolicy) {
            this.autoScalingPolicy = autoScalingPolicy != null ? autoScalingPolicy.build() : null;
        }

        @Override
        public InstanceGroup build() {
            return new InstanceGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
