/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticsearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a single step of the Upgrade or Upgrade Eligibility Check workflow.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpgradeStepItem implements SdkPojo, Serializable, ToCopyableBuilder<UpgradeStepItem.Builder, UpgradeStepItem> {
    private static final SdkField<String> UPGRADE_STEP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpgradeStepItem::upgradeStepAsString)).setter(setter(Builder::upgradeStep))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeStep").build()).build();

    private static final SdkField<String> UPGRADE_STEP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpgradeStepItem::upgradeStepStatusAsString)).setter(setter(Builder::upgradeStepStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeStepStatus").build()).build();

    private static final SdkField<List<String>> ISSUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(UpgradeStepItem::issues))
            .setter(setter(Builder::issues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Issues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> PROGRESS_PERCENT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(UpgradeStepItem::progressPercent)).setter(setter(Builder::progressPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProgressPercent").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(UPGRADE_STEP_FIELD,
            UPGRADE_STEP_STATUS_FIELD, ISSUES_FIELD, PROGRESS_PERCENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String upgradeStep;

    private final String upgradeStepStatus;

    private final List<String> issues;

    private final Double progressPercent;

    private UpgradeStepItem(BuilderImpl builder) {
        this.upgradeStep = builder.upgradeStep;
        this.upgradeStepStatus = builder.upgradeStepStatus;
        this.issues = builder.issues;
        this.progressPercent = builder.progressPercent;
    }

    /**
     * <p>
     * Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
     * <ul>
     * <li>PreUpgradeCheck</li>
     * <li>Snapshot</li>
     * <li>Upgrade</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #upgradeStep} will
     * return {@link UpgradeStep#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #upgradeStepAsString}.
     * </p>
     * 
     * @return Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
     *         <ul>
     *         <li>PreUpgradeCheck</li>
     *         <li>Snapshot</li>
     *         <li>Upgrade</li>
     *         </ul>
     * @see UpgradeStep
     */
    public UpgradeStep upgradeStep() {
        return UpgradeStep.fromValue(upgradeStep);
    }

    /**
     * <p>
     * Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
     * <ul>
     * <li>PreUpgradeCheck</li>
     * <li>Snapshot</li>
     * <li>Upgrade</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #upgradeStep} will
     * return {@link UpgradeStep#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #upgradeStepAsString}.
     * </p>
     * 
     * @return Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
     *         <ul>
     *         <li>PreUpgradeCheck</li>
     *         <li>Snapshot</li>
     *         <li>Upgrade</li>
     *         </ul>
     * @see UpgradeStep
     */
    public String upgradeStepAsString() {
        return upgradeStep;
    }

    /**
     * <p>
     * The status of a particular step during an upgrade. The status can take one of the following values:
     * <ul>
     * <li>In Progress</li>
     * <li>Succeeded</li>
     * <li>Succeeded with Issues</li>
     * <li>Failed</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #upgradeStepStatus}
     * will return {@link UpgradeStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #upgradeStepStatusAsString}.
     * </p>
     * 
     * @return The status of a particular step during an upgrade. The status can take one of the following values:
     *         <ul>
     *         <li>In Progress</li>
     *         <li>Succeeded</li>
     *         <li>Succeeded with Issues</li>
     *         <li>Failed</li>
     *         </ul>
     * @see UpgradeStatus
     */
    public UpgradeStatus upgradeStepStatus() {
        return UpgradeStatus.fromValue(upgradeStepStatus);
    }

    /**
     * <p>
     * The status of a particular step during an upgrade. The status can take one of the following values:
     * <ul>
     * <li>In Progress</li>
     * <li>Succeeded</li>
     * <li>Succeeded with Issues</li>
     * <li>Failed</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #upgradeStepStatus}
     * will return {@link UpgradeStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #upgradeStepStatusAsString}.
     * </p>
     * 
     * @return The status of a particular step during an upgrade. The status can take one of the following values:
     *         <ul>
     *         <li>In Progress</li>
     *         <li>Succeeded</li>
     *         <li>Succeeded with Issues</li>
     *         <li>Failed</li>
     *         </ul>
     * @see UpgradeStatus
     */
    public String upgradeStepStatusAsString() {
        return upgradeStepStatus;
    }

    /**
     * Returns true if the Issues property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasIssues() {
        return issues != null && !(issues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of strings containing detailed information about the errors encountered in a particular step.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIssues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of strings containing detailed information about the errors encountered in a particular step.
     */
    public List<String> issues() {
        return issues;
    }

    /**
     * <p>
     * The Floating point value representing progress percentage of a particular step.
     * </p>
     * 
     * @return The Floating point value representing progress percentage of a particular step.
     */
    public Double progressPercent() {
        return progressPercent;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(upgradeStepAsString());
        hashCode = 31 * hashCode + Objects.hashCode(upgradeStepStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(issues());
        hashCode = 31 * hashCode + Objects.hashCode(progressPercent());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpgradeStepItem)) {
            return false;
        }
        UpgradeStepItem other = (UpgradeStepItem) obj;
        return Objects.equals(upgradeStepAsString(), other.upgradeStepAsString())
                && Objects.equals(upgradeStepStatusAsString(), other.upgradeStepStatusAsString())
                && Objects.equals(issues(), other.issues()) && Objects.equals(progressPercent(), other.progressPercent());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UpgradeStepItem").add("UpgradeStep", upgradeStepAsString())
                .add("UpgradeStepStatus", upgradeStepStatusAsString()).add("Issues", issues())
                .add("ProgressPercent", progressPercent()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UpgradeStep":
            return Optional.ofNullable(clazz.cast(upgradeStepAsString()));
        case "UpgradeStepStatus":
            return Optional.ofNullable(clazz.cast(upgradeStepStatusAsString()));
        case "Issues":
            return Optional.ofNullable(clazz.cast(issues()));
        case "ProgressPercent":
            return Optional.ofNullable(clazz.cast(progressPercent()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpgradeStepItem, T> g) {
        return obj -> g.apply((UpgradeStepItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UpgradeStepItem> {
        /**
         * <p>
         * Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
         * <ul>
         * <li>PreUpgradeCheck</li>
         * <li>Snapshot</li>
         * <li>Upgrade</li>
         * </ul>
         * </p>
         * 
         * @param upgradeStep
         *        Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
         *        <ul>
         *        <li>PreUpgradeCheck</li>
         *        <li>Snapshot</li>
         *        <li>Upgrade</li>
         *        </ul>
         * @see UpgradeStep
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpgradeStep
         */
        Builder upgradeStep(String upgradeStep);

        /**
         * <p>
         * Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
         * <ul>
         * <li>PreUpgradeCheck</li>
         * <li>Snapshot</li>
         * <li>Upgrade</li>
         * </ul>
         * </p>
         * 
         * @param upgradeStep
         *        Represents one of 3 steps that an Upgrade or Upgrade Eligibility Check does through:
         *        <ul>
         *        <li>PreUpgradeCheck</li>
         *        <li>Snapshot</li>
         *        <li>Upgrade</li>
         *        </ul>
         * @see UpgradeStep
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpgradeStep
         */
        Builder upgradeStep(UpgradeStep upgradeStep);

        /**
         * <p>
         * The status of a particular step during an upgrade. The status can take one of the following values:
         * <ul>
         * <li>In Progress</li>
         * <li>Succeeded</li>
         * <li>Succeeded with Issues</li>
         * <li>Failed</li>
         * </ul>
         * </p>
         * 
         * @param upgradeStepStatus
         *        The status of a particular step during an upgrade. The status can take one of the following values:
         *        <ul>
         *        <li>In Progress</li>
         *        <li>Succeeded</li>
         *        <li>Succeeded with Issues</li>
         *        <li>Failed</li>
         *        </ul>
         * @see UpgradeStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpgradeStatus
         */
        Builder upgradeStepStatus(String upgradeStepStatus);

        /**
         * <p>
         * The status of a particular step during an upgrade. The status can take one of the following values:
         * <ul>
         * <li>In Progress</li>
         * <li>Succeeded</li>
         * <li>Succeeded with Issues</li>
         * <li>Failed</li>
         * </ul>
         * </p>
         * 
         * @param upgradeStepStatus
         *        The status of a particular step during an upgrade. The status can take one of the following values:
         *        <ul>
         *        <li>In Progress</li>
         *        <li>Succeeded</li>
         *        <li>Succeeded with Issues</li>
         *        <li>Failed</li>
         *        </ul>
         * @see UpgradeStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpgradeStatus
         */
        Builder upgradeStepStatus(UpgradeStatus upgradeStepStatus);

        /**
         * <p>
         * A list of strings containing detailed information about the errors encountered in a particular step.
         * </p>
         * 
         * @param issues
         *        A list of strings containing detailed information about the errors encountered in a particular step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issues(Collection<String> issues);

        /**
         * <p>
         * A list of strings containing detailed information about the errors encountered in a particular step.
         * </p>
         * 
         * @param issues
         *        A list of strings containing detailed information about the errors encountered in a particular step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issues(String... issues);

        /**
         * <p>
         * The Floating point value representing progress percentage of a particular step.
         * </p>
         * 
         * @param progressPercent
         *        The Floating point value representing progress percentage of a particular step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder progressPercent(Double progressPercent);
    }

    static final class BuilderImpl implements Builder {
        private String upgradeStep;

        private String upgradeStepStatus;

        private List<String> issues = DefaultSdkAutoConstructList.getInstance();

        private Double progressPercent;

        private BuilderImpl() {
        }

        private BuilderImpl(UpgradeStepItem model) {
            upgradeStep(model.upgradeStep);
            upgradeStepStatus(model.upgradeStepStatus);
            issues(model.issues);
            progressPercent(model.progressPercent);
        }

        public final String getUpgradeStep() {
            return upgradeStep;
        }

        @Override
        public final Builder upgradeStep(String upgradeStep) {
            this.upgradeStep = upgradeStep;
            return this;
        }

        @Override
        public final Builder upgradeStep(UpgradeStep upgradeStep) {
            this.upgradeStep(upgradeStep == null ? null : upgradeStep.toString());
            return this;
        }

        public final void setUpgradeStep(String upgradeStep) {
            this.upgradeStep = upgradeStep;
        }

        public final String getUpgradeStepStatus() {
            return upgradeStepStatus;
        }

        @Override
        public final Builder upgradeStepStatus(String upgradeStepStatus) {
            this.upgradeStepStatus = upgradeStepStatus;
            return this;
        }

        @Override
        public final Builder upgradeStepStatus(UpgradeStatus upgradeStepStatus) {
            this.upgradeStepStatus(upgradeStepStatus == null ? null : upgradeStepStatus.toString());
            return this;
        }

        public final void setUpgradeStepStatus(String upgradeStepStatus) {
            this.upgradeStepStatus = upgradeStepStatus;
        }

        public final Collection<String> getIssues() {
            return issues;
        }

        @Override
        public final Builder issues(Collection<String> issues) {
            this.issues = IssuesCopier.copy(issues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder issues(String... issues) {
            issues(Arrays.asList(issues));
            return this;
        }

        public final void setIssues(Collection<String> issues) {
            this.issues = IssuesCopier.copy(issues);
        }

        public final Double getProgressPercent() {
            return progressPercent;
        }

        @Override
        public final Builder progressPercent(Double progressPercent) {
            this.progressPercent = progressPercent;
            return this;
        }

        public final void setProgressPercent(Double progressPercent) {
            this.progressPercent = progressPercent;
        }

        @Override
        public UpgradeStepItem build() {
            return new UpgradeStepItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
