/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticsearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The current status of an Elasticsearch domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ElasticsearchDomainStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<ElasticsearchDomainStatus.Builder, ElasticsearchDomainStatus> {
    private static final SdkField<String> DOMAIN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ElasticsearchDomainStatus::domainId)).setter(setter(Builder::domainId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainId").build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ElasticsearchDomainStatus::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ElasticsearchDomainStatus::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<Boolean> CREATED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ElasticsearchDomainStatus::created)).setter(setter(Builder::created))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Created").build()).build();

    private static final SdkField<Boolean> DELETED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ElasticsearchDomainStatus::deleted)).setter(setter(Builder::deleted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Deleted").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ElasticsearchDomainStatus::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<Map<String, String>> ENDPOINTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(ElasticsearchDomainStatus::endpoints))
            .setter(setter(Builder::endpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> PROCESSING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ElasticsearchDomainStatus::processing)).setter(setter(Builder::processing))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Processing").build()).build();

    private static final SdkField<Boolean> UPGRADE_PROCESSING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ElasticsearchDomainStatus::upgradeProcessing)).setter(setter(Builder::upgradeProcessing))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeProcessing").build()).build();

    private static final SdkField<String> ELASTICSEARCH_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ElasticsearchDomainStatus::elasticsearchVersion)).setter(setter(Builder::elasticsearchVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ElasticsearchVersion").build())
            .build();

    private static final SdkField<ElasticsearchClusterConfig> ELASTICSEARCH_CLUSTER_CONFIG_FIELD = SdkField
            .<ElasticsearchClusterConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::elasticsearchClusterConfig))
            .setter(setter(Builder::elasticsearchClusterConfig))
            .constructor(ElasticsearchClusterConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ElasticsearchClusterConfig").build())
            .build();

    private static final SdkField<EBSOptions> EBS_OPTIONS_FIELD = SdkField.<EBSOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::ebsOptions)).setter(setter(Builder::ebsOptions))
            .constructor(EBSOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EBSOptions").build()).build();

    private static final SdkField<String> ACCESS_POLICIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ElasticsearchDomainStatus::accessPolicies)).setter(setter(Builder::accessPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessPolicies").build()).build();

    private static final SdkField<SnapshotOptions> SNAPSHOT_OPTIONS_FIELD = SdkField
            .<SnapshotOptions> builder(MarshallingType.SDK_POJO).getter(getter(ElasticsearchDomainStatus::snapshotOptions))
            .setter(setter(Builder::snapshotOptions)).constructor(SnapshotOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotOptions").build()).build();

    private static final SdkField<VPCDerivedInfo> VPC_OPTIONS_FIELD = SdkField.<VPCDerivedInfo> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::vpcOptions)).setter(setter(Builder::vpcOptions))
            .constructor(VPCDerivedInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VPCOptions").build()).build();

    private static final SdkField<CognitoOptions> COGNITO_OPTIONS_FIELD = SdkField
            .<CognitoOptions> builder(MarshallingType.SDK_POJO).getter(getter(ElasticsearchDomainStatus::cognitoOptions))
            .setter(setter(Builder::cognitoOptions)).constructor(CognitoOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CognitoOptions").build()).build();

    private static final SdkField<EncryptionAtRestOptions> ENCRYPTION_AT_REST_OPTIONS_FIELD = SdkField
            .<EncryptionAtRestOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::encryptionAtRestOptions)).setter(setter(Builder::encryptionAtRestOptions))
            .constructor(EncryptionAtRestOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAtRestOptions").build())
            .build();

    private static final SdkField<NodeToNodeEncryptionOptions> NODE_TO_NODE_ENCRYPTION_OPTIONS_FIELD = SdkField
            .<NodeToNodeEncryptionOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::nodeToNodeEncryptionOptions))
            .setter(setter(Builder::nodeToNodeEncryptionOptions))
            .constructor(NodeToNodeEncryptionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeToNodeEncryptionOptions")
                    .build()).build();

    private static final SdkField<Map<String, String>> ADVANCED_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(ElasticsearchDomainStatus::advancedOptions))
            .setter(setter(Builder::advancedOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, LogPublishingOption>> LOG_PUBLISHING_OPTIONS_FIELD = SdkField
            .<Map<String, LogPublishingOption>> builder(MarshallingType.MAP)
            .getter(getter(ElasticsearchDomainStatus::logPublishingOptionsAsStrings))
            .setter(setter(Builder::logPublishingOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPublishingOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<LogPublishingOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogPublishingOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ServiceSoftwareOptions> SERVICE_SOFTWARE_OPTIONS_FIELD = SdkField
            .<ServiceSoftwareOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::serviceSoftwareOptions)).setter(setter(Builder::serviceSoftwareOptions))
            .constructor(ServiceSoftwareOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceSoftwareOptions").build())
            .build();

    private static final SdkField<DomainEndpointOptions> DOMAIN_ENDPOINT_OPTIONS_FIELD = SdkField
            .<DomainEndpointOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(ElasticsearchDomainStatus::domainEndpointOptions)).setter(setter(Builder::domainEndpointOptions))
            .constructor(DomainEndpointOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainEndpointOptions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_ID_FIELD,
            DOMAIN_NAME_FIELD, ARN_FIELD, CREATED_FIELD, DELETED_FIELD, ENDPOINT_FIELD, ENDPOINTS_FIELD, PROCESSING_FIELD,
            UPGRADE_PROCESSING_FIELD, ELASTICSEARCH_VERSION_FIELD, ELASTICSEARCH_CLUSTER_CONFIG_FIELD, EBS_OPTIONS_FIELD,
            ACCESS_POLICIES_FIELD, SNAPSHOT_OPTIONS_FIELD, VPC_OPTIONS_FIELD, COGNITO_OPTIONS_FIELD,
            ENCRYPTION_AT_REST_OPTIONS_FIELD, NODE_TO_NODE_ENCRYPTION_OPTIONS_FIELD, ADVANCED_OPTIONS_FIELD,
            LOG_PUBLISHING_OPTIONS_FIELD, SERVICE_SOFTWARE_OPTIONS_FIELD, DOMAIN_ENDPOINT_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String domainId;

    private final String domainName;

    private final String arn;

    private final Boolean created;

    private final Boolean deleted;

    private final String endpoint;

    private final Map<String, String> endpoints;

    private final Boolean processing;

    private final Boolean upgradeProcessing;

    private final String elasticsearchVersion;

    private final ElasticsearchClusterConfig elasticsearchClusterConfig;

    private final EBSOptions ebsOptions;

    private final String accessPolicies;

    private final SnapshotOptions snapshotOptions;

    private final VPCDerivedInfo vpcOptions;

    private final CognitoOptions cognitoOptions;

    private final EncryptionAtRestOptions encryptionAtRestOptions;

    private final NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions;

    private final Map<String, String> advancedOptions;

    private final Map<String, LogPublishingOption> logPublishingOptions;

    private final ServiceSoftwareOptions serviceSoftwareOptions;

    private final DomainEndpointOptions domainEndpointOptions;

    private ElasticsearchDomainStatus(BuilderImpl builder) {
        this.domainId = builder.domainId;
        this.domainName = builder.domainName;
        this.arn = builder.arn;
        this.created = builder.created;
        this.deleted = builder.deleted;
        this.endpoint = builder.endpoint;
        this.endpoints = builder.endpoints;
        this.processing = builder.processing;
        this.upgradeProcessing = builder.upgradeProcessing;
        this.elasticsearchVersion = builder.elasticsearchVersion;
        this.elasticsearchClusterConfig = builder.elasticsearchClusterConfig;
        this.ebsOptions = builder.ebsOptions;
        this.accessPolicies = builder.accessPolicies;
        this.snapshotOptions = builder.snapshotOptions;
        this.vpcOptions = builder.vpcOptions;
        this.cognitoOptions = builder.cognitoOptions;
        this.encryptionAtRestOptions = builder.encryptionAtRestOptions;
        this.nodeToNodeEncryptionOptions = builder.nodeToNodeEncryptionOptions;
        this.advancedOptions = builder.advancedOptions;
        this.logPublishingOptions = builder.logPublishingOptions;
        this.serviceSoftwareOptions = builder.serviceSoftwareOptions;
        this.domainEndpointOptions = builder.domainEndpointOptions;
    }

    /**
     * <p>
     * The unique identifier for the specified Elasticsearch domain.
     * </p>
     * 
     * @return The unique identifier for the specified Elasticsearch domain.
     */
    public String domainId() {
        return domainId;
    }

    /**
     * <p>
     * The name of an Elasticsearch domain. Domain names are unique across the domains owned by an account within an AWS
     * region. Domain names start with a letter or number and can contain the following characters: a-z (lowercase),
     * 0-9, and - (hyphen).
     * </p>
     * 
     * @return The name of an Elasticsearch domain. Domain names are unique across the domains owned by an account
     *         within an AWS region. Domain names start with a letter or number and can contain the following
     *         characters: a-z (lowercase), 0-9, and - (hyphen).
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The Amazon resource name (ARN) of an Elasticsearch domain. See <a
     * href="http://docs.aws.amazon.com/IAM/latest/UserGuide/index.html?Using_Identifiers.html"
     * target="_blank">Identifiers for IAM Entities</a> in <i>Using AWS Identity and Access Management</i> for more
     * information.
     * </p>
     * 
     * @return The Amazon resource name (ARN) of an Elasticsearch domain. See <a
     *         href="http://docs.aws.amazon.com/IAM/latest/UserGuide/index.html?Using_Identifiers.html"
     *         target="_blank">Identifiers for IAM Entities</a> in <i>Using AWS Identity and Access Management</i> for
     *         more information.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The domain creation status. <code>True</code> if the creation of an Elasticsearch domain is complete.
     * <code>False</code> if domain creation is still in progress.
     * </p>
     * 
     * @return The domain creation status. <code>True</code> if the creation of an Elasticsearch domain is complete.
     *         <code>False</code> if domain creation is still in progress.
     */
    public Boolean created() {
        return created;
    }

    /**
     * <p>
     * The domain deletion status. <code>True</code> if a delete request has been received for the domain but resource
     * cleanup is still in progress. <code>False</code> if the domain has not been deleted. Once domain deletion is
     * complete, the status of the domain is no longer returned.
     * </p>
     * 
     * @return The domain deletion status. <code>True</code> if a delete request has been received for the domain but
     *         resource cleanup is still in progress. <code>False</code> if the domain has not been deleted. Once domain
     *         deletion is complete, the status of the domain is no longer returned.
     */
    public Boolean deleted() {
        return deleted;
    }

    /**
     * <p>
     * The Elasticsearch domain endpoint that you use to submit index and search requests.
     * </p>
     * 
     * @return The Elasticsearch domain endpoint that you use to submit index and search requests.
     */
    public String endpoint() {
        return endpoint;
    }

    /**
     * Returns true if the Endpoints property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEndpoints() {
        return endpoints != null && !(endpoints instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Map containing the Elasticsearch domain endpoints used to submit index and search requests. Example
     * <code>key, value</code>:
     * <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEndpoints()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Map containing the Elasticsearch domain endpoints used to submit index and search requests. Example
     *         <code>key, value</code>:
     *         <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
     */
    public Map<String, String> endpoints() {
        return endpoints;
    }

    /**
     * <p>
     * The status of the Elasticsearch domain configuration. <code>True</code> if Amazon Elasticsearch Service is
     * processing configuration changes. <code>False</code> if the configuration is active.
     * </p>
     * 
     * @return The status of the Elasticsearch domain configuration. <code>True</code> if Amazon Elasticsearch Service
     *         is processing configuration changes. <code>False</code> if the configuration is active.
     */
    public Boolean processing() {
        return processing;
    }

    /**
     * <p>
     * The status of an Elasticsearch domain version upgrade. <code>True</code> if Amazon Elasticsearch Service is
     * undergoing a version upgrade. <code>False</code> if the configuration is active.
     * </p>
     * 
     * @return The status of an Elasticsearch domain version upgrade. <code>True</code> if Amazon Elasticsearch Service
     *         is undergoing a version upgrade. <code>False</code> if the configuration is active.
     */
    public Boolean upgradeProcessing() {
        return upgradeProcessing;
    }

    /**
     * Returns the value of the ElasticsearchVersion property for this object.
     * 
     * @return The value of the ElasticsearchVersion property for this object.
     */
    public String elasticsearchVersion() {
        return elasticsearchVersion;
    }

    /**
     * <p>
     * The type and number of instances in the domain cluster.
     * </p>
     * 
     * @return The type and number of instances in the domain cluster.
     */
    public ElasticsearchClusterConfig elasticsearchClusterConfig() {
        return elasticsearchClusterConfig;
    }

    /**
     * <p>
     * The <code>EBSOptions</code> for the specified domain. See <a href=
     * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs"
     * target="_blank">Configuring EBS-based Storage</a> for more information.
     * </p>
     * 
     * @return The <code>EBSOptions</code> for the specified domain. See <a href=
     *         "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs"
     *         target="_blank">Configuring EBS-based Storage</a> for more information.
     */
    public EBSOptions ebsOptions() {
        return ebsOptions;
    }

    /**
     * <p>
     * IAM access policy as a JSON-formatted string.
     * </p>
     * 
     * @return IAM access policy as a JSON-formatted string.
     */
    public String accessPolicies() {
        return accessPolicies;
    }

    /**
     * <p>
     * Specifies the status of the <code>SnapshotOptions</code>
     * </p>
     * 
     * @return Specifies the status of the <code>SnapshotOptions</code>
     */
    public SnapshotOptions snapshotOptions() {
        return snapshotOptions;
    }

    /**
     * <p>
     * The <code>VPCOptions</code> for the specified domain. For more information, see <a
     * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html" target="_blank">VPC
     * Endpoints for Amazon Elasticsearch Service Domains</a>.
     * </p>
     * 
     * @return The <code>VPCOptions</code> for the specified domain. For more information, see <a
     *         href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html"
     *         target="_blank">VPC Endpoints for Amazon Elasticsearch Service Domains</a>.
     */
    public VPCDerivedInfo vpcOptions() {
        return vpcOptions;
    }

    /**
     * <p>
     * The <code>CognitoOptions</code> for the specified domain. For more information, see <a
     * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
     * target="_blank">Amazon Cognito Authentication for Kibana</a>.
     * </p>
     * 
     * @return The <code>CognitoOptions</code> for the specified domain. For more information, see <a
     *         href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
     *         target="_blank">Amazon Cognito Authentication for Kibana</a>.
     */
    public CognitoOptions cognitoOptions() {
        return cognitoOptions;
    }

    /**
     * <p>
     * Specifies the status of the <code>EncryptionAtRestOptions</code>.
     * </p>
     * 
     * @return Specifies the status of the <code>EncryptionAtRestOptions</code>.
     */
    public EncryptionAtRestOptions encryptionAtRestOptions() {
        return encryptionAtRestOptions;
    }

    /**
     * <p>
     * Specifies the status of the <code>NodeToNodeEncryptionOptions</code>.
     * </p>
     * 
     * @return Specifies the status of the <code>NodeToNodeEncryptionOptions</code>.
     */
    public NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions() {
        return nodeToNodeEncryptionOptions;
    }

    /**
     * Returns true if the AdvancedOptions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAdvancedOptions() {
        return advancedOptions != null && !(advancedOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the status of the <code>AdvancedOptions</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAdvancedOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the status of the <code>AdvancedOptions</code>
     */
    public Map<String, String> advancedOptions() {
        return advancedOptions;
    }

    /**
     * <p>
     * Log publishing options for the given domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogPublishingOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Log publishing options for the given domain.
     */
    public Map<LogType, LogPublishingOption> logPublishingOptions() {
        return LogPublishingOptionsCopier.copyStringToEnum(logPublishingOptions);
    }

    /**
     * Returns true if the LogPublishingOptions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasLogPublishingOptions() {
        return logPublishingOptions != null && !(logPublishingOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Log publishing options for the given domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogPublishingOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Log publishing options for the given domain.
     */
    public Map<String, LogPublishingOption> logPublishingOptionsAsStrings() {
        return logPublishingOptions;
    }

    /**
     * <p>
     * The current status of the Elasticsearch domain's service software.
     * </p>
     * 
     * @return The current status of the Elasticsearch domain's service software.
     */
    public ServiceSoftwareOptions serviceSoftwareOptions() {
        return serviceSoftwareOptions;
    }

    /**
     * <p>
     * The current status of the Elasticsearch domain's endpoint options.
     * </p>
     * 
     * @return The current status of the Elasticsearch domain's endpoint options.
     */
    public DomainEndpointOptions domainEndpointOptions() {
        return domainEndpointOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainId());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(created());
        hashCode = 31 * hashCode + Objects.hashCode(deleted());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(endpoints());
        hashCode = 31 * hashCode + Objects.hashCode(processing());
        hashCode = 31 * hashCode + Objects.hashCode(upgradeProcessing());
        hashCode = 31 * hashCode + Objects.hashCode(elasticsearchVersion());
        hashCode = 31 * hashCode + Objects.hashCode(elasticsearchClusterConfig());
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptions());
        hashCode = 31 * hashCode + Objects.hashCode(accessPolicies());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotOptions());
        hashCode = 31 * hashCode + Objects.hashCode(vpcOptions());
        hashCode = 31 * hashCode + Objects.hashCode(cognitoOptions());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAtRestOptions());
        hashCode = 31 * hashCode + Objects.hashCode(nodeToNodeEncryptionOptions());
        hashCode = 31 * hashCode + Objects.hashCode(advancedOptions());
        hashCode = 31 * hashCode + Objects.hashCode(logPublishingOptionsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(serviceSoftwareOptions());
        hashCode = 31 * hashCode + Objects.hashCode(domainEndpointOptions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ElasticsearchDomainStatus)) {
            return false;
        }
        ElasticsearchDomainStatus other = (ElasticsearchDomainStatus) obj;
        return Objects.equals(domainId(), other.domainId()) && Objects.equals(domainName(), other.domainName())
                && Objects.equals(arn(), other.arn()) && Objects.equals(created(), other.created())
                && Objects.equals(deleted(), other.deleted()) && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(endpoints(), other.endpoints()) && Objects.equals(processing(), other.processing())
                && Objects.equals(upgradeProcessing(), other.upgradeProcessing())
                && Objects.equals(elasticsearchVersion(), other.elasticsearchVersion())
                && Objects.equals(elasticsearchClusterConfig(), other.elasticsearchClusterConfig())
                && Objects.equals(ebsOptions(), other.ebsOptions()) && Objects.equals(accessPolicies(), other.accessPolicies())
                && Objects.equals(snapshotOptions(), other.snapshotOptions()) && Objects.equals(vpcOptions(), other.vpcOptions())
                && Objects.equals(cognitoOptions(), other.cognitoOptions())
                && Objects.equals(encryptionAtRestOptions(), other.encryptionAtRestOptions())
                && Objects.equals(nodeToNodeEncryptionOptions(), other.nodeToNodeEncryptionOptions())
                && Objects.equals(advancedOptions(), other.advancedOptions())
                && Objects.equals(logPublishingOptionsAsStrings(), other.logPublishingOptionsAsStrings())
                && Objects.equals(serviceSoftwareOptions(), other.serviceSoftwareOptions())
                && Objects.equals(domainEndpointOptions(), other.domainEndpointOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ElasticsearchDomainStatus").add("DomainId", domainId()).add("DomainName", domainName())
                .add("ARN", arn()).add("Created", created()).add("Deleted", deleted()).add("Endpoint", endpoint())
                .add("Endpoints", endpoints()).add("Processing", processing()).add("UpgradeProcessing", upgradeProcessing())
                .add("ElasticsearchVersion", elasticsearchVersion())
                .add("ElasticsearchClusterConfig", elasticsearchClusterConfig()).add("EBSOptions", ebsOptions())
                .add("AccessPolicies", accessPolicies()).add("SnapshotOptions", snapshotOptions())
                .add("VPCOptions", vpcOptions()).add("CognitoOptions", cognitoOptions())
                .add("EncryptionAtRestOptions", encryptionAtRestOptions())
                .add("NodeToNodeEncryptionOptions", nodeToNodeEncryptionOptions()).add("AdvancedOptions", advancedOptions())
                .add("LogPublishingOptions", logPublishingOptionsAsStrings())
                .add("ServiceSoftwareOptions", serviceSoftwareOptions()).add("DomainEndpointOptions", domainEndpointOptions())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainId":
            return Optional.ofNullable(clazz.cast(domainId()));
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Created":
            return Optional.ofNullable(clazz.cast(created()));
        case "Deleted":
            return Optional.ofNullable(clazz.cast(deleted()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "Processing":
            return Optional.ofNullable(clazz.cast(processing()));
        case "UpgradeProcessing":
            return Optional.ofNullable(clazz.cast(upgradeProcessing()));
        case "ElasticsearchVersion":
            return Optional.ofNullable(clazz.cast(elasticsearchVersion()));
        case "ElasticsearchClusterConfig":
            return Optional.ofNullable(clazz.cast(elasticsearchClusterConfig()));
        case "EBSOptions":
            return Optional.ofNullable(clazz.cast(ebsOptions()));
        case "AccessPolicies":
            return Optional.ofNullable(clazz.cast(accessPolicies()));
        case "SnapshotOptions":
            return Optional.ofNullable(clazz.cast(snapshotOptions()));
        case "VPCOptions":
            return Optional.ofNullable(clazz.cast(vpcOptions()));
        case "CognitoOptions":
            return Optional.ofNullable(clazz.cast(cognitoOptions()));
        case "EncryptionAtRestOptions":
            return Optional.ofNullable(clazz.cast(encryptionAtRestOptions()));
        case "NodeToNodeEncryptionOptions":
            return Optional.ofNullable(clazz.cast(nodeToNodeEncryptionOptions()));
        case "AdvancedOptions":
            return Optional.ofNullable(clazz.cast(advancedOptions()));
        case "LogPublishingOptions":
            return Optional.ofNullable(clazz.cast(logPublishingOptionsAsStrings()));
        case "ServiceSoftwareOptions":
            return Optional.ofNullable(clazz.cast(serviceSoftwareOptions()));
        case "DomainEndpointOptions":
            return Optional.ofNullable(clazz.cast(domainEndpointOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ElasticsearchDomainStatus, T> g) {
        return obj -> g.apply((ElasticsearchDomainStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ElasticsearchDomainStatus> {
        /**
         * <p>
         * The unique identifier for the specified Elasticsearch domain.
         * </p>
         * 
         * @param domainId
         *        The unique identifier for the specified Elasticsearch domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainId(String domainId);

        /**
         * <p>
         * The name of an Elasticsearch domain. Domain names are unique across the domains owned by an account within an
         * AWS region. Domain names start with a letter or number and can contain the following characters: a-z
         * (lowercase), 0-9, and - (hyphen).
         * </p>
         * 
         * @param domainName
         *        The name of an Elasticsearch domain. Domain names are unique across the domains owned by an account
         *        within an AWS region. Domain names start with a letter or number and can contain the following
         *        characters: a-z (lowercase), 0-9, and - (hyphen).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The Amazon resource name (ARN) of an Elasticsearch domain. See <a
         * href="http://docs.aws.amazon.com/IAM/latest/UserGuide/index.html?Using_Identifiers.html"
         * target="_blank">Identifiers for IAM Entities</a> in <i>Using AWS Identity and Access Management</i> for more
         * information.
         * </p>
         * 
         * @param arn
         *        The Amazon resource name (ARN) of an Elasticsearch domain. See <a
         *        href="http://docs.aws.amazon.com/IAM/latest/UserGuide/index.html?Using_Identifiers.html"
         *        target="_blank">Identifiers for IAM Entities</a> in <i>Using AWS Identity and Access Management</i>
         *        for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The domain creation status. <code>True</code> if the creation of an Elasticsearch domain is complete.
         * <code>False</code> if domain creation is still in progress.
         * </p>
         * 
         * @param created
         *        The domain creation status. <code>True</code> if the creation of an Elasticsearch domain is complete.
         *        <code>False</code> if domain creation is still in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder created(Boolean created);

        /**
         * <p>
         * The domain deletion status. <code>True</code> if a delete request has been received for the domain but
         * resource cleanup is still in progress. <code>False</code> if the domain has not been deleted. Once domain
         * deletion is complete, the status of the domain is no longer returned.
         * </p>
         * 
         * @param deleted
         *        The domain deletion status. <code>True</code> if a delete request has been received for the domain but
         *        resource cleanup is still in progress. <code>False</code> if the domain has not been deleted. Once
         *        domain deletion is complete, the status of the domain is no longer returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleted(Boolean deleted);

        /**
         * <p>
         * The Elasticsearch domain endpoint that you use to submit index and search requests.
         * </p>
         * 
         * @param endpoint
         *        The Elasticsearch domain endpoint that you use to submit index and search requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * Map containing the Elasticsearch domain endpoints used to submit index and search requests. Example
         * <code>key, value</code>:
         * <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
         * </p>
         * 
         * @param endpoints
         *        Map containing the Elasticsearch domain endpoints used to submit index and search requests. Example
         *        <code>key, value</code>:
         *        <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Map<String, String> endpoints);

        /**
         * <p>
         * The status of the Elasticsearch domain configuration. <code>True</code> if Amazon Elasticsearch Service is
         * processing configuration changes. <code>False</code> if the configuration is active.
         * </p>
         * 
         * @param processing
         *        The status of the Elasticsearch domain configuration. <code>True</code> if Amazon Elasticsearch
         *        Service is processing configuration changes. <code>False</code> if the configuration is active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processing(Boolean processing);

        /**
         * <p>
         * The status of an Elasticsearch domain version upgrade. <code>True</code> if Amazon Elasticsearch Service is
         * undergoing a version upgrade. <code>False</code> if the configuration is active.
         * </p>
         * 
         * @param upgradeProcessing
         *        The status of an Elasticsearch domain version upgrade. <code>True</code> if Amazon Elasticsearch
         *        Service is undergoing a version upgrade. <code>False</code> if the configuration is active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upgradeProcessing(Boolean upgradeProcessing);

        /**
         * Sets the value of the ElasticsearchVersion property for this object.
         *
         * @param elasticsearchVersion
         *        The new value for the ElasticsearchVersion property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder elasticsearchVersion(String elasticsearchVersion);

        /**
         * <p>
         * The type and number of instances in the domain cluster.
         * </p>
         * 
         * @param elasticsearchClusterConfig
         *        The type and number of instances in the domain cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder elasticsearchClusterConfig(ElasticsearchClusterConfig elasticsearchClusterConfig);

        /**
         * <p>
         * The type and number of instances in the domain cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link ElasticsearchClusterConfig.Builder} avoiding the
         * need to create one manually via {@link ElasticsearchClusterConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ElasticsearchClusterConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #elasticsearchClusterConfig(ElasticsearchClusterConfig)}.
         * 
         * @param elasticsearchClusterConfig
         *        a consumer that will call methods on {@link ElasticsearchClusterConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #elasticsearchClusterConfig(ElasticsearchClusterConfig)
         */
        default Builder elasticsearchClusterConfig(Consumer<ElasticsearchClusterConfig.Builder> elasticsearchClusterConfig) {
            return elasticsearchClusterConfig(ElasticsearchClusterConfig.builder().applyMutation(elasticsearchClusterConfig)
                    .build());
        }

        /**
         * <p>
         * The <code>EBSOptions</code> for the specified domain. See <a href=
         * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs"
         * target="_blank">Configuring EBS-based Storage</a> for more information.
         * </p>
         * 
         * @param ebsOptions
         *        The <code>EBSOptions</code> for the specified domain. See <a href=
         *        "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs"
         *        target="_blank">Configuring EBS-based Storage</a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptions(EBSOptions ebsOptions);

        /**
         * <p>
         * The <code>EBSOptions</code> for the specified domain. See <a href=
         * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs"
         * target="_blank">Configuring EBS-based Storage</a> for more information.
         * </p>
         * This is a convenience that creates an instance of the {@link EBSOptions.Builder} avoiding the need to create
         * one manually via {@link EBSOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link EBSOptions.Builder#build()} is called immediately and its result
         * is passed to {@link #ebsOptions(EBSOptions)}.
         * 
         * @param ebsOptions
         *        a consumer that will call methods on {@link EBSOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsOptions(EBSOptions)
         */
        default Builder ebsOptions(Consumer<EBSOptions.Builder> ebsOptions) {
            return ebsOptions(EBSOptions.builder().applyMutation(ebsOptions).build());
        }

        /**
         * <p>
         * IAM access policy as a JSON-formatted string.
         * </p>
         * 
         * @param accessPolicies
         *        IAM access policy as a JSON-formatted string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessPolicies(String accessPolicies);

        /**
         * <p>
         * Specifies the status of the <code>SnapshotOptions</code>
         * </p>
         * 
         * @param snapshotOptions
         *        Specifies the status of the <code>SnapshotOptions</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotOptions(SnapshotOptions snapshotOptions);

        /**
         * <p>
         * Specifies the status of the <code>SnapshotOptions</code>
         * </p>
         * This is a convenience that creates an instance of the {@link SnapshotOptions.Builder} avoiding the need to
         * create one manually via {@link SnapshotOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link SnapshotOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #snapshotOptions(SnapshotOptions)}.
         * 
         * @param snapshotOptions
         *        a consumer that will call methods on {@link SnapshotOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snapshotOptions(SnapshotOptions)
         */
        default Builder snapshotOptions(Consumer<SnapshotOptions.Builder> snapshotOptions) {
            return snapshotOptions(SnapshotOptions.builder().applyMutation(snapshotOptions).build());
        }

        /**
         * <p>
         * The <code>VPCOptions</code> for the specified domain. For more information, see <a
         * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html" target="_blank">VPC
         * Endpoints for Amazon Elasticsearch Service Domains</a>.
         * </p>
         * 
         * @param vpcOptions
         *        The <code>VPCOptions</code> for the specified domain. For more information, see <a
         *        href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html"
         *        target="_blank">VPC Endpoints for Amazon Elasticsearch Service Domains</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcOptions(VPCDerivedInfo vpcOptions);

        /**
         * <p>
         * The <code>VPCOptions</code> for the specified domain. For more information, see <a
         * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html" target="_blank">VPC
         * Endpoints for Amazon Elasticsearch Service Domains</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link VPCDerivedInfo.Builder} avoiding the need to
         * create one manually via {@link VPCDerivedInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link VPCDerivedInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcOptions(VPCDerivedInfo)}.
         * 
         * @param vpcOptions
         *        a consumer that will call methods on {@link VPCDerivedInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcOptions(VPCDerivedInfo)
         */
        default Builder vpcOptions(Consumer<VPCDerivedInfo.Builder> vpcOptions) {
            return vpcOptions(VPCDerivedInfo.builder().applyMutation(vpcOptions).build());
        }

        /**
         * <p>
         * The <code>CognitoOptions</code> for the specified domain. For more information, see <a
         * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
         * target="_blank">Amazon Cognito Authentication for Kibana</a>.
         * </p>
         * 
         * @param cognitoOptions
         *        The <code>CognitoOptions</code> for the specified domain. For more information, see <a
         *        href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
         *        target="_blank">Amazon Cognito Authentication for Kibana</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cognitoOptions(CognitoOptions cognitoOptions);

        /**
         * <p>
         * The <code>CognitoOptions</code> for the specified domain. For more information, see <a
         * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
         * target="_blank">Amazon Cognito Authentication for Kibana</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link CognitoOptions.Builder} avoiding the need to
         * create one manually via {@link CognitoOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link CognitoOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #cognitoOptions(CognitoOptions)}.
         * 
         * @param cognitoOptions
         *        a consumer that will call methods on {@link CognitoOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cognitoOptions(CognitoOptions)
         */
        default Builder cognitoOptions(Consumer<CognitoOptions.Builder> cognitoOptions) {
            return cognitoOptions(CognitoOptions.builder().applyMutation(cognitoOptions).build());
        }

        /**
         * <p>
         * Specifies the status of the <code>EncryptionAtRestOptions</code>.
         * </p>
         * 
         * @param encryptionAtRestOptions
         *        Specifies the status of the <code>EncryptionAtRestOptions</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions);

        /**
         * <p>
         * Specifies the status of the <code>EncryptionAtRestOptions</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link EncryptionAtRestOptions.Builder} avoiding the
         * need to create one manually via {@link EncryptionAtRestOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionAtRestOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionAtRestOptions(EncryptionAtRestOptions)}.
         * 
         * @param encryptionAtRestOptions
         *        a consumer that will call methods on {@link EncryptionAtRestOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionAtRestOptions(EncryptionAtRestOptions)
         */
        default Builder encryptionAtRestOptions(Consumer<EncryptionAtRestOptions.Builder> encryptionAtRestOptions) {
            return encryptionAtRestOptions(EncryptionAtRestOptions.builder().applyMutation(encryptionAtRestOptions).build());
        }

        /**
         * <p>
         * Specifies the status of the <code>NodeToNodeEncryptionOptions</code>.
         * </p>
         * 
         * @param nodeToNodeEncryptionOptions
         *        Specifies the status of the <code>NodeToNodeEncryptionOptions</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions);

        /**
         * <p>
         * Specifies the status of the <code>NodeToNodeEncryptionOptions</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link NodeToNodeEncryptionOptions.Builder} avoiding
         * the need to create one manually via {@link NodeToNodeEncryptionOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link NodeToNodeEncryptionOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions)}.
         * 
         * @param nodeToNodeEncryptionOptions
         *        a consumer that will call methods on {@link NodeToNodeEncryptionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions)
         */
        default Builder nodeToNodeEncryptionOptions(Consumer<NodeToNodeEncryptionOptions.Builder> nodeToNodeEncryptionOptions) {
            return nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions.builder().applyMutation(nodeToNodeEncryptionOptions)
                    .build());
        }

        /**
         * <p>
         * Specifies the status of the <code>AdvancedOptions</code>
         * </p>
         * 
         * @param advancedOptions
         *        Specifies the status of the <code>AdvancedOptions</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedOptions(Map<String, String> advancedOptions);

        /**
         * <p>
         * Log publishing options for the given domain.
         * </p>
         * 
         * @param logPublishingOptions
         *        Log publishing options for the given domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptionsWithStrings(Map<String, LogPublishingOption> logPublishingOptions);

        /**
         * <p>
         * Log publishing options for the given domain.
         * </p>
         * 
         * @param logPublishingOptions
         *        Log publishing options for the given domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptions(Map<LogType, LogPublishingOption> logPublishingOptions);

        /**
         * <p>
         * The current status of the Elasticsearch domain's service software.
         * </p>
         * 
         * @param serviceSoftwareOptions
         *        The current status of the Elasticsearch domain's service software.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceSoftwareOptions(ServiceSoftwareOptions serviceSoftwareOptions);

        /**
         * <p>
         * The current status of the Elasticsearch domain's service software.
         * </p>
         * This is a convenience that creates an instance of the {@link ServiceSoftwareOptions.Builder} avoiding the
         * need to create one manually via {@link ServiceSoftwareOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServiceSoftwareOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #serviceSoftwareOptions(ServiceSoftwareOptions)}.
         * 
         * @param serviceSoftwareOptions
         *        a consumer that will call methods on {@link ServiceSoftwareOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceSoftwareOptions(ServiceSoftwareOptions)
         */
        default Builder serviceSoftwareOptions(Consumer<ServiceSoftwareOptions.Builder> serviceSoftwareOptions) {
            return serviceSoftwareOptions(ServiceSoftwareOptions.builder().applyMutation(serviceSoftwareOptions).build());
        }

        /**
         * <p>
         * The current status of the Elasticsearch domain's endpoint options.
         * </p>
         * 
         * @param domainEndpointOptions
         *        The current status of the Elasticsearch domain's endpoint options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainEndpointOptions(DomainEndpointOptions domainEndpointOptions);

        /**
         * <p>
         * The current status of the Elasticsearch domain's endpoint options.
         * </p>
         * This is a convenience that creates an instance of the {@link DomainEndpointOptions.Builder} avoiding the need
         * to create one manually via {@link DomainEndpointOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link DomainEndpointOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #domainEndpointOptions(DomainEndpointOptions)}.
         * 
         * @param domainEndpointOptions
         *        a consumer that will call methods on {@link DomainEndpointOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainEndpointOptions(DomainEndpointOptions)
         */
        default Builder domainEndpointOptions(Consumer<DomainEndpointOptions.Builder> domainEndpointOptions) {
            return domainEndpointOptions(DomainEndpointOptions.builder().applyMutation(domainEndpointOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String domainId;

        private String domainName;

        private String arn;

        private Boolean created;

        private Boolean deleted;

        private String endpoint;

        private Map<String, String> endpoints = DefaultSdkAutoConstructMap.getInstance();

        private Boolean processing;

        private Boolean upgradeProcessing;

        private String elasticsearchVersion;

        private ElasticsearchClusterConfig elasticsearchClusterConfig;

        private EBSOptions ebsOptions;

        private String accessPolicies;

        private SnapshotOptions snapshotOptions;

        private VPCDerivedInfo vpcOptions;

        private CognitoOptions cognitoOptions;

        private EncryptionAtRestOptions encryptionAtRestOptions;

        private NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions;

        private Map<String, String> advancedOptions = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, LogPublishingOption> logPublishingOptions = DefaultSdkAutoConstructMap.getInstance();

        private ServiceSoftwareOptions serviceSoftwareOptions;

        private DomainEndpointOptions domainEndpointOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(ElasticsearchDomainStatus model) {
            domainId(model.domainId);
            domainName(model.domainName);
            arn(model.arn);
            created(model.created);
            deleted(model.deleted);
            endpoint(model.endpoint);
            endpoints(model.endpoints);
            processing(model.processing);
            upgradeProcessing(model.upgradeProcessing);
            elasticsearchVersion(model.elasticsearchVersion);
            elasticsearchClusterConfig(model.elasticsearchClusterConfig);
            ebsOptions(model.ebsOptions);
            accessPolicies(model.accessPolicies);
            snapshotOptions(model.snapshotOptions);
            vpcOptions(model.vpcOptions);
            cognitoOptions(model.cognitoOptions);
            encryptionAtRestOptions(model.encryptionAtRestOptions);
            nodeToNodeEncryptionOptions(model.nodeToNodeEncryptionOptions);
            advancedOptions(model.advancedOptions);
            logPublishingOptionsWithStrings(model.logPublishingOptions);
            serviceSoftwareOptions(model.serviceSoftwareOptions);
            domainEndpointOptions(model.domainEndpointOptions);
        }

        public final String getDomainId() {
            return domainId;
        }

        @Override
        public final Builder domainId(String domainId) {
            this.domainId = domainId;
            return this;
        }

        public final void setDomainId(String domainId) {
            this.domainId = domainId;
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Boolean getCreated() {
            return created;
        }

        @Override
        public final Builder created(Boolean created) {
            this.created = created;
            return this;
        }

        public final void setCreated(Boolean created) {
            this.created = created;
        }

        public final Boolean getDeleted() {
            return deleted;
        }

        @Override
        public final Builder deleted(Boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public final void setDeleted(Boolean deleted) {
            this.deleted = deleted;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        public final Map<String, String> getEndpoints() {
            return endpoints;
        }

        @Override
        public final Builder endpoints(Map<String, String> endpoints) {
            this.endpoints = EndpointsMapCopier.copy(endpoints);
            return this;
        }

        public final void setEndpoints(Map<String, String> endpoints) {
            this.endpoints = EndpointsMapCopier.copy(endpoints);
        }

        public final Boolean getProcessing() {
            return processing;
        }

        @Override
        public final Builder processing(Boolean processing) {
            this.processing = processing;
            return this;
        }

        public final void setProcessing(Boolean processing) {
            this.processing = processing;
        }

        public final Boolean getUpgradeProcessing() {
            return upgradeProcessing;
        }

        @Override
        public final Builder upgradeProcessing(Boolean upgradeProcessing) {
            this.upgradeProcessing = upgradeProcessing;
            return this;
        }

        public final void setUpgradeProcessing(Boolean upgradeProcessing) {
            this.upgradeProcessing = upgradeProcessing;
        }

        public final String getElasticsearchVersion() {
            return elasticsearchVersion;
        }

        @Override
        public final Builder elasticsearchVersion(String elasticsearchVersion) {
            this.elasticsearchVersion = elasticsearchVersion;
            return this;
        }

        public final void setElasticsearchVersion(String elasticsearchVersion) {
            this.elasticsearchVersion = elasticsearchVersion;
        }

        public final ElasticsearchClusterConfig.Builder getElasticsearchClusterConfig() {
            return elasticsearchClusterConfig != null ? elasticsearchClusterConfig.toBuilder() : null;
        }

        @Override
        public final Builder elasticsearchClusterConfig(ElasticsearchClusterConfig elasticsearchClusterConfig) {
            this.elasticsearchClusterConfig = elasticsearchClusterConfig;
            return this;
        }

        public final void setElasticsearchClusterConfig(ElasticsearchClusterConfig.BuilderImpl elasticsearchClusterConfig) {
            this.elasticsearchClusterConfig = elasticsearchClusterConfig != null ? elasticsearchClusterConfig.build() : null;
        }

        public final EBSOptions.Builder getEbsOptions() {
            return ebsOptions != null ? ebsOptions.toBuilder() : null;
        }

        @Override
        public final Builder ebsOptions(EBSOptions ebsOptions) {
            this.ebsOptions = ebsOptions;
            return this;
        }

        public final void setEbsOptions(EBSOptions.BuilderImpl ebsOptions) {
            this.ebsOptions = ebsOptions != null ? ebsOptions.build() : null;
        }

        public final String getAccessPolicies() {
            return accessPolicies;
        }

        @Override
        public final Builder accessPolicies(String accessPolicies) {
            this.accessPolicies = accessPolicies;
            return this;
        }

        public final void setAccessPolicies(String accessPolicies) {
            this.accessPolicies = accessPolicies;
        }

        public final SnapshotOptions.Builder getSnapshotOptions() {
            return snapshotOptions != null ? snapshotOptions.toBuilder() : null;
        }

        @Override
        public final Builder snapshotOptions(SnapshotOptions snapshotOptions) {
            this.snapshotOptions = snapshotOptions;
            return this;
        }

        public final void setSnapshotOptions(SnapshotOptions.BuilderImpl snapshotOptions) {
            this.snapshotOptions = snapshotOptions != null ? snapshotOptions.build() : null;
        }

        public final VPCDerivedInfo.Builder getVpcOptions() {
            return vpcOptions != null ? vpcOptions.toBuilder() : null;
        }

        @Override
        public final Builder vpcOptions(VPCDerivedInfo vpcOptions) {
            this.vpcOptions = vpcOptions;
            return this;
        }

        public final void setVpcOptions(VPCDerivedInfo.BuilderImpl vpcOptions) {
            this.vpcOptions = vpcOptions != null ? vpcOptions.build() : null;
        }

        public final CognitoOptions.Builder getCognitoOptions() {
            return cognitoOptions != null ? cognitoOptions.toBuilder() : null;
        }

        @Override
        public final Builder cognitoOptions(CognitoOptions cognitoOptions) {
            this.cognitoOptions = cognitoOptions;
            return this;
        }

        public final void setCognitoOptions(CognitoOptions.BuilderImpl cognitoOptions) {
            this.cognitoOptions = cognitoOptions != null ? cognitoOptions.build() : null;
        }

        public final EncryptionAtRestOptions.Builder getEncryptionAtRestOptions() {
            return encryptionAtRestOptions != null ? encryptionAtRestOptions.toBuilder() : null;
        }

        @Override
        public final Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions;
            return this;
        }

        public final void setEncryptionAtRestOptions(EncryptionAtRestOptions.BuilderImpl encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions != null ? encryptionAtRestOptions.build() : null;
        }

        public final NodeToNodeEncryptionOptions.Builder getNodeToNodeEncryptionOptions() {
            return nodeToNodeEncryptionOptions != null ? nodeToNodeEncryptionOptions.toBuilder() : null;
        }

        @Override
        public final Builder nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions) {
            this.nodeToNodeEncryptionOptions = nodeToNodeEncryptionOptions;
            return this;
        }

        public final void setNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions.BuilderImpl nodeToNodeEncryptionOptions) {
            this.nodeToNodeEncryptionOptions = nodeToNodeEncryptionOptions != null ? nodeToNodeEncryptionOptions.build() : null;
        }

        public final Map<String, String> getAdvancedOptions() {
            return advancedOptions;
        }

        @Override
        public final Builder advancedOptions(Map<String, String> advancedOptions) {
            this.advancedOptions = AdvancedOptionsCopier.copy(advancedOptions);
            return this;
        }

        public final void setAdvancedOptions(Map<String, String> advancedOptions) {
            this.advancedOptions = AdvancedOptionsCopier.copy(advancedOptions);
        }

        public final Map<String, LogPublishingOption.Builder> getLogPublishingOptionsAsStrings() {
            return logPublishingOptions != null ? CollectionUtils.mapValues(logPublishingOptions, LogPublishingOption::toBuilder)
                    : null;
        }

        @Override
        public final Builder logPublishingOptionsWithStrings(Map<String, LogPublishingOption> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copy(logPublishingOptions);
            return this;
        }

        @Override
        public final Builder logPublishingOptions(Map<LogType, LogPublishingOption> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copyEnumToString(logPublishingOptions);
            return this;
        }

        public final void setLogPublishingOptionsWithStrings(Map<String, LogPublishingOption.BuilderImpl> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copyFromBuilder(logPublishingOptions);
        }

        public final ServiceSoftwareOptions.Builder getServiceSoftwareOptions() {
            return serviceSoftwareOptions != null ? serviceSoftwareOptions.toBuilder() : null;
        }

        @Override
        public final Builder serviceSoftwareOptions(ServiceSoftwareOptions serviceSoftwareOptions) {
            this.serviceSoftwareOptions = serviceSoftwareOptions;
            return this;
        }

        public final void setServiceSoftwareOptions(ServiceSoftwareOptions.BuilderImpl serviceSoftwareOptions) {
            this.serviceSoftwareOptions = serviceSoftwareOptions != null ? serviceSoftwareOptions.build() : null;
        }

        public final DomainEndpointOptions.Builder getDomainEndpointOptions() {
            return domainEndpointOptions != null ? domainEndpointOptions.toBuilder() : null;
        }

        @Override
        public final Builder domainEndpointOptions(DomainEndpointOptions domainEndpointOptions) {
            this.domainEndpointOptions = domainEndpointOptions;
            return this;
        }

        public final void setDomainEndpointOptions(DomainEndpointOptions.BuilderImpl domainEndpointOptions) {
            this.domainEndpointOptions = domainEndpointOptions != null ? domainEndpointOptions.build() : null;
        }

        @Override
        public ElasticsearchDomainStatus build() {
            return new ElasticsearchDomainStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
