/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Node group (shard) configuration options when adding or removing replicas. Each node group (shard) configuration has
 * the following members: NodeGroupId, NewReplicaCount, and PreferredAvailabilityZones.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfigureShard implements SdkPojo, Serializable, ToCopyableBuilder<ConfigureShard.Builder, ConfigureShard> {
    private static final SdkField<String> NODE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ConfigureShard::nodeGroupId)).setter(setter(Builder::nodeGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupId").build()).build();

    private static final SdkField<Integer> NEW_REPLICA_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ConfigureShard::newReplicaCount)).setter(setter(Builder::newReplicaCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewReplicaCount").build()).build();

    private static final SdkField<List<String>> PREFERRED_AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ConfigureShard::preferredAvailabilityZones))
            .setter(setter(Builder::preferredAvailabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredAvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("PreferredAvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("PreferredAvailabilityZone").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NODE_GROUP_ID_FIELD,
            NEW_REPLICA_COUNT_FIELD, PREFERRED_AVAILABILITY_ZONES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nodeGroupId;

    private final Integer newReplicaCount;

    private final List<String> preferredAvailabilityZones;

    private ConfigureShard(BuilderImpl builder) {
        this.nodeGroupId = builder.nodeGroupId;
        this.newReplicaCount = builder.newReplicaCount;
        this.preferredAvailabilityZones = builder.preferredAvailabilityZones;
    }

    /**
     * <p>
     * The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication groups, the
     * node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's (shard's) id, see <a
     * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding a Shard's Id</a>.
     * </p>
     * 
     * @return The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication
     *         groups, the node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's (shard's)
     *         id, see <a href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding
     *         a Shard's Id</a>.
     */
    public String nodeGroupId() {
        return nodeGroupId;
    }

    /**
     * <p>
     * The number of replicas you want in this node group at the end of this operation. The maximum value for
     * <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group you are
     * working with.
     * </p>
     * <p>
     * The minimum number of replicas in a shard or replication group is:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Redis (cluster mode disabled)
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Multi-AZ with Automatic Failover is enabled: 1
     * </p>
     * </li>
     * <li>
     * <p>
     * If Multi-AZ with Automatic Failover is not enable: 0
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary node fails)
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of replicas you want in this node group at the end of this operation. The maximum value for
     *         <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group you
     *         are working with.</p>
     *         <p>
     *         The minimum number of replicas in a shard or replication group is:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Redis (cluster mode disabled)
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Multi-AZ with Automatic Failover is enabled: 1
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If Multi-AZ with Automatic Failover is not enable: 0
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary
     *         node fails)
     *         </p>
     *         </li>
     */
    public Integer newReplicaCount() {
        return newReplicaCount;
    }

    /**
     * Returns true if the PreferredAvailabilityZones property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasPreferredAvailabilityZones() {
        return preferredAvailabilityZones != null && !(preferredAvailabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the replication
     * group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must equal the value of
     * <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
     * <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for each of the
     * replicas.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPreferredAvailabilityZones()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
     *         replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must
     *         equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
     *         <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for
     *         each of the replicas.
     */
    public List<String> preferredAvailabilityZones() {
        return preferredAvailabilityZones;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nodeGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(newReplicaCount());
        hashCode = 31 * hashCode + Objects.hashCode(preferredAvailabilityZones());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigureShard)) {
            return false;
        }
        ConfigureShard other = (ConfigureShard) obj;
        return Objects.equals(nodeGroupId(), other.nodeGroupId()) && Objects.equals(newReplicaCount(), other.newReplicaCount())
                && Objects.equals(preferredAvailabilityZones(), other.preferredAvailabilityZones());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ConfigureShard").add("NodeGroupId", nodeGroupId()).add("NewReplicaCount", newReplicaCount())
                .add("PreferredAvailabilityZones", preferredAvailabilityZones()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NodeGroupId":
            return Optional.ofNullable(clazz.cast(nodeGroupId()));
        case "NewReplicaCount":
            return Optional.ofNullable(clazz.cast(newReplicaCount()));
        case "PreferredAvailabilityZones":
            return Optional.ofNullable(clazz.cast(preferredAvailabilityZones()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConfigureShard, T> g) {
        return obj -> g.apply((ConfigureShard) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfigureShard> {
        /**
         * <p>
         * The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication groups,
         * the node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's (shard's) id, see <a
         * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding a Shard's
         * Id</a>.
         * </p>
         * 
         * @param nodeGroupId
         *        The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication
         *        groups, the node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's
         *        (shard's) id, see <a
         *        href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding a
         *        Shard's Id</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupId(String nodeGroupId);

        /**
         * <p>
         * The number of replicas you want in this node group at the end of this operation. The maximum value for
         * <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group you are
         * working with.
         * </p>
         * <p>
         * The minimum number of replicas in a shard or replication group is:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Redis (cluster mode disabled)
         * </p>
         * <ul>
         * <li>
         * <p>
         * If Multi-AZ with Automatic Failover is enabled: 1
         * </p>
         * </li>
         * <li>
         * <p>
         * If Multi-AZ with Automatic Failover is not enable: 0
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary node
         * fails)
         * </p>
         * </li>
         * </ul>
         * 
         * @param newReplicaCount
         *        The number of replicas you want in this node group at the end of this operation. The maximum value for
         *        <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group
         *        you are working with.</p>
         *        <p>
         *        The minimum number of replicas in a shard or replication group is:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Redis (cluster mode disabled)
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If Multi-AZ with Automatic Failover is enabled: 1
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If Multi-AZ with Automatic Failover is not enable: 0
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary
         *        node fails)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newReplicaCount(Integer newReplicaCount);

        /**
         * <p>
         * A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         * replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must
         * equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
         * <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for each of
         * the replicas.
         * </p>
         * 
         * @param preferredAvailabilityZones
         *        A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         *        replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values
         *        must equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this
         *        member of <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability
         *        zone for each of the replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZones(Collection<String> preferredAvailabilityZones);

        /**
         * <p>
         * A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         * replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must
         * equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
         * <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for each of
         * the replicas.
         * </p>
         * 
         * @param preferredAvailabilityZones
         *        A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         *        replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values
         *        must equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this
         *        member of <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability
         *        zone for each of the replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZones(String... preferredAvailabilityZones);
    }

    static final class BuilderImpl implements Builder {
        private String nodeGroupId;

        private Integer newReplicaCount;

        private List<String> preferredAvailabilityZones = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigureShard model) {
            nodeGroupId(model.nodeGroupId);
            newReplicaCount(model.newReplicaCount);
            preferredAvailabilityZones(model.preferredAvailabilityZones);
        }

        public final String getNodeGroupId() {
            return nodeGroupId;
        }

        @Override
        public final Builder nodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
            return this;
        }

        public final void setNodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
        }

        public final Integer getNewReplicaCount() {
            return newReplicaCount;
        }

        @Override
        public final Builder newReplicaCount(Integer newReplicaCount) {
            this.newReplicaCount = newReplicaCount;
            return this;
        }

        public final void setNewReplicaCount(Integer newReplicaCount) {
            this.newReplicaCount = newReplicaCount;
        }

        public final Collection<String> getPreferredAvailabilityZones() {
            return preferredAvailabilityZones;
        }

        @Override
        public final Builder preferredAvailabilityZones(Collection<String> preferredAvailabilityZones) {
            this.preferredAvailabilityZones = PreferredAvailabilityZoneListCopier.copy(preferredAvailabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredAvailabilityZones(String... preferredAvailabilityZones) {
            preferredAvailabilityZones(Arrays.asList(preferredAvailabilityZones));
            return this;
        }

        public final void setPreferredAvailabilityZones(Collection<String> preferredAvailabilityZones) {
            this.preferredAvailabilityZones = PreferredAvailabilityZoneListCopier.copy(preferredAvailabilityZones);
        }

        @Override
        public ConfigureShard build() {
            return new ConfigureShard(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
