/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of one of the following operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>CreateCacheSubnetGroup</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ModifyCacheSubnetGroup</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CacheSubnetGroup implements SdkPojo, Serializable,
        ToCopyableBuilder<CacheSubnetGroup.Builder, CacheSubnetGroup> {
    private static final SdkField<String> CACHE_SUBNET_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheSubnetGroup::cacheSubnetGroupName)).setter(setter(Builder::cacheSubnetGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheSubnetGroupName").build())
            .build();

    private static final SdkField<String> CACHE_SUBNET_GROUP_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CacheSubnetGroup::cacheSubnetGroupDescription))
            .setter(setter(Builder::cacheSubnetGroupDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheSubnetGroupDescription")
                    .build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheSubnetGroup::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<Subnet>> SUBNETS_FIELD = SdkField
            .<List<Subnet>> builder(MarshallingType.LIST)
            .getter(getter(CacheSubnetGroup::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Subnet")
                            .memberFieldInfo(
                                    SdkField.<Subnet> builder(MarshallingType.SDK_POJO)
                                            .constructor(Subnet::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Subnet").build()).build()).build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheSubnetGroup::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CACHE_SUBNET_GROUP_NAME_FIELD,
            CACHE_SUBNET_GROUP_DESCRIPTION_FIELD, VPC_ID_FIELD, SUBNETS_FIELD, ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String cacheSubnetGroupName;

    private final String cacheSubnetGroupDescription;

    private final String vpcId;

    private final List<Subnet> subnets;

    private final String arn;

    private CacheSubnetGroup(BuilderImpl builder) {
        this.cacheSubnetGroupName = builder.cacheSubnetGroupName;
        this.cacheSubnetGroupDescription = builder.cacheSubnetGroupDescription;
        this.vpcId = builder.vpcId;
        this.subnets = builder.subnets;
        this.arn = builder.arn;
    }

    /**
     * <p>
     * The name of the cache subnet group.
     * </p>
     * 
     * @return The name of the cache subnet group.
     */
    public String cacheSubnetGroupName() {
        return cacheSubnetGroupName;
    }

    /**
     * <p>
     * The description of the cache subnet group.
     * </p>
     * 
     * @return The description of the cache subnet group.
     */
    public String cacheSubnetGroupDescription() {
        return cacheSubnetGroupDescription;
    }

    /**
     * <p>
     * The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group.
     * </p>
     * 
     * @return The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group.
     */
    public String vpcId() {
        return vpcId;
    }

    /**
     * Returns true if the Subnets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of subnets associated with the cache subnet group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of subnets associated with the cache subnet group.
     */
    public List<Subnet> subnets() {
        return subnets;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the cache subnet group.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the cache subnet group.
     */
    public String arn() {
        return arn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cacheSubnetGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(cacheSubnetGroupDescription());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(subnets());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CacheSubnetGroup)) {
            return false;
        }
        CacheSubnetGroup other = (CacheSubnetGroup) obj;
        return Objects.equals(cacheSubnetGroupName(), other.cacheSubnetGroupName())
                && Objects.equals(cacheSubnetGroupDescription(), other.cacheSubnetGroupDescription())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(subnets(), other.subnets())
                && Objects.equals(arn(), other.arn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CacheSubnetGroup").add("CacheSubnetGroupName", cacheSubnetGroupName())
                .add("CacheSubnetGroupDescription", cacheSubnetGroupDescription()).add("VpcId", vpcId())
                .add("Subnets", subnets()).add("ARN", arn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CacheSubnetGroupName":
            return Optional.ofNullable(clazz.cast(cacheSubnetGroupName()));
        case "CacheSubnetGroupDescription":
            return Optional.ofNullable(clazz.cast(cacheSubnetGroupDescription()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "Subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CacheSubnetGroup, T> g) {
        return obj -> g.apply((CacheSubnetGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CacheSubnetGroup> {
        /**
         * <p>
         * The name of the cache subnet group.
         * </p>
         * 
         * @param cacheSubnetGroupName
         *        The name of the cache subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheSubnetGroupName(String cacheSubnetGroupName);

        /**
         * <p>
         * The description of the cache subnet group.
         * </p>
         * 
         * @param cacheSubnetGroupDescription
         *        The description of the cache subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheSubnetGroupDescription(String cacheSubnetGroupDescription);

        /**
         * <p>
         * The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group.
         * </p>
         * 
         * @param vpcId
         *        The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * A list of subnets associated with the cache subnet group.
         * </p>
         * 
         * @param subnets
         *        A list of subnets associated with the cache subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<Subnet> subnets);

        /**
         * <p>
         * A list of subnets associated with the cache subnet group.
         * </p>
         * 
         * @param subnets
         *        A list of subnets associated with the cache subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Subnet... subnets);

        /**
         * <p>
         * A list of subnets associated with the cache subnet group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Subnet>.Builder} avoiding the need to
         * create one manually via {@link List<Subnet>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Subnet>.Builder#build()} is called immediately and its
         * result is passed to {@link #subnets(List<Subnet>)}.
         * 
         * @param subnets
         *        a consumer that will call methods on {@link List<Subnet>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subnets(List<Subnet>)
         */
        Builder subnets(Consumer<Subnet.Builder>... subnets);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the cache subnet group.
         * </p>
         * 
         * @param arn
         *        The ARN (Amazon Resource Name) of the cache subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);
    }

    static final class BuilderImpl implements Builder {
        private String cacheSubnetGroupName;

        private String cacheSubnetGroupDescription;

        private String vpcId;

        private List<Subnet> subnets = DefaultSdkAutoConstructList.getInstance();

        private String arn;

        private BuilderImpl() {
        }

        private BuilderImpl(CacheSubnetGroup model) {
            cacheSubnetGroupName(model.cacheSubnetGroupName);
            cacheSubnetGroupDescription(model.cacheSubnetGroupDescription);
            vpcId(model.vpcId);
            subnets(model.subnets);
            arn(model.arn);
        }

        public final String getCacheSubnetGroupName() {
            return cacheSubnetGroupName;
        }

        @Override
        public final Builder cacheSubnetGroupName(String cacheSubnetGroupName) {
            this.cacheSubnetGroupName = cacheSubnetGroupName;
            return this;
        }

        public final void setCacheSubnetGroupName(String cacheSubnetGroupName) {
            this.cacheSubnetGroupName = cacheSubnetGroupName;
        }

        public final String getCacheSubnetGroupDescription() {
            return cacheSubnetGroupDescription;
        }

        @Override
        public final Builder cacheSubnetGroupDescription(String cacheSubnetGroupDescription) {
            this.cacheSubnetGroupDescription = cacheSubnetGroupDescription;
            return this;
        }

        public final void setCacheSubnetGroupDescription(String cacheSubnetGroupDescription) {
            this.cacheSubnetGroupDescription = cacheSubnetGroupDescription;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Collection<Subnet.Builder> getSubnets() {
            return subnets != null ? subnets.stream().map(Subnet::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder subnets(Collection<Subnet> subnets) {
            this.subnets = SubnetListCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(Subnet... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(Consumer<Subnet.Builder>... subnets) {
            subnets(Stream.of(subnets).map(c -> Subnet.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSubnets(Collection<Subnet.BuilderImpl> subnets) {
            this.subnets = SubnetListCopier.copyFromBuilder(subnets);
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public CacheSubnetGroup build() {
            return new CacheSubnetGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
