/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Node group (shard) configuration options. Each node group (shard) configuration has the following: <code>Slots</code>, <code>PrimaryAvailabilityZone</code>, <code>ReplicaAvailabilityZones</code>, <code>ReplicaCount</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NodeGroupConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<NodeGroupConfiguration.Builder, NodeGroupConfiguration> {
    private static final SdkField<String> NODE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NodeGroupConfiguration::nodeGroupId)).setter(setter(Builder::nodeGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupId").build()).build();

    private static final SdkField<String> SLOTS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NodeGroupConfiguration::slots)).setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Slots").build()).build();

    private static final SdkField<Integer> REPLICA_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(NodeGroupConfiguration::replicaCount)).setter(setter(Builder::replicaCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaCount").build()).build();

    private static final SdkField<String> PRIMARY_AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NodeGroupConfiguration::primaryAvailabilityZone)).setter(setter(Builder::primaryAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrimaryAvailabilityZone").build())
            .build();

    private static final SdkField<List<String>> REPLICA_AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(NodeGroupConfiguration::replicaAvailabilityZones))
            .setter(setter(Builder::replicaAvailabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaAvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NODE_GROUP_ID_FIELD,
            SLOTS_FIELD, REPLICA_COUNT_FIELD, PRIMARY_AVAILABILITY_ZONE_FIELD, REPLICA_AVAILABILITY_ZONES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nodeGroupId;

    private final String slots;

    private final Integer replicaCount;

    private final String primaryAvailabilityZone;

    private final List<String> replicaAvailabilityZones;

    private NodeGroupConfiguration(BuilderImpl builder) {
        this.nodeGroupId = builder.nodeGroupId;
        this.slots = builder.slots;
        this.replicaCount = builder.replicaCount;
        this.primaryAvailabilityZone = builder.primaryAvailabilityZone;
        this.replicaAvailabilityZones = builder.replicaAvailabilityZones;
    }

    /**
     * <p>
     * Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these configuration
     * values apply to.
     * </p>
     * 
     * @return Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these
     *         configuration values apply to.
     */
    public String nodeGroupId() {
        return nodeGroupId;
    }

    /**
     * <p>
     * A string that specifies the keyspace for a particular node group. Keyspaces range from 0 to 16,383. The string is
     * in the format <code>startkey-endkey</code>.
     * </p>
     * <p>
     * Example: <code>"0-3999"</code>
     * </p>
     * 
     * @return A string that specifies the keyspace for a particular node group. Keyspaces range from 0 to 16,383. The
     *         string is in the format <code>startkey-endkey</code>.</p>
     *         <p>
     *         Example: <code>"0-3999"</code>
     */
    public String slots() {
        return slots;
    }

    /**
     * <p>
     * The number of read replica nodes in this node group (shard).
     * </p>
     * 
     * @return The number of read replica nodes in this node group (shard).
     */
    public Integer replicaCount() {
        return replicaCount;
    }

    /**
     * <p>
     * The Availability Zone where the primary node of this node group (shard) is launched.
     * </p>
     * 
     * @return The Availability Zone where the primary node of this node group (shard) is launched.
     */
    public String primaryAvailabilityZone() {
        return primaryAvailabilityZone;
    }

    /**
     * Returns true if the ReplicaAvailabilityZones property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasReplicaAvailabilityZones() {
        return replicaAvailabilityZones != null && !(replicaAvailabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Availability Zones to be used for the read replicas. The number of Availability Zones in this list must
     * match the value of <code>ReplicaCount</code> or <code>ReplicasPerNodeGroup</code> if not specified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasReplicaAvailabilityZones()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of Availability Zones to be used for the read replicas. The number of Availability Zones in this
     *         list must match the value of <code>ReplicaCount</code> or <code>ReplicasPerNodeGroup</code> if not
     *         specified.
     */
    public List<String> replicaAvailabilityZones() {
        return replicaAvailabilityZones;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nodeGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        hashCode = 31 * hashCode + Objects.hashCode(replicaCount());
        hashCode = 31 * hashCode + Objects.hashCode(primaryAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(replicaAvailabilityZones());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NodeGroupConfiguration)) {
            return false;
        }
        NodeGroupConfiguration other = (NodeGroupConfiguration) obj;
        return Objects.equals(nodeGroupId(), other.nodeGroupId()) && Objects.equals(slots(), other.slots())
                && Objects.equals(replicaCount(), other.replicaCount())
                && Objects.equals(primaryAvailabilityZone(), other.primaryAvailabilityZone())
                && Objects.equals(replicaAvailabilityZones(), other.replicaAvailabilityZones());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("NodeGroupConfiguration").add("NodeGroupId", nodeGroupId()).add("Slots", slots())
                .add("ReplicaCount", replicaCount()).add("PrimaryAvailabilityZone", primaryAvailabilityZone())
                .add("ReplicaAvailabilityZones", replicaAvailabilityZones()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NodeGroupId":
            return Optional.ofNullable(clazz.cast(nodeGroupId()));
        case "Slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "ReplicaCount":
            return Optional.ofNullable(clazz.cast(replicaCount()));
        case "PrimaryAvailabilityZone":
            return Optional.ofNullable(clazz.cast(primaryAvailabilityZone()));
        case "ReplicaAvailabilityZones":
            return Optional.ofNullable(clazz.cast(replicaAvailabilityZones()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NodeGroupConfiguration, T> g) {
        return obj -> g.apply((NodeGroupConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NodeGroupConfiguration> {
        /**
         * <p>
         * Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these
         * configuration values apply to.
         * </p>
         * 
         * @param nodeGroupId
         *        Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these
         *        configuration values apply to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupId(String nodeGroupId);

        /**
         * <p>
         * A string that specifies the keyspace for a particular node group. Keyspaces range from 0 to 16,383. The
         * string is in the format <code>startkey-endkey</code>.
         * </p>
         * <p>
         * Example: <code>"0-3999"</code>
         * </p>
         * 
         * @param slots
         *        A string that specifies the keyspace for a particular node group. Keyspaces range from 0 to 16,383.
         *        The string is in the format <code>startkey-endkey</code>.</p>
         *        <p>
         *        Example: <code>"0-3999"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(String slots);

        /**
         * <p>
         * The number of read replica nodes in this node group (shard).
         * </p>
         * 
         * @param replicaCount
         *        The number of read replica nodes in this node group (shard).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaCount(Integer replicaCount);

        /**
         * <p>
         * The Availability Zone where the primary node of this node group (shard) is launched.
         * </p>
         * 
         * @param primaryAvailabilityZone
         *        The Availability Zone where the primary node of this node group (shard) is launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder primaryAvailabilityZone(String primaryAvailabilityZone);

        /**
         * <p>
         * A list of Availability Zones to be used for the read replicas. The number of Availability Zones in this list
         * must match the value of <code>ReplicaCount</code> or <code>ReplicasPerNodeGroup</code> if not specified.
         * </p>
         * 
         * @param replicaAvailabilityZones
         *        A list of Availability Zones to be used for the read replicas. The number of Availability Zones in
         *        this list must match the value of <code>ReplicaCount</code> or <code>ReplicasPerNodeGroup</code> if
         *        not specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaAvailabilityZones(Collection<String> replicaAvailabilityZones);

        /**
         * <p>
         * A list of Availability Zones to be used for the read replicas. The number of Availability Zones in this list
         * must match the value of <code>ReplicaCount</code> or <code>ReplicasPerNodeGroup</code> if not specified.
         * </p>
         * 
         * @param replicaAvailabilityZones
         *        A list of Availability Zones to be used for the read replicas. The number of Availability Zones in
         *        this list must match the value of <code>ReplicaCount</code> or <code>ReplicasPerNodeGroup</code> if
         *        not specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaAvailabilityZones(String... replicaAvailabilityZones);
    }

    static final class BuilderImpl implements Builder {
        private String nodeGroupId;

        private String slots;

        private Integer replicaCount;

        private String primaryAvailabilityZone;

        private List<String> replicaAvailabilityZones = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NodeGroupConfiguration model) {
            nodeGroupId(model.nodeGroupId);
            slots(model.slots);
            replicaCount(model.replicaCount);
            primaryAvailabilityZone(model.primaryAvailabilityZone);
            replicaAvailabilityZones(model.replicaAvailabilityZones);
        }

        public final String getNodeGroupId() {
            return nodeGroupId;
        }

        @Override
        public final Builder nodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
            return this;
        }

        public final void setNodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
        }

        public final String getSlots() {
            return slots;
        }

        @Override
        public final Builder slots(String slots) {
            this.slots = slots;
            return this;
        }

        public final void setSlots(String slots) {
            this.slots = slots;
        }

        public final Integer getReplicaCount() {
            return replicaCount;
        }

        @Override
        public final Builder replicaCount(Integer replicaCount) {
            this.replicaCount = replicaCount;
            return this;
        }

        public final void setReplicaCount(Integer replicaCount) {
            this.replicaCount = replicaCount;
        }

        public final String getPrimaryAvailabilityZone() {
            return primaryAvailabilityZone;
        }

        @Override
        public final Builder primaryAvailabilityZone(String primaryAvailabilityZone) {
            this.primaryAvailabilityZone = primaryAvailabilityZone;
            return this;
        }

        public final void setPrimaryAvailabilityZone(String primaryAvailabilityZone) {
            this.primaryAvailabilityZone = primaryAvailabilityZone;
        }

        public final Collection<String> getReplicaAvailabilityZones() {
            return replicaAvailabilityZones;
        }

        @Override
        public final Builder replicaAvailabilityZones(Collection<String> replicaAvailabilityZones) {
            this.replicaAvailabilityZones = AvailabilityZonesListCopier.copy(replicaAvailabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replicaAvailabilityZones(String... replicaAvailabilityZones) {
            replicaAvailabilityZones(Arrays.asList(replicaAvailabilityZones));
            return this;
        }

        public final void setReplicaAvailabilityZones(Collection<String> replicaAvailabilityZones) {
            this.replicaAvailabilityZones = AvailabilityZonesListCopier.copy(replicaAvailabilityZones);
        }

        @Override
        public NodeGroupConfiguration build() {
            return new NodeGroupConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
