/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an individual cache node within a cluster. Each cache node runs its own instance of the cluster's
 * protocol-compliant caching software - either Memcached or Redis.
 * </p>
 * <p>
 * The following node types are supported by ElastiCache. Generally speaking, the current generation types provide more
 * memory and computational power at lower cost when compared to their equivalent previous generation counterparts.
 * </p>
 * <ul>
 * <li>
 * <p>
 * General purpose:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Current generation:
 * </p>
 * <p>
 * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>, <code>cache.m5.2xlarge</code>,
 * <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>, <code>cache.m5.24xlarge</code>
 * </p>
 * <p>
 * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>, <code>cache.m4.2xlarge</code>,
 * <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
 * </p>
 * <p>
 * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * Previous generation: (not recommended)
 * </p>
 * <p>
 * <b>T1 node types:</b> <code>cache.t1.micro</code>
 * </p>
 * <p>
 * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
 * <code>cache.m1.xlarge</code>
 * </p>
 * <p>
 * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>,
 * <code>cache.m3.2xlarge</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * Compute optimized:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Previous generation: (not recommended)
 * </p>
 * <p>
 * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * Memory optimized:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Current generation:
 * </p>
 * <p>
 * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>, <code>cache.r5.2xlarge</code>,
 * <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>, <code>cache.r5.24xlarge</code>
 * </p>
 * <p>
 * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>, <code>cache.r4.2xlarge</code>,
 * <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>, <code>cache.r4.16xlarge</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * Previous generation: (not recommended)
 * </p>
 * <p>
 * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>, <code>cache.m2.4xlarge</code>
 * </p>
 * <p>
 * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>, <code>cache.r3.2xlarge</code>,
 * <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * <b>Additional node type info</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * All current generation instance types are created in Amazon VPC by default.
 * </p>
 * </li>
 * <li>
 * <p>
 * Redis append-only files (AOF) are not supported for T1 or T2 instances.
 * </p>
 * </li>
 * <li>
 * <p>
 * Redis Multi-AZ with automatic failover is not supported on T1 instances.
 * </p>
 * </li>
 * <li>
 * <p>
 * Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Redis version
 * 2.8.22 and later.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CacheNode implements SdkPojo, Serializable, ToCopyableBuilder<CacheNode.Builder, CacheNode> {
    private static final SdkField<String> CACHE_NODE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheNode::cacheNodeId)).setter(setter(Builder::cacheNodeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeId").build()).build();

    private static final SdkField<String> CACHE_NODE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheNode::cacheNodeStatus)).setter(setter(Builder::cacheNodeStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeStatus").build()).build();

    private static final SdkField<Instant> CACHE_NODE_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(CacheNode::cacheNodeCreateTime)).setter(setter(Builder::cacheNodeCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeCreateTime").build())
            .build();

    private static final SdkField<Endpoint> ENDPOINT_FIELD = SdkField.<Endpoint> builder(MarshallingType.SDK_POJO)
            .getter(getter(CacheNode::endpoint)).setter(setter(Builder::endpoint)).constructor(Endpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<String> PARAMETER_GROUP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheNode::parameterGroupStatus)).setter(setter(Builder::parameterGroupStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterGroupStatus").build())
            .build();

    private static final SdkField<String> SOURCE_CACHE_NODE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheNode::sourceCacheNodeId)).setter(setter(Builder::sourceCacheNodeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceCacheNodeId").build()).build();

    private static final SdkField<String> CUSTOMER_AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CacheNode::customerAvailabilityZone)).setter(setter(Builder::customerAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerAvailabilityZone").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CACHE_NODE_ID_FIELD,
            CACHE_NODE_STATUS_FIELD, CACHE_NODE_CREATE_TIME_FIELD, ENDPOINT_FIELD, PARAMETER_GROUP_STATUS_FIELD,
            SOURCE_CACHE_NODE_ID_FIELD, CUSTOMER_AVAILABILITY_ZONE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String cacheNodeId;

    private final String cacheNodeStatus;

    private final Instant cacheNodeCreateTime;

    private final Endpoint endpoint;

    private final String parameterGroupStatus;

    private final String sourceCacheNodeId;

    private final String customerAvailabilityZone;

    private CacheNode(BuilderImpl builder) {
        this.cacheNodeId = builder.cacheNodeId;
        this.cacheNodeStatus = builder.cacheNodeStatus;
        this.cacheNodeCreateTime = builder.cacheNodeCreateTime;
        this.endpoint = builder.endpoint;
        this.parameterGroupStatus = builder.parameterGroupStatus;
        this.sourceCacheNodeId = builder.sourceCacheNodeId;
        this.customerAvailabilityZone = builder.customerAvailabilityZone;
    }

    /**
     * <p>
     * The cache node identifier. A node ID is a numeric identifier (0001, 0002, etc.). The combination of cluster ID
     * and node ID uniquely identifies every cache node used in a customer's AWS account.
     * </p>
     * 
     * @return The cache node identifier. A node ID is a numeric identifier (0001, 0002, etc.). The combination of
     *         cluster ID and node ID uniquely identifies every cache node used in a customer's AWS account.
     */
    public String cacheNodeId() {
        return cacheNodeId;
    }

    /**
     * <p>
     * The current state of this cache node.
     * </p>
     * 
     * @return The current state of this cache node.
     */
    public String cacheNodeStatus() {
        return cacheNodeStatus;
    }

    /**
     * <p>
     * The date and time when the cache node was created.
     * </p>
     * 
     * @return The date and time when the cache node was created.
     */
    public Instant cacheNodeCreateTime() {
        return cacheNodeCreateTime;
    }

    /**
     * <p>
     * The hostname for connecting to this cache node.
     * </p>
     * 
     * @return The hostname for connecting to this cache node.
     */
    public Endpoint endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The status of the parameter group applied to this cache node.
     * </p>
     * 
     * @return The status of the parameter group applied to this cache node.
     */
    public String parameterGroupStatus() {
        return parameterGroupStatus;
    }

    /**
     * <p>
     * The ID of the primary node to which this read replica node is synchronized. If this field is empty, this node is
     * not associated with a primary cluster.
     * </p>
     * 
     * @return The ID of the primary node to which this read replica node is synchronized. If this field is empty, this
     *         node is not associated with a primary cluster.
     */
    public String sourceCacheNodeId() {
        return sourceCacheNodeId;
    }

    /**
     * <p>
     * The Availability Zone where this node was created and now resides.
     * </p>
     * 
     * @return The Availability Zone where this node was created and now resides.
     */
    public String customerAvailabilityZone() {
        return customerAvailabilityZone;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeId());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeStatus());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(parameterGroupStatus());
        hashCode = 31 * hashCode + Objects.hashCode(sourceCacheNodeId());
        hashCode = 31 * hashCode + Objects.hashCode(customerAvailabilityZone());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CacheNode)) {
            return false;
        }
        CacheNode other = (CacheNode) obj;
        return Objects.equals(cacheNodeId(), other.cacheNodeId()) && Objects.equals(cacheNodeStatus(), other.cacheNodeStatus())
                && Objects.equals(cacheNodeCreateTime(), other.cacheNodeCreateTime())
                && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(parameterGroupStatus(), other.parameterGroupStatus())
                && Objects.equals(sourceCacheNodeId(), other.sourceCacheNodeId())
                && Objects.equals(customerAvailabilityZone(), other.customerAvailabilityZone());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CacheNode").add("CacheNodeId", cacheNodeId()).add("CacheNodeStatus", cacheNodeStatus())
                .add("CacheNodeCreateTime", cacheNodeCreateTime()).add("Endpoint", endpoint())
                .add("ParameterGroupStatus", parameterGroupStatus()).add("SourceCacheNodeId", sourceCacheNodeId())
                .add("CustomerAvailabilityZone", customerAvailabilityZone()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CacheNodeId":
            return Optional.ofNullable(clazz.cast(cacheNodeId()));
        case "CacheNodeStatus":
            return Optional.ofNullable(clazz.cast(cacheNodeStatus()));
        case "CacheNodeCreateTime":
            return Optional.ofNullable(clazz.cast(cacheNodeCreateTime()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "ParameterGroupStatus":
            return Optional.ofNullable(clazz.cast(parameterGroupStatus()));
        case "SourceCacheNodeId":
            return Optional.ofNullable(clazz.cast(sourceCacheNodeId()));
        case "CustomerAvailabilityZone":
            return Optional.ofNullable(clazz.cast(customerAvailabilityZone()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CacheNode, T> g) {
        return obj -> g.apply((CacheNode) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CacheNode> {
        /**
         * <p>
         * The cache node identifier. A node ID is a numeric identifier (0001, 0002, etc.). The combination of cluster
         * ID and node ID uniquely identifies every cache node used in a customer's AWS account.
         * </p>
         * 
         * @param cacheNodeId
         *        The cache node identifier. A node ID is a numeric identifier (0001, 0002, etc.). The combination of
         *        cluster ID and node ID uniquely identifies every cache node used in a customer's AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeId(String cacheNodeId);

        /**
         * <p>
         * The current state of this cache node.
         * </p>
         * 
         * @param cacheNodeStatus
         *        The current state of this cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeStatus(String cacheNodeStatus);

        /**
         * <p>
         * The date and time when the cache node was created.
         * </p>
         * 
         * @param cacheNodeCreateTime
         *        The date and time when the cache node was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeCreateTime(Instant cacheNodeCreateTime);

        /**
         * <p>
         * The hostname for connecting to this cache node.
         * </p>
         * 
         * @param endpoint
         *        The hostname for connecting to this cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(Endpoint endpoint);

        /**
         * <p>
         * The hostname for connecting to this cache node.
         * </p>
         * This is a convenience that creates an instance of the {@link Endpoint.Builder} avoiding the need to create
         * one manually via {@link Endpoint#builder()}.
         *
         * When the {@link Consumer} completes, {@link Endpoint.Builder#build()} is called immediately and its result is
         * passed to {@link #endpoint(Endpoint)}.
         * 
         * @param endpoint
         *        a consumer that will call methods on {@link Endpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpoint(Endpoint)
         */
        default Builder endpoint(Consumer<Endpoint.Builder> endpoint) {
            return endpoint(Endpoint.builder().applyMutation(endpoint).build());
        }

        /**
         * <p>
         * The status of the parameter group applied to this cache node.
         * </p>
         * 
         * @param parameterGroupStatus
         *        The status of the parameter group applied to this cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterGroupStatus(String parameterGroupStatus);

        /**
         * <p>
         * The ID of the primary node to which this read replica node is synchronized. If this field is empty, this node
         * is not associated with a primary cluster.
         * </p>
         * 
         * @param sourceCacheNodeId
         *        The ID of the primary node to which this read replica node is synchronized. If this field is empty,
         *        this node is not associated with a primary cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceCacheNodeId(String sourceCacheNodeId);

        /**
         * <p>
         * The Availability Zone where this node was created and now resides.
         * </p>
         * 
         * @param customerAvailabilityZone
         *        The Availability Zone where this node was created and now resides.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerAvailabilityZone(String customerAvailabilityZone);
    }

    static final class BuilderImpl implements Builder {
        private String cacheNodeId;

        private String cacheNodeStatus;

        private Instant cacheNodeCreateTime;

        private Endpoint endpoint;

        private String parameterGroupStatus;

        private String sourceCacheNodeId;

        private String customerAvailabilityZone;

        private BuilderImpl() {
        }

        private BuilderImpl(CacheNode model) {
            cacheNodeId(model.cacheNodeId);
            cacheNodeStatus(model.cacheNodeStatus);
            cacheNodeCreateTime(model.cacheNodeCreateTime);
            endpoint(model.endpoint);
            parameterGroupStatus(model.parameterGroupStatus);
            sourceCacheNodeId(model.sourceCacheNodeId);
            customerAvailabilityZone(model.customerAvailabilityZone);
        }

        public final String getCacheNodeId() {
            return cacheNodeId;
        }

        @Override
        public final Builder cacheNodeId(String cacheNodeId) {
            this.cacheNodeId = cacheNodeId;
            return this;
        }

        public final void setCacheNodeId(String cacheNodeId) {
            this.cacheNodeId = cacheNodeId;
        }

        public final String getCacheNodeStatus() {
            return cacheNodeStatus;
        }

        @Override
        public final Builder cacheNodeStatus(String cacheNodeStatus) {
            this.cacheNodeStatus = cacheNodeStatus;
            return this;
        }

        public final void setCacheNodeStatus(String cacheNodeStatus) {
            this.cacheNodeStatus = cacheNodeStatus;
        }

        public final Instant getCacheNodeCreateTime() {
            return cacheNodeCreateTime;
        }

        @Override
        public final Builder cacheNodeCreateTime(Instant cacheNodeCreateTime) {
            this.cacheNodeCreateTime = cacheNodeCreateTime;
            return this;
        }

        public final void setCacheNodeCreateTime(Instant cacheNodeCreateTime) {
            this.cacheNodeCreateTime = cacheNodeCreateTime;
        }

        public final Endpoint.Builder getEndpoint() {
            return endpoint != null ? endpoint.toBuilder() : null;
        }

        @Override
        public final Builder endpoint(Endpoint endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final void setEndpoint(Endpoint.BuilderImpl endpoint) {
            this.endpoint = endpoint != null ? endpoint.build() : null;
        }

        public final String getParameterGroupStatus() {
            return parameterGroupStatus;
        }

        @Override
        public final Builder parameterGroupStatus(String parameterGroupStatus) {
            this.parameterGroupStatus = parameterGroupStatus;
            return this;
        }

        public final void setParameterGroupStatus(String parameterGroupStatus) {
            this.parameterGroupStatus = parameterGroupStatus;
        }

        public final String getSourceCacheNodeId() {
            return sourceCacheNodeId;
        }

        @Override
        public final Builder sourceCacheNodeId(String sourceCacheNodeId) {
            this.sourceCacheNodeId = sourceCacheNodeId;
            return this;
        }

        public final void setSourceCacheNodeId(String sourceCacheNodeId) {
            this.sourceCacheNodeId = sourceCacheNodeId;
        }

        public final String getCustomerAvailabilityZone() {
            return customerAvailabilityZone;
        }

        @Override
        public final Builder customerAvailabilityZone(String customerAvailabilityZone) {
            this.customerAvailabilityZone = customerAvailabilityZone;
            return this;
        }

        public final void setCustomerAvailabilityZone(String customerAvailabilityZone) {
            this.customerAvailabilityZone = customerAvailabilityZone;
        }

        @Override
        public CacheNode build() {
            return new CacheNode(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
