/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.eks.EksAsyncClient;
import software.amazon.awssdk.services.eks.model.DescribeAddonRequest;
import software.amazon.awssdk.services.eks.model.DescribeAddonResponse;
import software.amazon.awssdk.services.eks.model.DescribeClusterRequest;
import software.amazon.awssdk.services.eks.model.DescribeClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupResponse;
import software.amazon.awssdk.services.eks.model.EksRequest;
import software.amazon.awssdk.services.eks.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultEksAsyncWaiter implements EksAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final EksAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeClusterResponse> clusterActiveWaiter;

    private final AsyncWaiter<DescribeClusterResponse> clusterDeletedWaiter;

    private final AsyncWaiter<DescribeNodegroupResponse> nodegroupActiveWaiter;

    private final AsyncWaiter<DescribeNodegroupResponse> nodegroupDeletedWaiter;

    private final AsyncWaiter<DescribeAddonResponse> addonActiveWaiter;

    private final AsyncWaiter<DescribeAddonResponse> addonDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultEksAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = EksAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.clusterActiveWaiter = AsyncWaiter.builder(DescribeClusterResponse.class).acceptors(clusterActiveWaiterAcceptors())
                .overrideConfiguration(clusterActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.clusterDeletedWaiter = AsyncWaiter.builder(DescribeClusterResponse.class).acceptors(clusterDeletedWaiterAcceptors())
                .overrideConfiguration(clusterDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.nodegroupActiveWaiter = AsyncWaiter.builder(DescribeNodegroupResponse.class)
                .acceptors(nodegroupActiveWaiterAcceptors())
                .overrideConfiguration(nodegroupActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.nodegroupDeletedWaiter = AsyncWaiter.builder(DescribeNodegroupResponse.class)
                .acceptors(nodegroupDeletedWaiterAcceptors())
                .overrideConfiguration(nodegroupDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.addonActiveWaiter = AsyncWaiter.builder(DescribeAddonResponse.class).acceptors(addonActiveWaiterAcceptors())
                .overrideConfiguration(addonActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.addonDeletedWaiter = AsyncWaiter.builder(DescribeAddonResponse.class).acceptors(addonDeletedWaiterAcceptors())
                .overrideConfiguration(addonDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonActive(DescribeAddonRequest describeAddonRequest) {
        return addonActiveWaiter.runAsync(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonActive(
            DescribeAddonRequest describeAddonRequest, WaiterOverrideConfiguration overrideConfig) {
        return addonActiveWaiter.runAsync(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)),
                addonActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonDeleted(
            DescribeAddonRequest describeAddonRequest) {
        return addonDeletedWaiter.runAsync(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonDeleted(
            DescribeAddonRequest describeAddonRequest, WaiterOverrideConfiguration overrideConfig) {
        return addonDeletedWaiter.runAsync(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)),
                addonDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterActive(
            DescribeClusterRequest describeClusterRequest) {
        return clusterActiveWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterActive(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        return clusterActiveWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest) {
        return clusterDeletedWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        return clusterDeletedWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupActive(
            DescribeNodegroupRequest describeNodegroupRequest) {
        return nodegroupActiveWaiter.runAsync(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupActive(
            DescribeNodegroupRequest describeNodegroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return nodegroupActiveWaiter.runAsync(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)),
                nodegroupActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupDeleted(
            DescribeNodegroupRequest describeNodegroupRequest) {
        return nodegroupDeletedWaiter.runAsync(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupDeleted(
            DescribeNodegroupRequest describeNodegroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return nodegroupDeletedWaiter.runAsync(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)),
                nodegroupDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("cluster").field("status").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("cluster").field("status").value(), "FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("cluster").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("cluster").field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("cluster").field("status").value(), "CREATING");
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodegroupResponse>> nodegroupActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodegroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("nodegroup").field("status").value(), "CREATE_FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("nodegroup").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodegroupResponse>> nodegroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodegroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("nodegroup").field("status").value(), "DELETE_FAILED");
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAddonResponse>> addonActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAddonResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("addon").field("status").value(), "CREATE_FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("addon").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAddonResponse>> addonDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAddonResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("addon").field("status").value(), "DELETE_FAILED");
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration clusterActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodegroupActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(80);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodegroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration addonActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration addonDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static EksAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends EksRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements EksAsyncWaiter.Builder {
        private EksAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public EksAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public EksAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public EksAsyncWaiter.Builder client(EksAsyncClient client) {
            this.client = client;
            return this;
        }

        public EksAsyncWaiter build() {
            return new DefaultEksAsyncWaiter(this);
        }
    }
}
