/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the auto-scaling configuration for an Express service. This determines how the service automatically adjusts
 * the number of running tasks based on demand metrics such as CPU utilization, memory utilization, or request count per
 * target.
 * </p>
 * <p>
 * Auto-scaling helps ensure your application can handle varying levels of traffic while optimizing costs by scaling
 * down during low-demand periods. You can specify the minimum and maximum number of tasks, the scaling metric, and the
 * target value for that metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExpressGatewayScalingTarget implements SdkPojo, Serializable,
        ToCopyableBuilder<ExpressGatewayScalingTarget.Builder, ExpressGatewayScalingTarget> {
    private static final SdkField<Integer> MIN_TASK_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("minTaskCount").getter(getter(ExpressGatewayScalingTarget::minTaskCount))
            .setter(setter(Builder::minTaskCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minTaskCount").build()).build();

    private static final SdkField<Integer> MAX_TASK_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxTaskCount").getter(getter(ExpressGatewayScalingTarget::maxTaskCount))
            .setter(setter(Builder::maxTaskCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxTaskCount").build()).build();

    private static final SdkField<String> AUTO_SCALING_METRIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("autoScalingMetric").getter(getter(ExpressGatewayScalingTarget::autoScalingMetricAsString))
            .setter(setter(Builder::autoScalingMetric))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingMetric").build()).build();

    private static final SdkField<Integer> AUTO_SCALING_TARGET_VALUE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("autoScalingTargetValue").getter(getter(ExpressGatewayScalingTarget::autoScalingTargetValue))
            .setter(setter(Builder::autoScalingTargetValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingTargetValue").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MIN_TASK_COUNT_FIELD,
            MAX_TASK_COUNT_FIELD, AUTO_SCALING_METRIC_FIELD, AUTO_SCALING_TARGET_VALUE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer minTaskCount;

    private final Integer maxTaskCount;

    private final String autoScalingMetric;

    private final Integer autoScalingTargetValue;

    private ExpressGatewayScalingTarget(BuilderImpl builder) {
        this.minTaskCount = builder.minTaskCount;
        this.maxTaskCount = builder.maxTaskCount;
        this.autoScalingMetric = builder.autoScalingMetric;
        this.autoScalingTargetValue = builder.autoScalingTargetValue;
    }

    /**
     * <p>
     * The minimum number of tasks to run in the Express service.
     * </p>
     * 
     * @return The minimum number of tasks to run in the Express service.
     */
    public final Integer minTaskCount() {
        return minTaskCount;
    }

    /**
     * <p>
     * The maximum number of tasks to run in the Express service.
     * </p>
     * 
     * @return The maximum number of tasks to run in the Express service.
     */
    public final Integer maxTaskCount() {
        return maxTaskCount;
    }

    /**
     * <p>
     * The metric used for auto-scaling decisions. The default metric used for an Express service is
     * <code>CPUUtilization</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoScalingMetric}
     * will return {@link ExpressGatewayServiceScalingMetric#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #autoScalingMetricAsString}.
     * </p>
     * 
     * @return The metric used for auto-scaling decisions. The default metric used for an Express service is
     *         <code>CPUUtilization</code>.
     * @see ExpressGatewayServiceScalingMetric
     */
    public final ExpressGatewayServiceScalingMetric autoScalingMetric() {
        return ExpressGatewayServiceScalingMetric.fromValue(autoScalingMetric);
    }

    /**
     * <p>
     * The metric used for auto-scaling decisions. The default metric used for an Express service is
     * <code>CPUUtilization</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoScalingMetric}
     * will return {@link ExpressGatewayServiceScalingMetric#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #autoScalingMetricAsString}.
     * </p>
     * 
     * @return The metric used for auto-scaling decisions. The default metric used for an Express service is
     *         <code>CPUUtilization</code>.
     * @see ExpressGatewayServiceScalingMetric
     */
    public final String autoScalingMetricAsString() {
        return autoScalingMetric;
    }

    /**
     * <p>
     * The target value for the auto-scaling metric. The default value for an Express service is 60.
     * </p>
     * 
     * @return The target value for the auto-scaling metric. The default value for an Express service is 60.
     */
    public final Integer autoScalingTargetValue() {
        return autoScalingTargetValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minTaskCount());
        hashCode = 31 * hashCode + Objects.hashCode(maxTaskCount());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingMetricAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingTargetValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExpressGatewayScalingTarget)) {
            return false;
        }
        ExpressGatewayScalingTarget other = (ExpressGatewayScalingTarget) obj;
        return Objects.equals(minTaskCount(), other.minTaskCount()) && Objects.equals(maxTaskCount(), other.maxTaskCount())
                && Objects.equals(autoScalingMetricAsString(), other.autoScalingMetricAsString())
                && Objects.equals(autoScalingTargetValue(), other.autoScalingTargetValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExpressGatewayScalingTarget").add("MinTaskCount", minTaskCount())
                .add("MaxTaskCount", maxTaskCount()).add("AutoScalingMetric", autoScalingMetricAsString())
                .add("AutoScalingTargetValue", autoScalingTargetValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "minTaskCount":
            return Optional.ofNullable(clazz.cast(minTaskCount()));
        case "maxTaskCount":
            return Optional.ofNullable(clazz.cast(maxTaskCount()));
        case "autoScalingMetric":
            return Optional.ofNullable(clazz.cast(autoScalingMetricAsString()));
        case "autoScalingTargetValue":
            return Optional.ofNullable(clazz.cast(autoScalingTargetValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("minTaskCount", MIN_TASK_COUNT_FIELD);
        map.put("maxTaskCount", MAX_TASK_COUNT_FIELD);
        map.put("autoScalingMetric", AUTO_SCALING_METRIC_FIELD);
        map.put("autoScalingTargetValue", AUTO_SCALING_TARGET_VALUE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ExpressGatewayScalingTarget, T> g) {
        return obj -> g.apply((ExpressGatewayScalingTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExpressGatewayScalingTarget> {
        /**
         * <p>
         * The minimum number of tasks to run in the Express service.
         * </p>
         * 
         * @param minTaskCount
         *        The minimum number of tasks to run in the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minTaskCount(Integer minTaskCount);

        /**
         * <p>
         * The maximum number of tasks to run in the Express service.
         * </p>
         * 
         * @param maxTaskCount
         *        The maximum number of tasks to run in the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTaskCount(Integer maxTaskCount);

        /**
         * <p>
         * The metric used for auto-scaling decisions. The default metric used for an Express service is
         * <code>CPUUtilization</code>.
         * </p>
         * 
         * @param autoScalingMetric
         *        The metric used for auto-scaling decisions. The default metric used for an Express service is
         *        <code>CPUUtilization</code>.
         * @see ExpressGatewayServiceScalingMetric
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpressGatewayServiceScalingMetric
         */
        Builder autoScalingMetric(String autoScalingMetric);

        /**
         * <p>
         * The metric used for auto-scaling decisions. The default metric used for an Express service is
         * <code>CPUUtilization</code>.
         * </p>
         * 
         * @param autoScalingMetric
         *        The metric used for auto-scaling decisions. The default metric used for an Express service is
         *        <code>CPUUtilization</code>.
         * @see ExpressGatewayServiceScalingMetric
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpressGatewayServiceScalingMetric
         */
        Builder autoScalingMetric(ExpressGatewayServiceScalingMetric autoScalingMetric);

        /**
         * <p>
         * The target value for the auto-scaling metric. The default value for an Express service is 60.
         * </p>
         * 
         * @param autoScalingTargetValue
         *        The target value for the auto-scaling metric. The default value for an Express service is 60.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingTargetValue(Integer autoScalingTargetValue);
    }

    static final class BuilderImpl implements Builder {
        private Integer minTaskCount;

        private Integer maxTaskCount;

        private String autoScalingMetric;

        private Integer autoScalingTargetValue;

        private BuilderImpl() {
        }

        private BuilderImpl(ExpressGatewayScalingTarget model) {
            minTaskCount(model.minTaskCount);
            maxTaskCount(model.maxTaskCount);
            autoScalingMetric(model.autoScalingMetric);
            autoScalingTargetValue(model.autoScalingTargetValue);
        }

        public final Integer getMinTaskCount() {
            return minTaskCount;
        }

        public final void setMinTaskCount(Integer minTaskCount) {
            this.minTaskCount = minTaskCount;
        }

        @Override
        public final Builder minTaskCount(Integer minTaskCount) {
            this.minTaskCount = minTaskCount;
            return this;
        }

        public final Integer getMaxTaskCount() {
            return maxTaskCount;
        }

        public final void setMaxTaskCount(Integer maxTaskCount) {
            this.maxTaskCount = maxTaskCount;
        }

        @Override
        public final Builder maxTaskCount(Integer maxTaskCount) {
            this.maxTaskCount = maxTaskCount;
            return this;
        }

        public final String getAutoScalingMetric() {
            return autoScalingMetric;
        }

        public final void setAutoScalingMetric(String autoScalingMetric) {
            this.autoScalingMetric = autoScalingMetric;
        }

        @Override
        public final Builder autoScalingMetric(String autoScalingMetric) {
            this.autoScalingMetric = autoScalingMetric;
            return this;
        }

        @Override
        public final Builder autoScalingMetric(ExpressGatewayServiceScalingMetric autoScalingMetric) {
            this.autoScalingMetric(autoScalingMetric == null ? null : autoScalingMetric.toString());
            return this;
        }

        public final Integer getAutoScalingTargetValue() {
            return autoScalingTargetValue;
        }

        public final void setAutoScalingTargetValue(Integer autoScalingTargetValue) {
            this.autoScalingTargetValue = autoScalingTargetValue;
        }

        @Override
        public final Builder autoScalingTargetValue(Integer autoScalingTargetValue) {
            this.autoScalingTargetValue = autoScalingTargetValue;
            return this;
        }

        @Override
        public ExpressGatewayScalingTarget build() {
            return new ExpressGatewayScalingTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
