/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an Express service, which provides a simplified way to deploy containerized web applications on Amazon ECS
 * with managed Amazon Web Services infrastructure. An Express service automatically provisions and manages Application
 * Load Balancers, target groups, security groups, and auto-scaling policies.
 * </p>
 * <p>
 * Express services use a service revision architecture where each service can have multiple active configurations,
 * enabling blue-green deployments and gradual rollouts. The service maintains a list of active configurations and
 * manages the lifecycle of the underlying Amazon Web Services resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ECSExpressGatewayService implements SdkPojo, Serializable,
        ToCopyableBuilder<ECSExpressGatewayService.Builder, ECSExpressGatewayService> {
    private static final SdkField<String> CLUSTER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("cluster")
            .getter(getter(ECSExpressGatewayService::cluster)).setter(setter(Builder::cluster))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cluster").build()).build();

    private static final SdkField<String> SERVICE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceName").getter(getter(ECSExpressGatewayService::serviceName)).setter(setter(Builder::serviceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceName").build()).build();

    private static final SdkField<String> SERVICE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceArn").getter(getter(ECSExpressGatewayService::serviceArn)).setter(setter(Builder::serviceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceArn").build()).build();

    private static final SdkField<String> INFRASTRUCTURE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("infrastructureRoleArn").getter(getter(ECSExpressGatewayService::infrastructureRoleArn))
            .setter(setter(Builder::infrastructureRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("infrastructureRoleArn").build())
            .build();

    private static final SdkField<ExpressGatewayServiceStatus> STATUS_FIELD = SdkField
            .<ExpressGatewayServiceStatus> builder(MarshallingType.SDK_POJO).memberName("status")
            .getter(getter(ECSExpressGatewayService::status)).setter(setter(Builder::status))
            .constructor(ExpressGatewayServiceStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> CURRENT_DEPLOYMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentDeployment").getter(getter(ECSExpressGatewayService::currentDeployment))
            .setter(setter(Builder::currentDeployment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentDeployment").build()).build();

    private static final SdkField<List<ExpressGatewayServiceConfiguration>> ACTIVE_CONFIGURATIONS_FIELD = SdkField
            .<List<ExpressGatewayServiceConfiguration>> builder(MarshallingType.LIST)
            .memberName("activeConfigurations")
            .getter(getter(ECSExpressGatewayService::activeConfigurations))
            .setter(setter(Builder::activeConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("activeConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExpressGatewayServiceConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExpressGatewayServiceConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(ECSExpressGatewayService::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(ECSExpressGatewayService::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(ECSExpressGatewayService::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_FIELD,
            SERVICE_NAME_FIELD, SERVICE_ARN_FIELD, INFRASTRUCTURE_ROLE_ARN_FIELD, STATUS_FIELD, CURRENT_DEPLOYMENT_FIELD,
            ACTIVE_CONFIGURATIONS_FIELD, TAGS_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String cluster;

    private final String serviceName;

    private final String serviceArn;

    private final String infrastructureRoleArn;

    private final ExpressGatewayServiceStatus status;

    private final String currentDeployment;

    private final List<ExpressGatewayServiceConfiguration> activeConfigurations;

    private final List<Tag> tags;

    private final Instant createdAt;

    private final Instant updatedAt;

    private ECSExpressGatewayService(BuilderImpl builder) {
        this.cluster = builder.cluster;
        this.serviceName = builder.serviceName;
        this.serviceArn = builder.serviceArn;
        this.infrastructureRoleArn = builder.infrastructureRoleArn;
        this.status = builder.status;
        this.currentDeployment = builder.currentDeployment;
        this.activeConfigurations = builder.activeConfigurations;
        this.tags = builder.tags;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The short name or full ARN of the cluster that hosts the Express service.
     * </p>
     * 
     * @return The short name or full ARN of the cluster that hosts the Express service.
     */
    public final String cluster() {
        return cluster;
    }

    /**
     * <p>
     * The name of the Express service.
     * </p>
     * 
     * @return The name of the Express service.
     */
    public final String serviceName() {
        return serviceName;
    }

    /**
     * <p>
     * The ARN that identifies the Express service.
     * </p>
     * 
     * @return The ARN that identifies the Express service.
     */
    public final String serviceArn() {
        return serviceArn;
    }

    /**
     * <p>
     * The ARN of the infrastructure role that manages Amazon Web Services resources for the Express service.
     * </p>
     * 
     * @return The ARN of the infrastructure role that manages Amazon Web Services resources for the Express service.
     */
    public final String infrastructureRoleArn() {
        return infrastructureRoleArn;
    }

    /**
     * <p>
     * The current status of the Express service.
     * </p>
     * 
     * @return The current status of the Express service.
     */
    public final ExpressGatewayServiceStatus status() {
        return status;
    }

    /**
     * <p>
     * The current deployment configuration for the Express service.
     * </p>
     * 
     * @return The current deployment configuration for the Express service.
     */
    public final String currentDeployment() {
        return currentDeployment;
    }

    /**
     * For responses, this returns true if the service returned a value for the ActiveConfigurations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasActiveConfigurations() {
        return activeConfigurations != null && !(activeConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of active service configurations for the Express service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActiveConfigurations} method.
     * </p>
     * 
     * @return The list of active service configurations for the Express service.
     */
    public final List<ExpressGatewayServiceConfiguration> activeConfigurations() {
        return activeConfigurations;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metadata applied to the Express service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The metadata applied to the Express service.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The Unix timestamp for when the Express service was created.
     * </p>
     * 
     * @return The Unix timestamp for when the Express service was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Unix timestamp for when the Express service was last updated.
     * </p>
     * 
     * @return The Unix timestamp for when the Express service was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cluster());
        hashCode = 31 * hashCode + Objects.hashCode(serviceName());
        hashCode = 31 * hashCode + Objects.hashCode(serviceArn());
        hashCode = 31 * hashCode + Objects.hashCode(infrastructureRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(currentDeployment());
        hashCode = 31 * hashCode + Objects.hashCode(hasActiveConfigurations() ? activeConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ECSExpressGatewayService)) {
            return false;
        }
        ECSExpressGatewayService other = (ECSExpressGatewayService) obj;
        return Objects.equals(cluster(), other.cluster()) && Objects.equals(serviceName(), other.serviceName())
                && Objects.equals(serviceArn(), other.serviceArn())
                && Objects.equals(infrastructureRoleArn(), other.infrastructureRoleArn())
                && Objects.equals(status(), other.status()) && Objects.equals(currentDeployment(), other.currentDeployment())
                && hasActiveConfigurations() == other.hasActiveConfigurations()
                && Objects.equals(activeConfigurations(), other.activeConfigurations()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ECSExpressGatewayService").add("Cluster", cluster()).add("ServiceName", serviceName())
                .add("ServiceArn", serviceArn()).add("InfrastructureRoleArn", infrastructureRoleArn()).add("Status", status())
                .add("CurrentDeployment", currentDeployment())
                .add("ActiveConfigurations", hasActiveConfigurations() ? activeConfigurations() : null)
                .add("Tags", hasTags() ? tags() : null).add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cluster":
            return Optional.ofNullable(clazz.cast(cluster()));
        case "serviceName":
            return Optional.ofNullable(clazz.cast(serviceName()));
        case "serviceArn":
            return Optional.ofNullable(clazz.cast(serviceArn()));
        case "infrastructureRoleArn":
            return Optional.ofNullable(clazz.cast(infrastructureRoleArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "currentDeployment":
            return Optional.ofNullable(clazz.cast(currentDeployment()));
        case "activeConfigurations":
            return Optional.ofNullable(clazz.cast(activeConfigurations()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("cluster", CLUSTER_FIELD);
        map.put("serviceName", SERVICE_NAME_FIELD);
        map.put("serviceArn", SERVICE_ARN_FIELD);
        map.put("infrastructureRoleArn", INFRASTRUCTURE_ROLE_ARN_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("currentDeployment", CURRENT_DEPLOYMENT_FIELD);
        map.put("activeConfigurations", ACTIVE_CONFIGURATIONS_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ECSExpressGatewayService, T> g) {
        return obj -> g.apply((ECSExpressGatewayService) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ECSExpressGatewayService> {
        /**
         * <p>
         * The short name or full ARN of the cluster that hosts the Express service.
         * </p>
         * 
         * @param cluster
         *        The short name or full ARN of the cluster that hosts the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cluster(String cluster);

        /**
         * <p>
         * The name of the Express service.
         * </p>
         * 
         * @param serviceName
         *        The name of the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceName(String serviceName);

        /**
         * <p>
         * The ARN that identifies the Express service.
         * </p>
         * 
         * @param serviceArn
         *        The ARN that identifies the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceArn(String serviceArn);

        /**
         * <p>
         * The ARN of the infrastructure role that manages Amazon Web Services resources for the Express service.
         * </p>
         * 
         * @param infrastructureRoleArn
         *        The ARN of the infrastructure role that manages Amazon Web Services resources for the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder infrastructureRoleArn(String infrastructureRoleArn);

        /**
         * <p>
         * The current status of the Express service.
         * </p>
         * 
         * @param status
         *        The current status of the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(ExpressGatewayServiceStatus status);

        /**
         * <p>
         * The current status of the Express service.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExpressGatewayServiceStatus.Builder}
         * avoiding the need to create one manually via {@link ExpressGatewayServiceStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExpressGatewayServiceStatus.Builder#build()} is called
         * immediately and its result is passed to {@link #status(ExpressGatewayServiceStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link ExpressGatewayServiceStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(ExpressGatewayServiceStatus)
         */
        default Builder status(Consumer<ExpressGatewayServiceStatus.Builder> status) {
            return status(ExpressGatewayServiceStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The current deployment configuration for the Express service.
         * </p>
         * 
         * @param currentDeployment
         *        The current deployment configuration for the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentDeployment(String currentDeployment);

        /**
         * <p>
         * The list of active service configurations for the Express service.
         * </p>
         * 
         * @param activeConfigurations
         *        The list of active service configurations for the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeConfigurations(Collection<ExpressGatewayServiceConfiguration> activeConfigurations);

        /**
         * <p>
         * The list of active service configurations for the Express service.
         * </p>
         * 
         * @param activeConfigurations
         *        The list of active service configurations for the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeConfigurations(ExpressGatewayServiceConfiguration... activeConfigurations);

        /**
         * <p>
         * The list of active service configurations for the Express service.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ExpressGatewayServiceConfiguration.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.ExpressGatewayServiceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ExpressGatewayServiceConfiguration.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #activeConfigurations(List<ExpressGatewayServiceConfiguration>)}.
         * 
         * @param activeConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ExpressGatewayServiceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #activeConfigurations(java.util.Collection<ExpressGatewayServiceConfiguration>)
         */
        Builder activeConfigurations(Consumer<ExpressGatewayServiceConfiguration.Builder>... activeConfigurations);

        /**
         * <p>
         * The metadata applied to the Express service.
         * </p>
         * 
         * @param tags
         *        The metadata applied to the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The metadata applied to the Express service.
         * </p>
         * 
         * @param tags
         *        The metadata applied to the Express service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The metadata applied to the Express service.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The Unix timestamp for when the Express service was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix timestamp for when the Express service was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Unix timestamp for when the Express service was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The Unix timestamp for when the Express service was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String cluster;

        private String serviceName;

        private String serviceArn;

        private String infrastructureRoleArn;

        private ExpressGatewayServiceStatus status;

        private String currentDeployment;

        private List<ExpressGatewayServiceConfiguration> activeConfigurations = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(ECSExpressGatewayService model) {
            cluster(model.cluster);
            serviceName(model.serviceName);
            serviceArn(model.serviceArn);
            infrastructureRoleArn(model.infrastructureRoleArn);
            status(model.status);
            currentDeployment(model.currentDeployment);
            activeConfigurations(model.activeConfigurations);
            tags(model.tags);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
        }

        public final String getCluster() {
            return cluster;
        }

        public final void setCluster(String cluster) {
            this.cluster = cluster;
        }

        @Override
        public final Builder cluster(String cluster) {
            this.cluster = cluster;
            return this;
        }

        public final String getServiceName() {
            return serviceName;
        }

        public final void setServiceName(String serviceName) {
            this.serviceName = serviceName;
        }

        @Override
        public final Builder serviceName(String serviceName) {
            this.serviceName = serviceName;
            return this;
        }

        public final String getServiceArn() {
            return serviceArn;
        }

        public final void setServiceArn(String serviceArn) {
            this.serviceArn = serviceArn;
        }

        @Override
        public final Builder serviceArn(String serviceArn) {
            this.serviceArn = serviceArn;
            return this;
        }

        public final String getInfrastructureRoleArn() {
            return infrastructureRoleArn;
        }

        public final void setInfrastructureRoleArn(String infrastructureRoleArn) {
            this.infrastructureRoleArn = infrastructureRoleArn;
        }

        @Override
        public final Builder infrastructureRoleArn(String infrastructureRoleArn) {
            this.infrastructureRoleArn = infrastructureRoleArn;
            return this;
        }

        public final ExpressGatewayServiceStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(ExpressGatewayServiceStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(ExpressGatewayServiceStatus status) {
            this.status = status;
            return this;
        }

        public final String getCurrentDeployment() {
            return currentDeployment;
        }

        public final void setCurrentDeployment(String currentDeployment) {
            this.currentDeployment = currentDeployment;
        }

        @Override
        public final Builder currentDeployment(String currentDeployment) {
            this.currentDeployment = currentDeployment;
            return this;
        }

        public final List<ExpressGatewayServiceConfiguration.Builder> getActiveConfigurations() {
            List<ExpressGatewayServiceConfiguration.Builder> result = ExpressGatewayServiceConfigurationsCopier
                    .copyToBuilder(this.activeConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActiveConfigurations(Collection<ExpressGatewayServiceConfiguration.BuilderImpl> activeConfigurations) {
            this.activeConfigurations = ExpressGatewayServiceConfigurationsCopier.copyFromBuilder(activeConfigurations);
        }

        @Override
        public final Builder activeConfigurations(Collection<ExpressGatewayServiceConfiguration> activeConfigurations) {
            this.activeConfigurations = ExpressGatewayServiceConfigurationsCopier.copy(activeConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder activeConfigurations(ExpressGatewayServiceConfiguration... activeConfigurations) {
            activeConfigurations(Arrays.asList(activeConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder activeConfigurations(Consumer<ExpressGatewayServiceConfiguration.Builder>... activeConfigurations) {
            activeConfigurations(Stream.of(activeConfigurations)
                    .map(c -> ExpressGatewayServiceConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public ECSExpressGatewayService build() {
            return new ECSExpressGatewayService(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
