/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The overrides that should be sent to a container. An empty container override can be passed in. An example of an
 * empty container override would be <code>{"containerOverrides": [ ] }</code>. If a non-empty container override is
 * specified, the <code>name</code> parameter must be included.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerOverride.Builder, ContainerOverride> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerOverride::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ContainerOverride::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .getter(getter(ContainerOverride::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> CPU_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerOverride::cpu)).setter(setter(Builder::cpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpu").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerOverride::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<Integer> MEMORY_RESERVATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerOverride::memoryReservation)).setter(setter(Builder::memoryReservation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryReservation").build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .getter(getter(ContainerOverride::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, COMMAND_FIELD,
            ENVIRONMENT_FIELD, CPU_FIELD, MEMORY_FIELD, MEMORY_RESERVATION_FIELD, RESOURCE_REQUIREMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> command;

    private final List<KeyValuePair> environment;

    private final Integer cpu;

    private final Integer memory;

    private final Integer memoryReservation;

    private final List<ResourceRequirement> resourceRequirements;

    private ContainerOverride(BuilderImpl builder) {
        this.name = builder.name;
        this.command = builder.command;
        this.environment = builder.environment;
        this.cpu = builder.cpu;
        this.memory = builder.memory;
        this.memoryReservation = builder.memoryReservation;
        this.resourceRequirements = builder.resourceRequirements;
    }

    /**
     * <p>
     * The name of the container that receives the override. This parameter is required if any override is specified.
     * </p>
     * 
     * @return The name of the container that receives the override. This parameter is required if any override is
     *         specified.
     */
    public String name() {
        return name;
    }

    /**
     * Returns true if the Command property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command to send to the container that overrides the default command from the Docker image or the task
     * definition. You must also specify a container name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCommand()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The command to send to the container that overrides the default command from the Docker image or the task
     *         definition. You must also specify a container name.
     */
    public List<String> command() {
        return command;
    }

    /**
     * Returns true if the Environment property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to send to the container. You can add new environment variables, which are added to the
     * container at launch, or you can override the existing environment variables from the Docker image or the task
     * definition. You must also specify a container name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The environment variables to send to the container. You can add new environment variables, which are
     *         added to the container at launch, or you can override the existing environment variables from the Docker
     *         image or the task definition. You must also specify a container name.
     */
    public List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * <p>
     * The number of <code>cpu</code> units reserved for the container, instead of the default value from the task
     * definition. You must also specify a container name.
     * </p>
     * 
     * @return The number of <code>cpu</code> units reserved for the container, instead of the default value from the
     *         task definition. You must also specify a container name.
     */
    public Integer cpu() {
        return cpu;
    }

    /**
     * <p>
     * The hard limit (in MiB) of memory to present to the container, instead of the default value from the task
     * definition. If your container attempts to exceed the memory specified here, the container is killed. You must
     * also specify a container name.
     * </p>
     * 
     * @return The hard limit (in MiB) of memory to present to the container, instead of the default value from the task
     *         definition. If your container attempts to exceed the memory specified here, the container is killed. You
     *         must also specify a container name.
     */
    public Integer memory() {
        return memory;
    }

    /**
     * <p>
     * The soft limit (in MiB) of memory to reserve for the container, instead of the default value from the task
     * definition. You must also specify a container name.
     * </p>
     * 
     * @return The soft limit (in MiB) of memory to reserve for the container, instead of the default value from the
     *         task definition. You must also specify a container name.
     */
    public Integer memoryReservation() {
        return memoryReservation;
    }

    /**
     * Returns true if the ResourceRequirements property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of a resource to assign to a container, instead of the default value from the task
     * definition. The only supported resource is a GPU.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceRequirements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type and amount of a resource to assign to a container, instead of the default value from the task
     *         definition. The only supported resource is a GPU.
     */
    public List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(command());
        hashCode = 31 * hashCode + Objects.hashCode(environment());
        hashCode = 31 * hashCode + Objects.hashCode(cpu());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(memoryReservation());
        hashCode = 31 * hashCode + Objects.hashCode(resourceRequirements());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerOverride)) {
            return false;
        }
        ContainerOverride other = (ContainerOverride) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(command(), other.command())
                && Objects.equals(environment(), other.environment()) && Objects.equals(cpu(), other.cpu())
                && Objects.equals(memory(), other.memory()) && Objects.equals(memoryReservation(), other.memoryReservation())
                && Objects.equals(resourceRequirements(), other.resourceRequirements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContainerOverride").add("Name", name()).add("Command", command())
                .add("Environment", environment()).add("Cpu", cpu()).add("Memory", memory())
                .add("MemoryReservation", memoryReservation()).add("ResourceRequirements", resourceRequirements()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "cpu":
            return Optional.ofNullable(clazz.cast(cpu()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "memoryReservation":
            return Optional.ofNullable(clazz.cast(memoryReservation()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerOverride, T> g) {
        return obj -> g.apply((ContainerOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerOverride> {
        /**
         * <p>
         * The name of the container that receives the override. This parameter is required if any override is
         * specified.
         * </p>
         * 
         * @param name
         *        The name of the container that receives the override. This parameter is required if any override is
         *        specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the task
         * definition. You must also specify a container name.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        task definition. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the task
         * definition. You must also specify a container name.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        task definition. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * task definition. You must also specify a container name.
         * </p>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the task definition. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * task definition. You must also specify a container name.
         * </p>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the task definition. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * task definition. You must also specify a container name.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the need to
         * create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(List<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * The number of <code>cpu</code> units reserved for the container, instead of the default value from the task
         * definition. You must also specify a container name.
         * </p>
         * 
         * @param cpu
         *        The number of <code>cpu</code> units reserved for the container, instead of the default value from the
         *        task definition. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpu(Integer cpu);

        /**
         * <p>
         * The hard limit (in MiB) of memory to present to the container, instead of the default value from the task
         * definition. If your container attempts to exceed the memory specified here, the container is killed. You must
         * also specify a container name.
         * </p>
         * 
         * @param memory
         *        The hard limit (in MiB) of memory to present to the container, instead of the default value from the
         *        task definition. If your container attempts to exceed the memory specified here, the container is
         *        killed. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The soft limit (in MiB) of memory to reserve for the container, instead of the default value from the task
         * definition. You must also specify a container name.
         * </p>
         * 
         * @param memoryReservation
         *        The soft limit (in MiB) of memory to reserve for the container, instead of the default value from the
         *        task definition. You must also specify a container name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryReservation(Integer memoryReservation);

        /**
         * <p>
         * The type and amount of a resource to assign to a container, instead of the default value from the task
         * definition. The only supported resource is a GPU.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container, instead of the default value from the task
         *        definition. The only supported resource is a GPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container, instead of the default value from the task
         * definition. The only supported resource is a GPU.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container, instead of the default value from the task
         *        definition. The only supported resource is a GPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container, instead of the default value from the task
         * definition. The only supported resource is a GPU.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceRequirement>.Builder} avoiding the
         * need to create one manually via {@link List<ResourceRequirement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceRequirement>.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on {@link List<ResourceRequirement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(List<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private Integer cpu;

        private Integer memory;

        private Integer memoryReservation;

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerOverride model) {
            name(model.name);
            command(model.command);
            environment(model.environment);
            cpu(model.cpu);
            memory(model.memory);
            memoryReservation(model.memoryReservation);
            resourceRequirements(model.resourceRequirements);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<String> getCommand() {
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        public final Collection<KeyValuePair.Builder> getEnvironment() {
            return environment != null ? environment.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        public final Integer getCpu() {
            return cpu;
        }

        @Override
        public final Builder cpu(Integer cpu) {
            this.cpu = cpu;
            return this;
        }

        public final void setCpu(Integer cpu) {
            this.cpu = cpu;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Integer getMemoryReservation() {
            return memoryReservation;
        }

        @Override
        public final Builder memoryReservation(Integer memoryReservation) {
            this.memoryReservation = memoryReservation;
            return this;
        }

        public final void setMemoryReservation(Integer memoryReservation) {
            this.memoryReservation = memoryReservation;
        }

        public final Collection<ResourceRequirement.Builder> getResourceRequirements() {
            return resourceRequirements != null ? resourceRequirements.stream().map(ResourceRequirement::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public ContainerOverride build() {
            return new ContainerOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
