/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of the Auto Scaling group for the capacity provider.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingGroupProvider implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingGroupProvider.Builder, AutoScalingGroupProvider> {
    private static final SdkField<String> AUTO_SCALING_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutoScalingGroupProvider::autoScalingGroupArn)).setter(setter(Builder::autoScalingGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingGroupArn").build())
            .build();

    private static final SdkField<ManagedScaling> MANAGED_SCALING_FIELD = SdkField
            .<ManagedScaling> builder(MarshallingType.SDK_POJO).getter(getter(AutoScalingGroupProvider::managedScaling))
            .setter(setter(Builder::managedScaling)).constructor(ManagedScaling::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("managedScaling").build()).build();

    private static final SdkField<String> MANAGED_TERMINATION_PROTECTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(AutoScalingGroupProvider::managedTerminationProtectionAsString))
            .setter(setter(Builder::managedTerminationProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("managedTerminationProtection")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTO_SCALING_GROUP_ARN_FIELD,
            MANAGED_SCALING_FIELD, MANAGED_TERMINATION_PROTECTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String autoScalingGroupArn;

    private final ManagedScaling managedScaling;

    private final String managedTerminationProtection;

    private AutoScalingGroupProvider(BuilderImpl builder) {
        this.autoScalingGroupArn = builder.autoScalingGroupArn;
        this.managedScaling = builder.managedScaling;
        this.managedTerminationProtection = builder.managedTerminationProtection;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that identifies the Auto Scaling group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that identifies the Auto Scaling group.
     */
    public String autoScalingGroupArn() {
        return autoScalingGroupArn;
    }

    /**
     * <p>
     * The managed scaling settings for the Auto Scaling group capacity provider.
     * </p>
     * 
     * @return The managed scaling settings for the Auto Scaling group capacity provider.
     */
    public ManagedScaling managedScaling() {
        return managedScaling;
    }

    /**
     * <p>
     * The managed termination protection setting to use for the Auto Scaling group capacity provider. This determines
     * whether the Auto Scaling group has managed termination protection.
     * </p>
     * <important>
     * <p>
     * When using managed termination protection, managed scaling must also be used otherwise managed termination
     * protection will not work.
     * </p>
     * </important>
     * <p>
     * When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto Scaling
     * group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and each instance
     * in the Auto Scaling group must have instance protection from scale-in actions enabled as well. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
     * >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * When managed termination protection is disabled, your Amazon EC2 instances are not protected from termination
     * when the Auto Scaling group scales in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #managedTerminationProtection} will return {@link ManagedTerminationProtection#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #managedTerminationProtectionAsString}.
     * </p>
     * 
     * @return The managed termination protection setting to use for the Auto Scaling group capacity provider. This
     *         determines whether the Auto Scaling group has managed termination protection.</p> <important>
     *         <p>
     *         When using managed termination protection, managed scaling must also be used otherwise managed
     *         termination protection will not work.
     *         </p>
     *         </important>
     *         <p>
     *         When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto
     *         Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group
     *         and each instance in the Auto Scaling group must have instance protection from scale-in actions enabled
     *         as well. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
     *         >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
     *         </p>
     *         <p>
     *         When managed termination protection is disabled, your Amazon EC2 instances are not protected from
     *         termination when the Auto Scaling group scales in.
     * @see ManagedTerminationProtection
     */
    public ManagedTerminationProtection managedTerminationProtection() {
        return ManagedTerminationProtection.fromValue(managedTerminationProtection);
    }

    /**
     * <p>
     * The managed termination protection setting to use for the Auto Scaling group capacity provider. This determines
     * whether the Auto Scaling group has managed termination protection.
     * </p>
     * <important>
     * <p>
     * When using managed termination protection, managed scaling must also be used otherwise managed termination
     * protection will not work.
     * </p>
     * </important>
     * <p>
     * When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto Scaling
     * group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and each instance
     * in the Auto Scaling group must have instance protection from scale-in actions enabled as well. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
     * >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * When managed termination protection is disabled, your Amazon EC2 instances are not protected from termination
     * when the Auto Scaling group scales in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #managedTerminationProtection} will return {@link ManagedTerminationProtection#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #managedTerminationProtectionAsString}.
     * </p>
     * 
     * @return The managed termination protection setting to use for the Auto Scaling group capacity provider. This
     *         determines whether the Auto Scaling group has managed termination protection.</p> <important>
     *         <p>
     *         When using managed termination protection, managed scaling must also be used otherwise managed
     *         termination protection will not work.
     *         </p>
     *         </important>
     *         <p>
     *         When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto
     *         Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group
     *         and each instance in the Auto Scaling group must have instance protection from scale-in actions enabled
     *         as well. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
     *         >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
     *         </p>
     *         <p>
     *         When managed termination protection is disabled, your Amazon EC2 instances are not protected from
     *         termination when the Auto Scaling group scales in.
     * @see ManagedTerminationProtection
     */
    public String managedTerminationProtectionAsString() {
        return managedTerminationProtection;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(managedScaling());
        hashCode = 31 * hashCode + Objects.hashCode(managedTerminationProtectionAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingGroupProvider)) {
            return false;
        }
        AutoScalingGroupProvider other = (AutoScalingGroupProvider) obj;
        return Objects.equals(autoScalingGroupArn(), other.autoScalingGroupArn())
                && Objects.equals(managedScaling(), other.managedScaling())
                && Objects.equals(managedTerminationProtectionAsString(), other.managedTerminationProtectionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AutoScalingGroupProvider").add("AutoScalingGroupArn", autoScalingGroupArn())
                .add("ManagedScaling", managedScaling())
                .add("ManagedTerminationProtection", managedTerminationProtectionAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "autoScalingGroupArn":
            return Optional.ofNullable(clazz.cast(autoScalingGroupArn()));
        case "managedScaling":
            return Optional.ofNullable(clazz.cast(managedScaling()));
        case "managedTerminationProtection":
            return Optional.ofNullable(clazz.cast(managedTerminationProtectionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingGroupProvider, T> g) {
        return obj -> g.apply((AutoScalingGroupProvider) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingGroupProvider> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) that identifies the Auto Scaling group.
         * </p>
         * 
         * @param autoScalingGroupArn
         *        The Amazon Resource Name (ARN) that identifies the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupArn(String autoScalingGroupArn);

        /**
         * <p>
         * The managed scaling settings for the Auto Scaling group capacity provider.
         * </p>
         * 
         * @param managedScaling
         *        The managed scaling settings for the Auto Scaling group capacity provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedScaling(ManagedScaling managedScaling);

        /**
         * <p>
         * The managed scaling settings for the Auto Scaling group capacity provider.
         * </p>
         * This is a convenience that creates an instance of the {@link ManagedScaling.Builder} avoiding the need to
         * create one manually via {@link ManagedScaling#builder()}.
         *
         * When the {@link Consumer} completes, {@link ManagedScaling.Builder#build()} is called immediately and its
         * result is passed to {@link #managedScaling(ManagedScaling)}.
         * 
         * @param managedScaling
         *        a consumer that will call methods on {@link ManagedScaling.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #managedScaling(ManagedScaling)
         */
        default Builder managedScaling(Consumer<ManagedScaling.Builder> managedScaling) {
            return managedScaling(ManagedScaling.builder().applyMutation(managedScaling).build());
        }

        /**
         * <p>
         * The managed termination protection setting to use for the Auto Scaling group capacity provider. This
         * determines whether the Auto Scaling group has managed termination protection.
         * </p>
         * <important>
         * <p>
         * When using managed termination protection, managed scaling must also be used otherwise managed termination
         * protection will not work.
         * </p>
         * </important>
         * <p>
         * When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto
         * Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and
         * each instance in the Auto Scaling group must have instance protection from scale-in actions enabled as well.
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
         * >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
         * </p>
         * <p>
         * When managed termination protection is disabled, your Amazon EC2 instances are not protected from termination
         * when the Auto Scaling group scales in.
         * </p>
         * 
         * @param managedTerminationProtection
         *        The managed termination protection setting to use for the Auto Scaling group capacity provider. This
         *        determines whether the Auto Scaling group has managed termination protection.</p> <important>
         *        <p>
         *        When using managed termination protection, managed scaling must also be used otherwise managed
         *        termination protection will not work.
         *        </p>
         *        </important>
         *        <p>
         *        When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an
         *        Auto Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling
         *        group and each instance in the Auto Scaling group must have instance protection from scale-in actions
         *        enabled as well. For more information, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
         *        >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
         *        </p>
         *        <p>
         *        When managed termination protection is disabled, your Amazon EC2 instances are not protected from
         *        termination when the Auto Scaling group scales in.
         * @see ManagedTerminationProtection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ManagedTerminationProtection
         */
        Builder managedTerminationProtection(String managedTerminationProtection);

        /**
         * <p>
         * The managed termination protection setting to use for the Auto Scaling group capacity provider. This
         * determines whether the Auto Scaling group has managed termination protection.
         * </p>
         * <important>
         * <p>
         * When using managed termination protection, managed scaling must also be used otherwise managed termination
         * protection will not work.
         * </p>
         * </important>
         * <p>
         * When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto
         * Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and
         * each instance in the Auto Scaling group must have instance protection from scale-in actions enabled as well.
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
         * >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
         * </p>
         * <p>
         * When managed termination protection is disabled, your Amazon EC2 instances are not protected from termination
         * when the Auto Scaling group scales in.
         * </p>
         * 
         * @param managedTerminationProtection
         *        The managed termination protection setting to use for the Auto Scaling group capacity provider. This
         *        determines whether the Auto Scaling group has managed termination protection.</p> <important>
         *        <p>
         *        When using managed termination protection, managed scaling must also be used otherwise managed
         *        termination protection will not work.
         *        </p>
         *        </important>
         *        <p>
         *        When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an
         *        Auto Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling
         *        group and each instance in the Auto Scaling group must have instance protection from scale-in actions
         *        enabled as well. For more information, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection"
         *        >Instance Protection</a> in the <i>AWS Auto Scaling User Guide</i>.
         *        </p>
         *        <p>
         *        When managed termination protection is disabled, your Amazon EC2 instances are not protected from
         *        termination when the Auto Scaling group scales in.
         * @see ManagedTerminationProtection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ManagedTerminationProtection
         */
        Builder managedTerminationProtection(ManagedTerminationProtection managedTerminationProtection);
    }

    static final class BuilderImpl implements Builder {
        private String autoScalingGroupArn;

        private ManagedScaling managedScaling;

        private String managedTerminationProtection;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingGroupProvider model) {
            autoScalingGroupArn(model.autoScalingGroupArn);
            managedScaling(model.managedScaling);
            managedTerminationProtection(model.managedTerminationProtection);
        }

        public final String getAutoScalingGroupArn() {
            return autoScalingGroupArn;
        }

        @Override
        public final Builder autoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
            return this;
        }

        public final void setAutoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
        }

        public final ManagedScaling.Builder getManagedScaling() {
            return managedScaling != null ? managedScaling.toBuilder() : null;
        }

        @Override
        public final Builder managedScaling(ManagedScaling managedScaling) {
            this.managedScaling = managedScaling;
            return this;
        }

        public final void setManagedScaling(ManagedScaling.BuilderImpl managedScaling) {
            this.managedScaling = managedScaling != null ? managedScaling.build() : null;
        }

        public final String getManagedTerminationProtectionAsString() {
            return managedTerminationProtection;
        }

        @Override
        public final Builder managedTerminationProtection(String managedTerminationProtection) {
            this.managedTerminationProtection = managedTerminationProtection;
            return this;
        }

        @Override
        public final Builder managedTerminationProtection(ManagedTerminationProtection managedTerminationProtection) {
            this.managedTerminationProtection(managedTerminationProtection == null ? null : managedTerminationProtection
                    .toString());
            return this;
        }

        public final void setManagedTerminationProtection(String managedTerminationProtection) {
            this.managedTerminationProtection = managedTerminationProtection;
        }

        @Override
        public AutoScalingGroupProvider build() {
            return new AutoScalingGroupProvider(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
