/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Deletes specified images within a specified repository. Images are specified with either the <code>imageTag</code> or
 * <code>imageDigest</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchDeleteImageRequest extends EcrRequest implements
        ToCopyableBuilder<BatchDeleteImageRequest.Builder, BatchDeleteImageRequest> {
    private static final SdkField<String> REGISTRY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BatchDeleteImageRequest::registryId)).setter(setter(Builder::registryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registryId").build()).build();

    private static final SdkField<String> REPOSITORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BatchDeleteImageRequest::repositoryName)).setter(setter(Builder::repositoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryName").build()).build();

    private static final SdkField<List<ImageIdentifier>> IMAGE_IDS_FIELD = SdkField
            .<List<ImageIdentifier>> builder(MarshallingType.LIST)
            .getter(getter(BatchDeleteImageRequest::imageIds))
            .setter(setter(Builder::imageIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ImageIdentifier> builder(MarshallingType.SDK_POJO)
                                            .constructor(ImageIdentifier::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REGISTRY_ID_FIELD,
            REPOSITORY_NAME_FIELD, IMAGE_IDS_FIELD));

    private final String registryId;

    private final String repositoryName;

    private final List<ImageIdentifier> imageIds;

    private BatchDeleteImageRequest(BuilderImpl builder) {
        super(builder);
        this.registryId = builder.registryId;
        this.repositoryName = builder.repositoryName;
        this.imageIds = builder.imageIds;
    }

    /**
     * <p>
     * The AWS account ID associated with the registry that contains the image to delete. If you do not specify a
     * registry, the default registry is assumed.
     * </p>
     * 
     * @return The AWS account ID associated with the registry that contains the image to delete. If you do not specify
     *         a registry, the default registry is assumed.
     */
    public String registryId() {
        return registryId;
    }

    /**
     * <p>
     * The repository that contains the image to delete.
     * </p>
     * 
     * @return The repository that contains the image to delete.
     */
    public String repositoryName() {
        return repositoryName;
    }

    /**
     * Returns true if the ImageIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasImageIds() {
        return imageIds != null && !(imageIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of image ID references that correspond to images to delete. The format of the <code>imageIds</code>
     * reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasImageIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of image ID references that correspond to images to delete. The format of the
     *         <code>imageIds</code> reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
     */
    public List<ImageIdentifier> imageIds() {
        return imageIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(registryId());
        hashCode = 31 * hashCode + Objects.hashCode(repositoryName());
        hashCode = 31 * hashCode + Objects.hashCode(imageIds());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchDeleteImageRequest)) {
            return false;
        }
        BatchDeleteImageRequest other = (BatchDeleteImageRequest) obj;
        return Objects.equals(registryId(), other.registryId()) && Objects.equals(repositoryName(), other.repositoryName())
                && Objects.equals(imageIds(), other.imageIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BatchDeleteImageRequest").add("RegistryId", registryId())
                .add("RepositoryName", repositoryName()).add("ImageIds", imageIds()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "registryId":
            return Optional.ofNullable(clazz.cast(registryId()));
        case "repositoryName":
            return Optional.ofNullable(clazz.cast(repositoryName()));
        case "imageIds":
            return Optional.ofNullable(clazz.cast(imageIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchDeleteImageRequest, T> g) {
        return obj -> g.apply((BatchDeleteImageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EcrRequest.Builder, SdkPojo, CopyableBuilder<Builder, BatchDeleteImageRequest> {
        /**
         * <p>
         * The AWS account ID associated with the registry that contains the image to delete. If you do not specify a
         * registry, the default registry is assumed.
         * </p>
         * 
         * @param registryId
         *        The AWS account ID associated with the registry that contains the image to delete. If you do not
         *        specify a registry, the default registry is assumed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registryId(String registryId);

        /**
         * <p>
         * The repository that contains the image to delete.
         * </p>
         * 
         * @param repositoryName
         *        The repository that contains the image to delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryName(String repositoryName);

        /**
         * <p>
         * A list of image ID references that correspond to images to delete. The format of the <code>imageIds</code>
         * reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
         * </p>
         * 
         * @param imageIds
         *        A list of image ID references that correspond to images to delete. The format of the
         *        <code>imageIds</code> reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageIds(Collection<ImageIdentifier> imageIds);

        /**
         * <p>
         * A list of image ID references that correspond to images to delete. The format of the <code>imageIds</code>
         * reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
         * </p>
         * 
         * @param imageIds
         *        A list of image ID references that correspond to images to delete. The format of the
         *        <code>imageIds</code> reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageIds(ImageIdentifier... imageIds);

        /**
         * <p>
         * A list of image ID references that correspond to images to delete. The format of the <code>imageIds</code>
         * reference is <code>imageTag=tag</code> or <code>imageDigest=digest</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ImageIdentifier>.Builder} avoiding the need
         * to create one manually via {@link List<ImageIdentifier>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ImageIdentifier>.Builder#build()} is called immediately and
         * its result is passed to {@link #imageIds(List<ImageIdentifier>)}.
         * 
         * @param imageIds
         *        a consumer that will call methods on {@link List<ImageIdentifier>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageIds(List<ImageIdentifier>)
         */
        Builder imageIds(Consumer<ImageIdentifier.Builder>... imageIds);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EcrRequest.BuilderImpl implements Builder {
        private String registryId;

        private String repositoryName;

        private List<ImageIdentifier> imageIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BatchDeleteImageRequest model) {
            super(model);
            registryId(model.registryId);
            repositoryName(model.repositoryName);
            imageIds(model.imageIds);
        }

        public final String getRegistryId() {
            return registryId;
        }

        @Override
        public final Builder registryId(String registryId) {
            this.registryId = registryId;
            return this;
        }

        public final void setRegistryId(String registryId) {
            this.registryId = registryId;
        }

        public final String getRepositoryName() {
            return repositoryName;
        }

        @Override
        public final Builder repositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
            return this;
        }

        public final void setRepositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
        }

        public final Collection<ImageIdentifier.Builder> getImageIds() {
            return imageIds != null ? imageIds.stream().map(ImageIdentifier::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder imageIds(Collection<ImageIdentifier> imageIds) {
            this.imageIds = ImageIdentifierListCopier.copy(imageIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageIds(ImageIdentifier... imageIds) {
            imageIds(Arrays.asList(imageIds));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageIds(Consumer<ImageIdentifier.Builder>... imageIds) {
            imageIds(Stream.of(imageIds).map(c -> ImageIdentifier.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setImageIds(Collection<ImageIdentifier.BuilderImpl> imageIds) {
            this.imageIds = ImageIdentifierListCopier.copyFromBuilder(imageIds);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public BatchDeleteImageRequest build() {
            return new BatchDeleteImageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
