/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration settings for VPC Encryption Control.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-encryption-controls.html">Enforce VPC encryption in
 * transit</a> in the <i>Amazon VPC User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpcEncryptionControlConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<VpcEncryptionControlConfiguration.Builder, VpcEncryptionControlConfiguration> {
    private static final SdkField<String> MODE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Mode")
            .getter(getter(VpcEncryptionControlConfiguration::modeAsString))
            .setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode")
                    .unmarshallLocationName("Mode").build()).build();

    private static final SdkField<String> INTERNET_GATEWAY_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InternetGatewayExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::internetGatewayExclusionAsString))
            .setter(setter(Builder::internetGatewayExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InternetGatewayExclusion")
                    .unmarshallLocationName("InternetGatewayExclusion").build()).build();

    private static final SdkField<String> EGRESS_ONLY_INTERNET_GATEWAY_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EgressOnlyInternetGatewayExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::egressOnlyInternetGatewayExclusionAsString))
            .setter(setter(Builder::egressOnlyInternetGatewayExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EgressOnlyInternetGatewayExclusion")
                    .unmarshallLocationName("EgressOnlyInternetGatewayExclusion").build()).build();

    private static final SdkField<String> NAT_GATEWAY_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NatGatewayExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::natGatewayExclusionAsString))
            .setter(setter(Builder::natGatewayExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NatGatewayExclusion")
                    .unmarshallLocationName("NatGatewayExclusion").build()).build();

    private static final SdkField<String> VIRTUAL_PRIVATE_GATEWAY_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VirtualPrivateGatewayExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::virtualPrivateGatewayExclusionAsString))
            .setter(setter(Builder::virtualPrivateGatewayExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VirtualPrivateGatewayExclusion")
                    .unmarshallLocationName("VirtualPrivateGatewayExclusion").build()).build();

    private static final SdkField<String> VPC_PEERING_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcPeeringExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::vpcPeeringExclusionAsString))
            .setter(setter(Builder::vpcPeeringExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcPeeringExclusion")
                    .unmarshallLocationName("VpcPeeringExclusion").build()).build();

    private static final SdkField<String> LAMBDA_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LambdaExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::lambdaExclusionAsString))
            .setter(setter(Builder::lambdaExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LambdaExclusion")
                    .unmarshallLocationName("LambdaExclusion").build()).build();

    private static final SdkField<String> VPC_LATTICE_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcLatticeExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::vpcLatticeExclusionAsString))
            .setter(setter(Builder::vpcLatticeExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcLatticeExclusion")
                    .unmarshallLocationName("VpcLatticeExclusion").build()).build();

    private static final SdkField<String> ELASTIC_FILE_SYSTEM_EXCLUSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ElasticFileSystemExclusion")
            .getter(getter(VpcEncryptionControlConfiguration::elasticFileSystemExclusionAsString))
            .setter(setter(Builder::elasticFileSystemExclusion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ElasticFileSystemExclusion")
                    .unmarshallLocationName("ElasticFileSystemExclusion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MODE_FIELD,
            INTERNET_GATEWAY_EXCLUSION_FIELD, EGRESS_ONLY_INTERNET_GATEWAY_EXCLUSION_FIELD, NAT_GATEWAY_EXCLUSION_FIELD,
            VIRTUAL_PRIVATE_GATEWAY_EXCLUSION_FIELD, VPC_PEERING_EXCLUSION_FIELD, LAMBDA_EXCLUSION_FIELD,
            VPC_LATTICE_EXCLUSION_FIELD, ELASTIC_FILE_SYSTEM_EXCLUSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String mode;

    private final String internetGatewayExclusion;

    private final String egressOnlyInternetGatewayExclusion;

    private final String natGatewayExclusion;

    private final String virtualPrivateGatewayExclusion;

    private final String vpcPeeringExclusion;

    private final String lambdaExclusion;

    private final String vpcLatticeExclusion;

    private final String elasticFileSystemExclusion;

    private VpcEncryptionControlConfiguration(BuilderImpl builder) {
        this.mode = builder.mode;
        this.internetGatewayExclusion = builder.internetGatewayExclusion;
        this.egressOnlyInternetGatewayExclusion = builder.egressOnlyInternetGatewayExclusion;
        this.natGatewayExclusion = builder.natGatewayExclusion;
        this.virtualPrivateGatewayExclusion = builder.virtualPrivateGatewayExclusion;
        this.vpcPeeringExclusion = builder.vpcPeeringExclusion;
        this.lambdaExclusion = builder.lambdaExclusion;
        this.vpcLatticeExclusion = builder.vpcLatticeExclusion;
        this.elasticFileSystemExclusion = builder.elasticFileSystemExclusion;
    }

    /**
     * <p>
     * The encryption mode for the VPC Encryption Control configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link VpcEncryptionControlMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The encryption mode for the VPC Encryption Control configuration.
     * @see VpcEncryptionControlMode
     */
    public final VpcEncryptionControlMode mode() {
        return VpcEncryptionControlMode.fromValue(mode);
    }

    /**
     * <p>
     * The encryption mode for the VPC Encryption Control configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link VpcEncryptionControlMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The encryption mode for the VPC Encryption Control configuration.
     * @see VpcEncryptionControlMode
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * Specifies whether to exclude internet gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #internetGatewayExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #internetGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude internet gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput internetGatewayExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(internetGatewayExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude internet gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #internetGatewayExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #internetGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude internet gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String internetGatewayExclusionAsString() {
        return internetGatewayExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #egressOnlyInternetGatewayExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #egressOnlyInternetGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput egressOnlyInternetGatewayExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(egressOnlyInternetGatewayExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #egressOnlyInternetGatewayExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #egressOnlyInternetGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String egressOnlyInternetGatewayExclusionAsString() {
        return egressOnlyInternetGatewayExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude NAT gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #natGatewayExclusion} will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #natGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude NAT gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput natGatewayExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(natGatewayExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude NAT gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #natGatewayExclusion} will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #natGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude NAT gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String natGatewayExclusionAsString() {
        return natGatewayExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #virtualPrivateGatewayExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #virtualPrivateGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput virtualPrivateGatewayExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(virtualPrivateGatewayExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #virtualPrivateGatewayExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #virtualPrivateGatewayExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String virtualPrivateGatewayExclusionAsString() {
        return virtualPrivateGatewayExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vpcPeeringExclusion} will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #vpcPeeringExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput vpcPeeringExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(vpcPeeringExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vpcPeeringExclusion} will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #vpcPeeringExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String vpcPeeringExclusionAsString() {
        return vpcPeeringExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude Lambda function traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lambdaExclusion}
     * will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #lambdaExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude Lambda function traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput lambdaExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(lambdaExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude Lambda function traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lambdaExclusion}
     * will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #lambdaExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude Lambda function traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String lambdaExclusionAsString() {
        return lambdaExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vpcLatticeExclusion} will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #vpcLatticeExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput vpcLatticeExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(vpcLatticeExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vpcLatticeExclusion} will return {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #vpcLatticeExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String vpcLatticeExclusionAsString() {
        return vpcLatticeExclusion;
    }

    /**
     * <p>
     * Specifies whether to exclude Elastic File System traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #elasticFileSystemExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #elasticFileSystemExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude Elastic File System traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final VpcEncryptionControlExclusionStateInput elasticFileSystemExclusion() {
        return VpcEncryptionControlExclusionStateInput.fromValue(elasticFileSystemExclusion);
    }

    /**
     * <p>
     * Specifies whether to exclude Elastic File System traffic from encryption enforcement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #elasticFileSystemExclusion} will return
     * {@link VpcEncryptionControlExclusionStateInput#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #elasticFileSystemExclusionAsString}.
     * </p>
     * 
     * @return Specifies whether to exclude Elastic File System traffic from encryption enforcement.
     * @see VpcEncryptionControlExclusionStateInput
     */
    public final String elasticFileSystemExclusionAsString() {
        return elasticFileSystemExclusion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(internetGatewayExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(egressOnlyInternetGatewayExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(natGatewayExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(virtualPrivateGatewayExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcPeeringExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcLatticeExclusionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(elasticFileSystemExclusionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpcEncryptionControlConfiguration)) {
            return false;
        }
        VpcEncryptionControlConfiguration other = (VpcEncryptionControlConfiguration) obj;
        return Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(internetGatewayExclusionAsString(), other.internetGatewayExclusionAsString())
                && Objects.equals(egressOnlyInternetGatewayExclusionAsString(),
                        other.egressOnlyInternetGatewayExclusionAsString())
                && Objects.equals(natGatewayExclusionAsString(), other.natGatewayExclusionAsString())
                && Objects.equals(virtualPrivateGatewayExclusionAsString(), other.virtualPrivateGatewayExclusionAsString())
                && Objects.equals(vpcPeeringExclusionAsString(), other.vpcPeeringExclusionAsString())
                && Objects.equals(lambdaExclusionAsString(), other.lambdaExclusionAsString())
                && Objects.equals(vpcLatticeExclusionAsString(), other.vpcLatticeExclusionAsString())
                && Objects.equals(elasticFileSystemExclusionAsString(), other.elasticFileSystemExclusionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VpcEncryptionControlConfiguration").add("Mode", modeAsString())
                .add("InternetGatewayExclusion", internetGatewayExclusionAsString())
                .add("EgressOnlyInternetGatewayExclusion", egressOnlyInternetGatewayExclusionAsString())
                .add("NatGatewayExclusion", natGatewayExclusionAsString())
                .add("VirtualPrivateGatewayExclusion", virtualPrivateGatewayExclusionAsString())
                .add("VpcPeeringExclusion", vpcPeeringExclusionAsString()).add("LambdaExclusion", lambdaExclusionAsString())
                .add("VpcLatticeExclusion", vpcLatticeExclusionAsString())
                .add("ElasticFileSystemExclusion", elasticFileSystemExclusionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "InternetGatewayExclusion":
            return Optional.ofNullable(clazz.cast(internetGatewayExclusionAsString()));
        case "EgressOnlyInternetGatewayExclusion":
            return Optional.ofNullable(clazz.cast(egressOnlyInternetGatewayExclusionAsString()));
        case "NatGatewayExclusion":
            return Optional.ofNullable(clazz.cast(natGatewayExclusionAsString()));
        case "VirtualPrivateGatewayExclusion":
            return Optional.ofNullable(clazz.cast(virtualPrivateGatewayExclusionAsString()));
        case "VpcPeeringExclusion":
            return Optional.ofNullable(clazz.cast(vpcPeeringExclusionAsString()));
        case "LambdaExclusion":
            return Optional.ofNullable(clazz.cast(lambdaExclusionAsString()));
        case "VpcLatticeExclusion":
            return Optional.ofNullable(clazz.cast(vpcLatticeExclusionAsString()));
        case "ElasticFileSystemExclusion":
            return Optional.ofNullable(clazz.cast(elasticFileSystemExclusionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Mode", MODE_FIELD);
        map.put("InternetGatewayExclusion", INTERNET_GATEWAY_EXCLUSION_FIELD);
        map.put("EgressOnlyInternetGatewayExclusion", EGRESS_ONLY_INTERNET_GATEWAY_EXCLUSION_FIELD);
        map.put("NatGatewayExclusion", NAT_GATEWAY_EXCLUSION_FIELD);
        map.put("VirtualPrivateGatewayExclusion", VIRTUAL_PRIVATE_GATEWAY_EXCLUSION_FIELD);
        map.put("VpcPeeringExclusion", VPC_PEERING_EXCLUSION_FIELD);
        map.put("LambdaExclusion", LAMBDA_EXCLUSION_FIELD);
        map.put("VpcLatticeExclusion", VPC_LATTICE_EXCLUSION_FIELD);
        map.put("ElasticFileSystemExclusion", ELASTIC_FILE_SYSTEM_EXCLUSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<VpcEncryptionControlConfiguration, T> g) {
        return obj -> g.apply((VpcEncryptionControlConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpcEncryptionControlConfiguration> {
        /**
         * <p>
         * The encryption mode for the VPC Encryption Control configuration.
         * </p>
         * 
         * @param mode
         *        The encryption mode for the VPC Encryption Control configuration.
         * @see VpcEncryptionControlMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * The encryption mode for the VPC Encryption Control configuration.
         * </p>
         * 
         * @param mode
         *        The encryption mode for the VPC Encryption Control configuration.
         * @see VpcEncryptionControlMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlMode
         */
        Builder mode(VpcEncryptionControlMode mode);

        /**
         * <p>
         * Specifies whether to exclude internet gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param internetGatewayExclusion
         *        Specifies whether to exclude internet gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder internetGatewayExclusion(String internetGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude internet gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param internetGatewayExclusion
         *        Specifies whether to exclude internet gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder internetGatewayExclusion(VpcEncryptionControlExclusionStateInput internetGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param egressOnlyInternetGatewayExclusion
         *        Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder egressOnlyInternetGatewayExclusion(String egressOnlyInternetGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param egressOnlyInternetGatewayExclusion
         *        Specifies whether to exclude egress-only internet gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder egressOnlyInternetGatewayExclusion(VpcEncryptionControlExclusionStateInput egressOnlyInternetGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude NAT gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param natGatewayExclusion
         *        Specifies whether to exclude NAT gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder natGatewayExclusion(String natGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude NAT gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param natGatewayExclusion
         *        Specifies whether to exclude NAT gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder natGatewayExclusion(VpcEncryptionControlExclusionStateInput natGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param virtualPrivateGatewayExclusion
         *        Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder virtualPrivateGatewayExclusion(String virtualPrivateGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
         * </p>
         * 
         * @param virtualPrivateGatewayExclusion
         *        Specifies whether to exclude virtual private gateway traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder virtualPrivateGatewayExclusion(VpcEncryptionControlExclusionStateInput virtualPrivateGatewayExclusion);

        /**
         * <p>
         * Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
         * </p>
         * 
         * @param vpcPeeringExclusion
         *        Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder vpcPeeringExclusion(String vpcPeeringExclusion);

        /**
         * <p>
         * Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
         * </p>
         * 
         * @param vpcPeeringExclusion
         *        Specifies whether to exclude VPC peering connection traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder vpcPeeringExclusion(VpcEncryptionControlExclusionStateInput vpcPeeringExclusion);

        /**
         * <p>
         * Specifies whether to exclude Lambda function traffic from encryption enforcement.
         * </p>
         * 
         * @param lambdaExclusion
         *        Specifies whether to exclude Lambda function traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder lambdaExclusion(String lambdaExclusion);

        /**
         * <p>
         * Specifies whether to exclude Lambda function traffic from encryption enforcement.
         * </p>
         * 
         * @param lambdaExclusion
         *        Specifies whether to exclude Lambda function traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder lambdaExclusion(VpcEncryptionControlExclusionStateInput lambdaExclusion);

        /**
         * <p>
         * Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
         * </p>
         * 
         * @param vpcLatticeExclusion
         *        Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder vpcLatticeExclusion(String vpcLatticeExclusion);

        /**
         * <p>
         * Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
         * </p>
         * 
         * @param vpcLatticeExclusion
         *        Specifies whether to exclude VPC Lattice traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder vpcLatticeExclusion(VpcEncryptionControlExclusionStateInput vpcLatticeExclusion);

        /**
         * <p>
         * Specifies whether to exclude Elastic File System traffic from encryption enforcement.
         * </p>
         * 
         * @param elasticFileSystemExclusion
         *        Specifies whether to exclude Elastic File System traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder elasticFileSystemExclusion(String elasticFileSystemExclusion);

        /**
         * <p>
         * Specifies whether to exclude Elastic File System traffic from encryption enforcement.
         * </p>
         * 
         * @param elasticFileSystemExclusion
         *        Specifies whether to exclude Elastic File System traffic from encryption enforcement.
         * @see VpcEncryptionControlExclusionStateInput
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEncryptionControlExclusionStateInput
         */
        Builder elasticFileSystemExclusion(VpcEncryptionControlExclusionStateInput elasticFileSystemExclusion);
    }

    static final class BuilderImpl implements Builder {
        private String mode;

        private String internetGatewayExclusion;

        private String egressOnlyInternetGatewayExclusion;

        private String natGatewayExclusion;

        private String virtualPrivateGatewayExclusion;

        private String vpcPeeringExclusion;

        private String lambdaExclusion;

        private String vpcLatticeExclusion;

        private String elasticFileSystemExclusion;

        private BuilderImpl() {
        }

        private BuilderImpl(VpcEncryptionControlConfiguration model) {
            mode(model.mode);
            internetGatewayExclusion(model.internetGatewayExclusion);
            egressOnlyInternetGatewayExclusion(model.egressOnlyInternetGatewayExclusion);
            natGatewayExclusion(model.natGatewayExclusion);
            virtualPrivateGatewayExclusion(model.virtualPrivateGatewayExclusion);
            vpcPeeringExclusion(model.vpcPeeringExclusion);
            lambdaExclusion(model.lambdaExclusion);
            vpcLatticeExclusion(model.vpcLatticeExclusion);
            elasticFileSystemExclusion(model.elasticFileSystemExclusion);
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(VpcEncryptionControlMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final String getInternetGatewayExclusion() {
            return internetGatewayExclusion;
        }

        public final void setInternetGatewayExclusion(String internetGatewayExclusion) {
            this.internetGatewayExclusion = internetGatewayExclusion;
        }

        @Override
        public final Builder internetGatewayExclusion(String internetGatewayExclusion) {
            this.internetGatewayExclusion = internetGatewayExclusion;
            return this;
        }

        @Override
        public final Builder internetGatewayExclusion(VpcEncryptionControlExclusionStateInput internetGatewayExclusion) {
            this.internetGatewayExclusion(internetGatewayExclusion == null ? null : internetGatewayExclusion.toString());
            return this;
        }

        public final String getEgressOnlyInternetGatewayExclusion() {
            return egressOnlyInternetGatewayExclusion;
        }

        public final void setEgressOnlyInternetGatewayExclusion(String egressOnlyInternetGatewayExclusion) {
            this.egressOnlyInternetGatewayExclusion = egressOnlyInternetGatewayExclusion;
        }

        @Override
        public final Builder egressOnlyInternetGatewayExclusion(String egressOnlyInternetGatewayExclusion) {
            this.egressOnlyInternetGatewayExclusion = egressOnlyInternetGatewayExclusion;
            return this;
        }

        @Override
        public final Builder egressOnlyInternetGatewayExclusion(
                VpcEncryptionControlExclusionStateInput egressOnlyInternetGatewayExclusion) {
            this.egressOnlyInternetGatewayExclusion(egressOnlyInternetGatewayExclusion == null ? null
                    : egressOnlyInternetGatewayExclusion.toString());
            return this;
        }

        public final String getNatGatewayExclusion() {
            return natGatewayExclusion;
        }

        public final void setNatGatewayExclusion(String natGatewayExclusion) {
            this.natGatewayExclusion = natGatewayExclusion;
        }

        @Override
        public final Builder natGatewayExclusion(String natGatewayExclusion) {
            this.natGatewayExclusion = natGatewayExclusion;
            return this;
        }

        @Override
        public final Builder natGatewayExclusion(VpcEncryptionControlExclusionStateInput natGatewayExclusion) {
            this.natGatewayExclusion(natGatewayExclusion == null ? null : natGatewayExclusion.toString());
            return this;
        }

        public final String getVirtualPrivateGatewayExclusion() {
            return virtualPrivateGatewayExclusion;
        }

        public final void setVirtualPrivateGatewayExclusion(String virtualPrivateGatewayExclusion) {
            this.virtualPrivateGatewayExclusion = virtualPrivateGatewayExclusion;
        }

        @Override
        public final Builder virtualPrivateGatewayExclusion(String virtualPrivateGatewayExclusion) {
            this.virtualPrivateGatewayExclusion = virtualPrivateGatewayExclusion;
            return this;
        }

        @Override
        public final Builder virtualPrivateGatewayExclusion(VpcEncryptionControlExclusionStateInput virtualPrivateGatewayExclusion) {
            this.virtualPrivateGatewayExclusion(virtualPrivateGatewayExclusion == null ? null : virtualPrivateGatewayExclusion
                    .toString());
            return this;
        }

        public final String getVpcPeeringExclusion() {
            return vpcPeeringExclusion;
        }

        public final void setVpcPeeringExclusion(String vpcPeeringExclusion) {
            this.vpcPeeringExclusion = vpcPeeringExclusion;
        }

        @Override
        public final Builder vpcPeeringExclusion(String vpcPeeringExclusion) {
            this.vpcPeeringExclusion = vpcPeeringExclusion;
            return this;
        }

        @Override
        public final Builder vpcPeeringExclusion(VpcEncryptionControlExclusionStateInput vpcPeeringExclusion) {
            this.vpcPeeringExclusion(vpcPeeringExclusion == null ? null : vpcPeeringExclusion.toString());
            return this;
        }

        public final String getLambdaExclusion() {
            return lambdaExclusion;
        }

        public final void setLambdaExclusion(String lambdaExclusion) {
            this.lambdaExclusion = lambdaExclusion;
        }

        @Override
        public final Builder lambdaExclusion(String lambdaExclusion) {
            this.lambdaExclusion = lambdaExclusion;
            return this;
        }

        @Override
        public final Builder lambdaExclusion(VpcEncryptionControlExclusionStateInput lambdaExclusion) {
            this.lambdaExclusion(lambdaExclusion == null ? null : lambdaExclusion.toString());
            return this;
        }

        public final String getVpcLatticeExclusion() {
            return vpcLatticeExclusion;
        }

        public final void setVpcLatticeExclusion(String vpcLatticeExclusion) {
            this.vpcLatticeExclusion = vpcLatticeExclusion;
        }

        @Override
        public final Builder vpcLatticeExclusion(String vpcLatticeExclusion) {
            this.vpcLatticeExclusion = vpcLatticeExclusion;
            return this;
        }

        @Override
        public final Builder vpcLatticeExclusion(VpcEncryptionControlExclusionStateInput vpcLatticeExclusion) {
            this.vpcLatticeExclusion(vpcLatticeExclusion == null ? null : vpcLatticeExclusion.toString());
            return this;
        }

        public final String getElasticFileSystemExclusion() {
            return elasticFileSystemExclusion;
        }

        public final void setElasticFileSystemExclusion(String elasticFileSystemExclusion) {
            this.elasticFileSystemExclusion = elasticFileSystemExclusion;
        }

        @Override
        public final Builder elasticFileSystemExclusion(String elasticFileSystemExclusion) {
            this.elasticFileSystemExclusion = elasticFileSystemExclusion;
            return this;
        }

        @Override
        public final Builder elasticFileSystemExclusion(VpcEncryptionControlExclusionStateInput elasticFileSystemExclusion) {
            this.elasticFileSystemExclusion(elasticFileSystemExclusion == null ? null : elasticFileSystemExclusion.toString());
            return this;
        }

        @Override
        public VpcEncryptionControlConfiguration build() {
            return new VpcEncryptionControlConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
