/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ebs;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.ebs.model.EbsException;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ebs.model.ValidationException;
import software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable;
import software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable;

/**
 * Service client for accessing Amazon EBS. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * You can use the Amazon Elastic Block Store (EBS) direct APIs to directly read the data on your EBS snapshots, and
 * identify the difference between two snapshots. You can view the details of blocks in an EBS snapshot, compare the
 * block difference between two snapshots, and directly access the data in a snapshot. If you’re an independent software
 * vendor (ISV) who offers backup services for EBS, the EBS direct APIs makes it easier and more cost-effective to track
 * incremental changes on your EBS volumes via EBS snapshots. This can be done without having to create new volumes from
 * EBS snapshots, and then use EC2 instances to compare the differences.
 * </p>
 * <p>
 * This API reference provides detailed information about the actions, data types, parameters, and errors of the EBS
 * direct APIs. For more information about the elements that make up the EBS direct APIs, and examples of how to use
 * them effectively, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshot.html">Accessing the Contents of an
 * EBS Snapshot</a>. For more information about how to use the EBS direct APIs, see the <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-accessing-snapshots.html">EBS direct APIs User
 * Guide</a>. To view the currently supported AWS Regions and endpoints for the EBS direct APIs, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#ebs_region">AWS Service Endpoints</a> in the <i>AWS
 * General Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface EbsClient extends SdkClient {
    String SERVICE_NAME = "ebs";

    /**
     * Create a {@link EbsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EbsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EbsClient}.
     */
    static EbsClientBuilder builder() {
        return new DefaultEbsClientBuilder();
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetSnapshotBlockResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest,
            ResponseTransformer<GetSnapshotBlockResponse, ReturnT> responseTransformer) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetSnapshotBlockResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> ReturnT getSnapshotBlock(Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest,
            ResponseTransformer<GetSnapshotBlockResponse, ReturnT> responseTransformer) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build(),
                responseTransformer);
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getSnapshotBlock(GetSnapshotBlockRequest, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSnapshotBlockResponse getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest, Path filePath)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(getSnapshotBlockRequest, ResponseTransformer.toFile(filePath));
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getSnapshotBlock(GetSnapshotBlockRequest, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSnapshotBlockResponse getSnapshotBlock(Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest,
            Path filePath) throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            EbsException {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build(), filePath);
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<GetSnapshotBlockResponse> getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(getSnapshotBlockRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseInputStream<GetSnapshotBlockResponse> getSnapshotBlock(
            Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<GetSnapshotBlockResponse> getSnapshotBlockAsBytes(GetSnapshotBlockRequest getSnapshotBlockRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlock(getSnapshotBlockRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSnapshotBlockRequest.Builder} avoiding the need
     * to create one manually via {@link GetSnapshotBlockRequest#builder()}
     * </p>
     *
     * @param getSnapshotBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetSnapshotBlockRequest.Builder} to create a request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The data content of the block.
     *         </p>
     *         '.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see #getObject(getSnapshotBlock, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    default ResponseBytes<GetSnapshotBlockResponse> getSnapshotBlockAsBytes(
            Consumer<GetSnapshotBlockRequest.Builder> getSnapshotBlockRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return getSnapshotBlockAsBytes(GetSnapshotBlockRequest.builder().applyMutation(getSnapshotBlockRequest).build());
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store
     * snapshots of the same volume/snapshot lineage.
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return Result of the ListChangedBlocks operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksResponse listChangedBlocks(ListChangedBlocksRequest listChangedBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store
     * snapshots of the same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChangedBlocksRequest.Builder} avoiding the need
     * to create one manually via {@link ListChangedBlocksRequest#builder()}
     * </p>
     *
     * @param listChangedBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListChangedBlocksRequest.Builder} to create a request.
     * @return Result of the ListChangedBlocks operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksResponse listChangedBlocks(Consumer<ListChangedBlocksRequest.Builder> listChangedBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return listChangedBlocks(ListChangedBlocksRequest.builder().applyMutation(listChangedBlocksRequest).build());
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store
     * snapshots of the same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client
     *             .listChangedBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksIterable listChangedBlocksPaginator(ListChangedBlocksRequest listChangedBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store
     * snapshots of the same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client
     *             .listChangedBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChangedBlocksRequest.Builder} avoiding the need
     * to create one manually via {@link ListChangedBlocksRequest#builder()}
     * </p>
     *
     * @param listChangedBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListChangedBlocksRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChangedBlocksIterable listChangedBlocksPaginator(
            Consumer<ListChangedBlocksRequest.Builder> listChangedBlocksRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return listChangedBlocksPaginator(ListChangedBlocksRequest.builder().applyMutation(listChangedBlocksRequest).build());
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return Result of the ListSnapshotBlocks operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksResponse listSnapshotBlocks(ListSnapshotBlocksRequest listSnapshotBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSnapshotBlocksRequest.Builder} avoiding the
     * need to create one manually via {@link ListSnapshotBlocksRequest#builder()}
     * </p>
     *
     * @param listSnapshotBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListSnapshotBlocksRequest.Builder} to create a
     *        request.
     * @return Result of the ListSnapshotBlocks operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksResponse listSnapshotBlocks(Consumer<ListSnapshotBlocksRequest.Builder> listSnapshotBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return listSnapshotBlocks(ListSnapshotBlocksRequest.builder().applyMutation(listSnapshotBlocksRequest).build());
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client
     *             .listSnapshotBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksIterable listSnapshotBlocksPaginator(ListSnapshotBlocksRequest listSnapshotBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client
     *             .listSnapshotBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSnapshotBlocksRequest.Builder} avoiding the
     * need to create one manually via {@link ListSnapshotBlocksRequest#builder()}
     * </p>
     *
     * @param listSnapshotBlocksRequest
     *        A {@link Consumer} that will call methods on {@link ListSnapshotBlocksRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSnapshotBlocksIterable listSnapshotBlocksPaginator(
            Consumer<ListSnapshotBlocksRequest.Builder> listSnapshotBlocksRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return listSnapshotBlocksPaginator(ListSnapshotBlocksRequest.builder().applyMutation(listSnapshotBlocksRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("ebs");
    }
}
