/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input of a <code>GetItem</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetItemRequest extends DynamoDbRequest implements ToCopyableBuilder<GetItemRequest.Builder, GetItemRequest> {
    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetItemRequest::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableName").build()).build();

    private static final SdkField<Map<String, AttributeValue>> KEY_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .getter(getter(GetItemRequest::key))
            .setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Key").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> ATTRIBUTES_TO_GET_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GetItemRequest::attributesToGet))
            .setter(setter(Builder::attributesToGet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributesToGet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> CONSISTENT_READ_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(GetItemRequest::consistentRead)).setter(setter(Builder::consistentRead))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsistentRead").build()).build();

    private static final SdkField<String> RETURN_CONSUMED_CAPACITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetItemRequest::returnConsumedCapacityAsString)).setter(setter(Builder::returnConsumedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnConsumedCapacity").build())
            .build();

    private static final SdkField<String> PROJECTION_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetItemRequest::projectionExpression)).setter(setter(Builder::projectionExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProjectionExpression").build())
            .build();

    private static final SdkField<Map<String, String>> EXPRESSION_ATTRIBUTE_NAMES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(GetItemRequest::expressionAttributeNames))
            .setter(setter(Builder::expressionAttributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpressionAttributeNames").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TABLE_NAME_FIELD, KEY_FIELD,
            ATTRIBUTES_TO_GET_FIELD, CONSISTENT_READ_FIELD, RETURN_CONSUMED_CAPACITY_FIELD, PROJECTION_EXPRESSION_FIELD,
            EXPRESSION_ATTRIBUTE_NAMES_FIELD));

    private final String tableName;

    private final Map<String, AttributeValue> key;

    private final List<String> attributesToGet;

    private final Boolean consistentRead;

    private final String returnConsumedCapacity;

    private final String projectionExpression;

    private final Map<String, String> expressionAttributeNames;

    private GetItemRequest(BuilderImpl builder) {
        super(builder);
        this.tableName = builder.tableName;
        this.key = builder.key;
        this.attributesToGet = builder.attributesToGet;
        this.consistentRead = builder.consistentRead;
        this.returnConsumedCapacity = builder.returnConsumedCapacity;
        this.projectionExpression = builder.projectionExpression;
        this.expressionAttributeNames = builder.expressionAttributeNames;
    }

    /**
     * <p>
     * The name of the table containing the requested item.
     * </p>
     * 
     * @return The name of the table containing the requested item.
     */
    public String tableName() {
        return tableName;
    }

    /**
     * Returns true if the Key property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasKey() {
        return key != null && !(key instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of attribute names to <code>AttributeValue</code> objects, representing the primary key of the item to
     * retrieve.
     * </p>
     * <p>
     * For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only
     * need to provide a value for the partition key. For a composite primary key, you must provide values for both the
     * partition key and the sort key.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasKey()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of attribute names to <code>AttributeValue</code> objects, representing the primary key of the item
     *         to retrieve.</p>
     *         <p>
     *         For the primary key, you must provide all of the attributes. For example, with a simple primary key, you
     *         only need to provide a value for the partition key. For a composite primary key, you must provide values
     *         for both the partition key and the sort key.
     */
    public Map<String, AttributeValue> key() {
        return key;
    }

    /**
     * Returns true if the AttributesToGet property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAttributesToGet() {
        return attributesToGet != null && !(attributesToGet instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This is a legacy parameter. Use <code>ProjectionExpression</code> instead. For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html"
     * >AttributesToGet</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributesToGet()} to see if a value was sent in this field.
     * </p>
     * 
     * @return This is a legacy parameter. Use <code>ProjectionExpression</code> instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html"
     *         >AttributesToGet</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public List<String> attributesToGet() {
        return attributesToGet;
    }

    /**
     * <p>
     * Determines the read consistency model: If set to <code>true</code>, then the operation uses strongly consistent
     * reads; otherwise, the operation uses eventually consistent reads.
     * </p>
     * 
     * @return Determines the read consistency model: If set to <code>true</code>, then the operation uses strongly
     *         consistent reads; otherwise, the operation uses eventually consistent reads.
     */
    public Boolean consistentRead() {
        return consistentRead;
    }

    /**
     * Returns the value of the ReturnConsumedCapacity property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #returnConsumedCapacity} will return {@link ReturnConsumedCapacity#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #returnConsumedCapacityAsString}.
     * </p>
     * 
     * @return The value of the ReturnConsumedCapacity property for this object.
     * @see ReturnConsumedCapacity
     */
    public ReturnConsumedCapacity returnConsumedCapacity() {
        return ReturnConsumedCapacity.fromValue(returnConsumedCapacity);
    }

    /**
     * Returns the value of the ReturnConsumedCapacity property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #returnConsumedCapacity} will return {@link ReturnConsumedCapacity#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #returnConsumedCapacityAsString}.
     * </p>
     * 
     * @return The value of the ReturnConsumedCapacity property for this object.
     * @see ReturnConsumedCapacity
     */
    public String returnConsumedCapacityAsString() {
        return returnConsumedCapacity;
    }

    /**
     * <p>
     * A string that identifies one or more attributes to retrieve from the table. These attributes can include scalars,
     * sets, or elements of a JSON document. The attributes in the expression must be separated by commas.
     * </p>
     * <p>
     * If no attribute names are specified, then all attributes are returned. If any of the requested attributes are not
     * found, they do not appear in the result.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
     * >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * 
     * @return A string that identifies one or more attributes to retrieve from the table. These attributes can include
     *         scalars, sets, or elements of a JSON document. The attributes in the expression must be separated by
     *         commas.</p>
     *         <p>
     *         If no attribute names are specified, then all attributes are returned. If any of the requested attributes
     *         are not found, they do not appear in the result.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
     *         >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public String projectionExpression() {
        return projectionExpression;
    }

    /**
     * Returns true if the ExpressionAttributeNames property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasExpressionAttributeNames() {
        return expressionAttributeNames != null && !(expressionAttributeNames instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more substitution tokens for attribute names in an expression. The following are some use cases for using
     * <code>ExpressionAttributeNames</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To access an attribute whose name conflicts with a DynamoDB reserved word.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create a placeholder for repeating occurrences of an attribute name in an expression.
     * </p>
     * </li>
     * <li>
     * <p>
     * To prevent special characters in an attribute name from being misinterpreted in an expression.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Use the <b>#</b> character in an expression to dereference an attribute name. For example, consider the following
     * attribute name:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Percentile</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * The name of this attribute conflicts with a reserved word, so it cannot be used directly in an expression. (For
     * the complete list of reserved words, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ReservedWords.html">Reserved Words</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>). To work around this, you could specify the following for
     * <code>ExpressionAttributeNames</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>{"#P":"Percentile"}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * You could then use this substitution in an expression, as in this example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>#P = :val</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Tokens that begin with the <b>:</b> character are <i>expression attribute values</i>, which are placeholders for
     * the actual value at runtime.
     * </p>
     * </note>
     * <p>
     * For more information on expression attribute names, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
     * >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExpressionAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more substitution tokens for attribute names in an expression. The following are some use cases
     *         for using <code>ExpressionAttributeNames</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To access an attribute whose name conflicts with a DynamoDB reserved word.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create a placeholder for repeating occurrences of an attribute name in an expression.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To prevent special characters in an attribute name from being misinterpreted in an expression.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Use the <b>#</b> character in an expression to dereference an attribute name. For example, consider the
     *         following attribute name:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Percentile</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The name of this attribute conflicts with a reserved word, so it cannot be used directly in an
     *         expression. (For the complete list of reserved words, see <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ReservedWords.html">Reserved
     *         Words</a> in the <i>Amazon DynamoDB Developer Guide</i>). To work around this, you could specify the
     *         following for <code>ExpressionAttributeNames</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>{"#P":"Percentile"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You could then use this substitution in an expression, as in this example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>#P = :val</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Tokens that begin with the <b>:</b> character are <i>expression attribute values</i>, which are
     *         placeholders for the actual value at runtime.
     *         </p>
     *         </note>
     *         <p>
     *         For more information on expression attribute names, see <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
     *         >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public Map<String, String> expressionAttributeNames() {
        return expressionAttributeNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(attributesToGet());
        hashCode = 31 * hashCode + Objects.hashCode(consistentRead());
        hashCode = 31 * hashCode + Objects.hashCode(returnConsumedCapacityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(projectionExpression());
        hashCode = 31 * hashCode + Objects.hashCode(expressionAttributeNames());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetItemRequest)) {
            return false;
        }
        GetItemRequest other = (GetItemRequest) obj;
        return Objects.equals(tableName(), other.tableName()) && Objects.equals(key(), other.key())
                && Objects.equals(attributesToGet(), other.attributesToGet())
                && Objects.equals(consistentRead(), other.consistentRead())
                && Objects.equals(returnConsumedCapacityAsString(), other.returnConsumedCapacityAsString())
                && Objects.equals(projectionExpression(), other.projectionExpression())
                && Objects.equals(expressionAttributeNames(), other.expressionAttributeNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetItemRequest").add("TableName", tableName()).add("Key", key())
                .add("AttributesToGet", attributesToGet()).add("ConsistentRead", consistentRead())
                .add("ReturnConsumedCapacity", returnConsumedCapacityAsString())
                .add("ProjectionExpression", projectionExpression()).add("ExpressionAttributeNames", expressionAttributeNames())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "AttributesToGet":
            return Optional.ofNullable(clazz.cast(attributesToGet()));
        case "ConsistentRead":
            return Optional.ofNullable(clazz.cast(consistentRead()));
        case "ReturnConsumedCapacity":
            return Optional.ofNullable(clazz.cast(returnConsumedCapacityAsString()));
        case "ProjectionExpression":
            return Optional.ofNullable(clazz.cast(projectionExpression()));
        case "ExpressionAttributeNames":
            return Optional.ofNullable(clazz.cast(expressionAttributeNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetItemRequest, T> g) {
        return obj -> g.apply((GetItemRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DynamoDbRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetItemRequest> {
        /**
         * <p>
         * The name of the table containing the requested item.
         * </p>
         * 
         * @param tableName
         *        The name of the table containing the requested item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * A map of attribute names to <code>AttributeValue</code> objects, representing the primary key of the item to
         * retrieve.
         * </p>
         * <p>
         * For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only
         * need to provide a value for the partition key. For a composite primary key, you must provide values for both
         * the partition key and the sort key.
         * </p>
         * 
         * @param key
         *        A map of attribute names to <code>AttributeValue</code> objects, representing the primary key of the
         *        item to retrieve.</p>
         *        <p>
         *        For the primary key, you must provide all of the attributes. For example, with a simple primary key,
         *        you only need to provide a value for the partition key. For a composite primary key, you must provide
         *        values for both the partition key and the sort key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(Map<String, AttributeValue> key);

        /**
         * <p>
         * This is a legacy parameter. Use <code>ProjectionExpression</code> instead. For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html"
         * >AttributesToGet</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param attributesToGet
         *        This is a legacy parameter. Use <code>ProjectionExpression</code> instead. For more information, see
         *        <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html"
         *        >AttributesToGet</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributesToGet(Collection<String> attributesToGet);

        /**
         * <p>
         * This is a legacy parameter. Use <code>ProjectionExpression</code> instead. For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html"
         * >AttributesToGet</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param attributesToGet
         *        This is a legacy parameter. Use <code>ProjectionExpression</code> instead. For more information, see
         *        <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html"
         *        >AttributesToGet</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributesToGet(String... attributesToGet);

        /**
         * <p>
         * Determines the read consistency model: If set to <code>true</code>, then the operation uses strongly
         * consistent reads; otherwise, the operation uses eventually consistent reads.
         * </p>
         * 
         * @param consistentRead
         *        Determines the read consistency model: If set to <code>true</code>, then the operation uses strongly
         *        consistent reads; otherwise, the operation uses eventually consistent reads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consistentRead(Boolean consistentRead);

        /**
         * Sets the value of the ReturnConsumedCapacity property for this object.
         *
         * @param returnConsumedCapacity
         *        The new value for the ReturnConsumedCapacity property for this object.
         * @see ReturnConsumedCapacity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReturnConsumedCapacity
         */
        Builder returnConsumedCapacity(String returnConsumedCapacity);

        /**
         * Sets the value of the ReturnConsumedCapacity property for this object.
         *
         * @param returnConsumedCapacity
         *        The new value for the ReturnConsumedCapacity property for this object.
         * @see ReturnConsumedCapacity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReturnConsumedCapacity
         */
        Builder returnConsumedCapacity(ReturnConsumedCapacity returnConsumedCapacity);

        /**
         * <p>
         * A string that identifies one or more attributes to retrieve from the table. These attributes can include
         * scalars, sets, or elements of a JSON document. The attributes in the expression must be separated by commas.
         * </p>
         * <p>
         * If no attribute names are specified, then all attributes are returned. If any of the requested attributes are
         * not found, they do not appear in the result.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
         * >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param projectionExpression
         *        A string that identifies one or more attributes to retrieve from the table. These attributes can
         *        include scalars, sets, or elements of a JSON document. The attributes in the expression must be
         *        separated by commas.</p>
         *        <p>
         *        If no attribute names are specified, then all attributes are returned. If any of the requested
         *        attributes are not found, they do not appear in the result.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
         *        >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectionExpression(String projectionExpression);

        /**
         * <p>
         * One or more substitution tokens for attribute names in an expression. The following are some use cases for
         * using <code>ExpressionAttributeNames</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To access an attribute whose name conflicts with a DynamoDB reserved word.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create a placeholder for repeating occurrences of an attribute name in an expression.
         * </p>
         * </li>
         * <li>
         * <p>
         * To prevent special characters in an attribute name from being misinterpreted in an expression.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Use the <b>#</b> character in an expression to dereference an attribute name. For example, consider the
         * following attribute name:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Percentile</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * The name of this attribute conflicts with a reserved word, so it cannot be used directly in an expression.
         * (For the complete list of reserved words, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ReservedWords.html">Reserved Words</a>
         * in the <i>Amazon DynamoDB Developer Guide</i>). To work around this, you could specify the following for
         * <code>ExpressionAttributeNames</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>{"#P":"Percentile"}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * You could then use this substitution in an expression, as in this example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>#P = :val</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Tokens that begin with the <b>:</b> character are <i>expression attribute values</i>, which are placeholders
         * for the actual value at runtime.
         * </p>
         * </note>
         * <p>
         * For more information on expression attribute names, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
         * >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param expressionAttributeNames
         *        One or more substitution tokens for attribute names in an expression. The following are some use cases
         *        for using <code>ExpressionAttributeNames</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To access an attribute whose name conflicts with a DynamoDB reserved word.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create a placeholder for repeating occurrences of an attribute name in an expression.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To prevent special characters in an attribute name from being misinterpreted in an expression.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Use the <b>#</b> character in an expression to dereference an attribute name. For example, consider
         *        the following attribute name:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Percentile</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The name of this attribute conflicts with a reserved word, so it cannot be used directly in an
         *        expression. (For the complete list of reserved words, see <a
         *        href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ReservedWords.html">Reserved
         *        Words</a> in the <i>Amazon DynamoDB Developer Guide</i>). To work around this, you could specify the
         *        following for <code>ExpressionAttributeNames</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>{"#P":"Percentile"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You could then use this substitution in an expression, as in this example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>#P = :val</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Tokens that begin with the <b>:</b> character are <i>expression attribute values</i>, which are
         *        placeholders for the actual value at runtime.
         *        </p>
         *        </note>
         *        <p>
         *        For more information on expression attribute names, see <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html"
         *        >Specifying Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expressionAttributeNames(Map<String, String> expressionAttributeNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DynamoDbRequest.BuilderImpl implements Builder {
        private String tableName;

        private Map<String, AttributeValue> key = DefaultSdkAutoConstructMap.getInstance();

        private List<String> attributesToGet = DefaultSdkAutoConstructList.getInstance();

        private Boolean consistentRead;

        private String returnConsumedCapacity;

        private String projectionExpression;

        private Map<String, String> expressionAttributeNames = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetItemRequest model) {
            super(model);
            tableName(model.tableName);
            key(model.key);
            attributesToGet(model.attributesToGet);
            consistentRead(model.consistentRead);
            returnConsumedCapacity(model.returnConsumedCapacity);
            projectionExpression(model.projectionExpression);
            expressionAttributeNames(model.expressionAttributeNames);
        }

        public final String getTableName() {
            return tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public final Map<String, AttributeValue.Builder> getKey() {
            return key != null ? CollectionUtils.mapValues(key, AttributeValue::toBuilder) : null;
        }

        @Override
        public final Builder key(Map<String, AttributeValue> key) {
            this.key = KeyCopier.copy(key);
            return this;
        }

        public final void setKey(Map<String, AttributeValue.BuilderImpl> key) {
            this.key = KeyCopier.copyFromBuilder(key);
        }

        public final Collection<String> getAttributesToGet() {
            return attributesToGet;
        }

        @Override
        public final Builder attributesToGet(Collection<String> attributesToGet) {
            this.attributesToGet = AttributeNameListCopier.copy(attributesToGet);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributesToGet(String... attributesToGet) {
            attributesToGet(Arrays.asList(attributesToGet));
            return this;
        }

        public final void setAttributesToGet(Collection<String> attributesToGet) {
            this.attributesToGet = AttributeNameListCopier.copy(attributesToGet);
        }

        public final Boolean getConsistentRead() {
            return consistentRead;
        }

        @Override
        public final Builder consistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
            return this;
        }

        public final void setConsistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
        }

        public final String getReturnConsumedCapacityAsString() {
            return returnConsumedCapacity;
        }

        @Override
        public final Builder returnConsumedCapacity(String returnConsumedCapacity) {
            this.returnConsumedCapacity = returnConsumedCapacity;
            return this;
        }

        @Override
        public final Builder returnConsumedCapacity(ReturnConsumedCapacity returnConsumedCapacity) {
            this.returnConsumedCapacity(returnConsumedCapacity == null ? null : returnConsumedCapacity.toString());
            return this;
        }

        public final void setReturnConsumedCapacity(String returnConsumedCapacity) {
            this.returnConsumedCapacity = returnConsumedCapacity;
        }

        public final String getProjectionExpression() {
            return projectionExpression;
        }

        @Override
        public final Builder projectionExpression(String projectionExpression) {
            this.projectionExpression = projectionExpression;
            return this;
        }

        public final void setProjectionExpression(String projectionExpression) {
            this.projectionExpression = projectionExpression;
        }

        public final Map<String, String> getExpressionAttributeNames() {
            return expressionAttributeNames;
        }

        @Override
        public final Builder expressionAttributeNames(Map<String, String> expressionAttributeNames) {
            this.expressionAttributeNames = ExpressionAttributeNameMapCopier.copy(expressionAttributeNames);
            return this;
        }

        public final void setExpressionAttributeNames(Map<String, String> expressionAttributeNames) {
            this.expressionAttributeNames = ExpressionAttributeNameMapCopier.copy(expressionAttributeNames);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetItemRequest build() {
            return new GetItemRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
