/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the version of the Server Message Block (SMB) protocol that DataSync uses to access an SMB file server.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SmbMountOptions implements SdkPojo, Serializable, ToCopyableBuilder<SmbMountOptions.Builder, SmbMountOptions> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(SmbMountOptions::versionAsString)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String version;

    private SmbMountOptions(BuilderImpl builder) {
        this.version = builder.version;
    }

    /**
     * <p>
     * By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB file
     * server. You also can configure DataSync to use a specific SMB version, but we recommend doing this only if
     * DataSync has trouble negotiating with the SMB file server automatically.
     * </p>
     * <p>
     * These are the following options for configuring the SMB version:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of SMB that they
     * mutually support between 2.1 and 3.1.1.
     * </p>
     * <p>
     * This is the recommended option. If you instead choose a specific version that your file server doesn't support,
     * you may get an <code>Operation Not Supported</code> error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
     * </p>
     * <note>
     * <p>
     * The <code>SMB1</code> option isn't available when <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating an Amazon
     * FSx for NetApp ONTAP location</a>.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #version} will
     * return {@link SmbVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #versionAsString}.
     * </p>
     * 
     * @return By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB
     *         file server. You also can configure DataSync to use a specific SMB version, but we recommend doing this
     *         only if DataSync has trouble negotiating with the SMB file server automatically.</p>
     *         <p>
     *         These are the following options for configuring the SMB version:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of SMB
     *         that they mutually support between 2.1 and 3.1.1.
     *         </p>
     *         <p>
     *         This is the recommended option. If you instead choose a specific version that your file server doesn't
     *         support, you may get an <code>Operation Not Supported</code> error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
     *         </p>
     *         <note>
     *         <p>
     *         The <code>SMB1</code> option isn't available when <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating an
     *         Amazon FSx for NetApp ONTAP location</a>.
     *         </p>
     *         </note></li>
     * @see SmbVersion
     */
    public final SmbVersion version() {
        return SmbVersion.fromValue(version);
    }

    /**
     * <p>
     * By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB file
     * server. You also can configure DataSync to use a specific SMB version, but we recommend doing this only if
     * DataSync has trouble negotiating with the SMB file server automatically.
     * </p>
     * <p>
     * These are the following options for configuring the SMB version:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of SMB that they
     * mutually support between 2.1 and 3.1.1.
     * </p>
     * <p>
     * This is the recommended option. If you instead choose a specific version that your file server doesn't support,
     * you may get an <code>Operation Not Supported</code> error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
     * </p>
     * <note>
     * <p>
     * The <code>SMB1</code> option isn't available when <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating an Amazon
     * FSx for NetApp ONTAP location</a>.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #version} will
     * return {@link SmbVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #versionAsString}.
     * </p>
     * 
     * @return By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB
     *         file server. You also can configure DataSync to use a specific SMB version, but we recommend doing this
     *         only if DataSync has trouble negotiating with the SMB file server automatically.</p>
     *         <p>
     *         These are the following options for configuring the SMB version:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of SMB
     *         that they mutually support between 2.1 and 3.1.1.
     *         </p>
     *         <p>
     *         This is the recommended option. If you instead choose a specific version that your file server doesn't
     *         support, you may get an <code>Operation Not Supported</code> error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
     *         </p>
     *         <note>
     *         <p>
     *         The <code>SMB1</code> option isn't available when <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating an
     *         Amazon FSx for NetApp ONTAP location</a>.
     *         </p>
     *         </note></li>
     * @see SmbVersion
     */
    public final String versionAsString() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(versionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SmbMountOptions)) {
            return false;
        }
        SmbMountOptions other = (SmbMountOptions) obj;
        return Objects.equals(versionAsString(), other.versionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SmbMountOptions").add("Version", versionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Version":
            return Optional.ofNullable(clazz.cast(versionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Version", VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SmbMountOptions, T> g) {
        return obj -> g.apply((SmbMountOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SmbMountOptions> {
        /**
         * <p>
         * By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB file
         * server. You also can configure DataSync to use a specific SMB version, but we recommend doing this only if
         * DataSync has trouble negotiating with the SMB file server automatically.
         * </p>
         * <p>
         * These are the following options for configuring the SMB version:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of SMB that
         * they mutually support between 2.1 and 3.1.1.
         * </p>
         * <p>
         * This is the recommended option. If you instead choose a specific version that your file server doesn't
         * support, you may get an <code>Operation Not Supported</code> error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
         * </p>
         * <note>
         * <p>
         * The <code>SMB1</code> option isn't available when <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating an
         * Amazon FSx for NetApp ONTAP location</a>.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param version
         *        By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB
         *        file server. You also can configure DataSync to use a specific SMB version, but we recommend doing
         *        this only if DataSync has trouble negotiating with the SMB file server automatically.</p>
         *        <p>
         *        These are the following options for configuring the SMB version:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of
         *        SMB that they mutually support between 2.1 and 3.1.1.
         *        </p>
         *        <p>
         *        This is the recommended option. If you instead choose a specific version that your file server doesn't
         *        support, you may get an <code>Operation Not Supported</code> error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
         *        </p>
         *        <note>
         *        <p>
         *        The <code>SMB1</code> option isn't available when <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating
         *        an Amazon FSx for NetApp ONTAP location</a>.
         *        </p>
         *        </note></li>
         * @see SmbVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SmbVersion
         */
        Builder version(String version);

        /**
         * <p>
         * By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB file
         * server. You also can configure DataSync to use a specific SMB version, but we recommend doing this only if
         * DataSync has trouble negotiating with the SMB file server automatically.
         * </p>
         * <p>
         * These are the following options for configuring the SMB version:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of SMB that
         * they mutually support between 2.1 and 3.1.1.
         * </p>
         * <p>
         * This is the recommended option. If you instead choose a specific version that your file server doesn't
         * support, you may get an <code>Operation Not Supported</code> error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
         * </p>
         * <note>
         * <p>
         * The <code>SMB1</code> option isn't available when <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating an
         * Amazon FSx for NetApp ONTAP location</a>.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param version
         *        By default, DataSync automatically chooses an SMB protocol version based on negotiation with your SMB
         *        file server. You also can configure DataSync to use a specific SMB version, but we recommend doing
         *        this only if DataSync has trouble negotiating with the SMB file server automatically.</p>
         *        <p>
         *        These are the following options for configuring the SMB version:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AUTOMATIC</code> (default): DataSync and the SMB file server negotiate the highest version of
         *        SMB that they mutually support between 2.1 and 3.1.1.
         *        </p>
         *        <p>
         *        This is the recommended option. If you instead choose a specific version that your file server doesn't
         *        support, you may get an <code>Operation Not Supported</code> error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB3</code>: Restricts the protocol negotiation to only SMB version 3.0.2.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB2</code>: Restricts the protocol negotiation to only SMB version 2.1.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB2_0</code>: Restricts the protocol negotiation to only SMB version 2.0.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMB1</code>: Restricts the protocol negotiation to only SMB version 1.0.
         *        </p>
         *        <note>
         *        <p>
         *        The <code>SMB1</code> option isn't available when <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html">creating
         *        an Amazon FSx for NetApp ONTAP location</a>.
         *        </p>
         *        </note></li>
         * @see SmbVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SmbVersion
         */
        Builder version(SmbVersion version);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private BuilderImpl() {
        }

        private BuilderImpl(SmbMountOptions model) {
            version(model.version);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        @Override
        public final Builder version(SmbVersion version) {
            this.version(version == null ? null : version.toString());
            return this;
        }

        @Override
        public SmbMountOptions build() {
            return new SmbMountOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
